
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>

#include "../utilities/openglheader.h"

#include "app3.h"
#include "../utilities/utilities.h"
#include "../utilities/GPUsparsemat.h"
#include "../utilities/meshes.h"
#include "palm.h"
#include "trans.h"
#include "lights.h"
#include "app3proc.h"
#include "app3struct.h"

static AppData appdata;

void InitPalmMeshes ( AppData *ad )
{
  static const GLfloat edges_colour[3] = {0.0,0.5,0.7};
  static const GLfloat facets_colour[3] = {0.91,0.65,0.5};
  KLMesh *palm;
  int    i;

  palm = &ad->palm;
  if ( (palm->mesh[0] = EnterPalmToGPU ()) ) {
    for ( i = 1; i <= NPALMMESHES; i++ ) {
      if ( !(palm->mesh[i] = malloc ( sizeof(GPUmesh) )) )
        ExitOnError ( "InitPalmMeshes 0" );
      memset ( palm->mesh[i], 0, sizeof(GPUmesh) );
      if ( !GPUmeshRefinement ( MESHDEG, palm->mesh[i-1], palm->mesh[i] ) )
        ExitOnError ( "InitPalmMeshes 1" );
      printf ( "%d: nv = %d, nhe = %d, nfac = %d\n",
               i, palm->mesh[i]->nv, palm->mesh[i]->nhe, palm->mesh[i]->nfac );
    }
    memcpy ( palm->ecolour, edges_colour, 3*sizeof(GLfloat) );
    memcpy ( palm->fcolour, facets_colour, 3*sizeof(GLfloat) );
  }
  else
    ExitOnError ( "InitPalmMeshes" );
} /*InitPalmMeshes*/

static void _ResizeMyWorld ( AppData *ad, int width, int height )
{
  float lr;

  glViewport ( 0, 0, ad->camera.win_width = width,
               ad->camera.win_height = height );  
  lr = 0.5533*(float)width/(float)height;
  M4x4Frustumf ( ad->trans.pm, NULL, ad->camera.left = -lr, ad->camera.right = lr,
                 ad->camera.bottom = -0.5533, ad->camera.top = 0.5533,
                 ad->camera.near = 5.0, ad->camera.far = 15.0 );
  ad->camera.rl = 2.0*lr;  ad->camera.tb = 2.0*0.5533;
  LoadVPMatrix ( &ad->trans );
} /*_ResizeMyWorld*/

void ResizeMyWorld ( int width, int height )
{
  _ResizeMyWorld ( &appdata, width, height );
} /*ResizeMyWorld*/

void InitCamera ( AppData *ad, int width, int height )
{
  static const float viewer_rvec[3] = {1.0,0.0,0.0}; 
  static const float viewer_pos0[4] = {0.0,0.0,10.0,1.0};   

  memcpy ( ad->camera.viewer_rvec, viewer_rvec, 3*sizeof(float) );
  memcpy ( ad->camera.viewer_pos0, viewer_pos0, 4*sizeof(float) );
  memcpy ( ad->trans.eyepos, viewer_pos0, 4*sizeof(GLfloat) );   
  ad->camera.viewer_rangle = 0.0;
  M4x4InvTranslatefv ( ad->trans.vm, viewer_pos0 );
  _ResizeMyWorld ( ad, width, height );
} /*InitCamera*/

void InitLights ( AppData *ad )
{
  GLfloat amb0[3] = { 0.2, 0.2, 0.3 };
  GLfloat dir0[3] = { 0.8, 0.8, 0.8 };
  GLfloat pos0[4] = { 0.0, 1.0, 1.0, 0.0 };
  GLfloat atn0[3] = { 1.0, 0.0, 0.0 };

  SetLightAmbient ( &ad->light, 0, amb0 );
  SetLightDirect ( &ad->light, 0, dir0 );
  SetLightPosition ( &ad->light, 0, pos0 );
  SetLightAttenuation ( &ad->light, 0, atn0 );
  SetLightOnOff ( &ad->light, 0, 1 );
} /*InitLights*/

void SetupModelMatrix ( AppData *ad )
{
  GLfloat mm[16];

  M4x4RotateVfv ( mm, ad->model_rot_axis, ad->model_rot_angle );
  LoadMMatrix ( &ad->trans, mm );
} /*SetupModelMatrix*/

void LoadMyShaders ( AppData *ad )
{
  LoadMeshRefinementProgram ();
  LoadMeshRenderingPrograms ( &ad->mrprog );
} /*LoadMyShaders*/

AppWidgets *InitMyWorld ( int argc, char *argv[], int width, int height )
{
  static const float model_rot_axis[3] = {0.0,1.0,0.0};

  memset ( &appdata, 0, sizeof(AppData) );
  LoadMyShaders ( &appdata );
  ConstructEmptyVAO ();
  appdata.trans.trbuf = NewUniformTransBlock ();
  appdata.light.lsbuf = NewUniformLightBlock ();
  TimerInit ();
  memcpy ( appdata.model_rot_axis, model_rot_axis, 3*sizeof(float) );
  appdata.speed = 0.5*3.1415926;
  SetupModelMatrix ( &appdata );
  InitCamera ( &appdata, width, height );
  InitLights ( &appdata );
  appdata.wdg.sw[0] = appdata.wdg.sw[2] = true;
  appdata.lod = 2;
  appdata.edges = appdata.wdg.animation = false;
  InitPalmMeshes ( &appdata );
  ExitIfGLError ( "InitMyObject" );
  return &appdata.wdg;
} /*InitMyWorld*/

void _RotateViewer ( AppData *ad, double delta_xi, double delta_eta )
{
  float   vi[3], lgt, vk[3];
  double  angi, angk;

  vi[0] = (float)delta_eta*ad->camera.tb/(float)ad->camera.win_height;
  vi[1] = (float)delta_xi*ad->camera.rl/(float)ad->camera.win_width;
  vi[2] = 0.0;
  lgt = sqrt ( V3DotProductf ( vi, vi ) );
  vi[0] /= lgt;  vi[1] /= lgt;
  angi = -0.052359878;  /* -3 stopnie */
  V3CompRotationsf ( vk, &angk, ad->camera.viewer_rvec,
                     ad->camera.viewer_rangle, vi, angi );
  memcpy ( ad->camera.viewer_rvec, vk, 3*sizeof(float) );
  ad->camera.viewer_rangle = angk;
  M4x4RotateVfv ( ad->trans.vm, ad->camera.viewer_rvec,
                  -ad->camera.viewer_rangle );
  M4x4MultMTVf ( ad->trans.eyepos, ad->trans.vm, ad->camera.viewer_pos0 );
  M4x4InvTranslateMfv ( ad->trans.vm, ad->camera.viewer_pos0 );
  LoadVPMatrix ( &ad->trans );
} /*_RotateViewer*/

void RotateViewer ( double delta_xi, double delta_eta )
{
  if ( delta_xi == 0.0 && delta_eta == 0.0 )
    return;  /* natychmiast uciekamy - nie chcemy dzielic przez 0 */
  _RotateViewer ( &appdata, delta_xi, delta_eta );
} /*RotateViewer*/

void DrawMyScene ( AppData *ad, AppWidgets *wdg )
{
  int i;

  glClearColor ( 1.0, 1.0, 1.0, 1.0 );
  glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
  glEnable ( GL_DEPTH_TEST );
  if ( wdg->sw[0] )
    DrawMeshEdges ( &ad->mrprog, ad->palm.mesh[0], ad->palm.ecolour );
  for ( i = 1; i <= NPALMMESHES; i++ )
    if ( wdg->sw[i] ) {
      if ( ad->edges )
        DrawMeshEdges ( &ad->mrprog, ad->palm.mesh[i], ad->palm.fcolour );
      else
        DrawMeshFacets ( &ad->mrprog, ad->palm.mesh[i], ad->palm.fcolour );
    }
} /*DrawMyScene*/

void RedrawMyWorld ( void )
{
  DrawMyScene ( &appdata, &appdata.wdg );
  glFlush ();
} /*RedrawMyWorld*/

void DeleteMyShaders ( AppData *ad )
{
  DeleteMeshRefinementProgram ();
  DeleteMeshRenderingPrograms ( &ad->mrprog );
} /*DeleteMyShaders*/

void DeleteMyWorld ( void )
{
  int i;

  DeleteMyShaders ( &appdata );
  for ( i = 0; i <= NPALMMESHES; i++ )
    DeleteGPUmesh ( appdata.palm.mesh[i] );
  glDeleteBuffers ( 1, &appdata.trans.trbuf );
  glDeleteBuffers ( 1, &appdata.light.lsbuf );
  DeleteEmptyVAO ();
#ifdef DEBUG_BUFFERS_ALLOCATION
  DumpBufferIdentifiers ();
#endif
  ExitIfGLError ( "DeleteMyWorld" );
} /*DeleteMyWorld*/

char ProcessSwitchCommand ( int wdg_id )
{
  switch ( wdg_id ) {
case SW_ID_MESH1:  case SW_ID_MESH2:  case SW_ID_MESH3:  case SW_ID_MESH4:
    if ( appdata.wdg.sw[wdg_id-SW_ID_MESH0] ) {
      memset ( &appdata.wdg.sw[1], false, NPALMMESHES );
      appdata.wdg.sw[(int)(appdata.lod = wdg_id-SW_ID_MESH0)] = true;
    }
    return true;
case SW_ID_MESH0:
    return true;
default:  
    return false;
  }
} /*ProcessSwitchCommand*/

void ToggleAnimation ( AppData *ad )
{
  if ( (ad->wdg.animation = !ad->wdg.animation) ) {
    ProcessWorldRequest ( WMSG_ANIMATION_ON, NULL, NULL );
    TimerTic ();
  }
  else
    ProcessWorldRequest ( WMSG_ANIMATION_OFF, NULL, NULL );
} /*ToggleAnimation */

char ProcessCharCommand ( char charcode )
{
  switch ( toupper ( charcode ) ) {
case ' ':
    ToggleAnimation ( &appdata );
    return true;
case 'K':
    appdata.edges = !appdata.edges;
    return true;
default:
    return false;
  } 
} /*ProcessCharCommand*/

char MoveOn ( void )
{
  if ( appdata.wdg.animation ) {
    if ( (appdata.model_rot_angle += appdata.speed * TimerTocTic ()) >= PI )
      appdata.model_rot_angle -= 2.0*PI;
    SetupModelMatrix ( &appdata );
  }
  return appdata.wdg.animation;
} /*MoveOn*/

