#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include "../utilities/openglheader.h"

#include "app2k.h"
#include "../utilities/utilities.h"
#include "../utilities/linkage.h"
#include "../utilities/bezpatches.h"
#include "../utilities/teapot.h"
#include "../utilities/sproduct.h"
#include "accumbuf.h"
#include "texture.h"
#include "trans.h"
#include "lights.h"
#include "app2kstruct.h"

void LoadLinkageArticulationProgram ( KLArticulationProgram *prog )
{
  static const char *filename[] = { "app2h.comp.glsl" };
  static const GLchar *CPINames[] = { "CPoints" };
  static const GLchar *CPONames[] = { "CPointsOut" };
  static const GLchar *CompTrNames[] =  { "Tr" };
  static const GLchar *CompTrIndNames[] = { "TrInd" };
  static const GLchar DimName[] = "dim";
  static const GLchar TrNumName[] = "trnum";
  GLuint shader_id;
  GLint  i;

  shader_id = CompileShaderFiles ( GL_COMPUTE_SHADER, 1, &filename[0] );
  prog->program_id = LinkShaderProgram ( 1, &shader_id, "articulate" );
  prog->cpibp = prog->cpobp = prog->ctrbp = prog->ctribp = GL_INVALID_INDEX;
  GetAccessToStorageBlock ( prog->program_id, 0, &CPINames[0], &i, &i, &prog->cpibp );
  GetAccessToStorageBlock ( prog->program_id, 0, &CPONames[0], &i, &i, &prog->cpobp );
  GetAccessToStorageBlock ( prog->program_id, 0, &CompTrNames[0],
                            &i, &i, &prog->ctrbp );
  GetAccessToStorageBlock ( prog->program_id, 0, &CompTrIndNames[0],
                            &i, &i, &prog->ctribp );
  prog->dim_loc = glGetUniformLocation ( prog->program_id, DimName );
  prog->trnum_loc = glGetUniformLocation ( prog->program_id, TrNumName );
  glDeleteShader ( shader_id );
  ExitIfGLError ( "LoadLinkageArticulationProgram" );
} /*LoadLinkageArticulationProgram*/

void DeleteLinkageArticulationProgram ( KLArticulationProgram *prog )
{
  glUseProgram ( 0 );
  glDeleteProgram ( prog->program_id );
} /*DeleteLinkageArticulationProgram*/

GLuint LoadMyTextures ( void )
{
  GLuint tex;
  const char *fn[4] = { "../tiff/jaszczur.tif", "../tiff/salamandra.tif",
                  "../tiff/lis.tif", "../tiff/kwiatki.tif" };

  if ( (tex = CreateMyTexture ( 1024 )) ) {
    LoadMyTextureImage ( tex, 1024, 1024, 0, 0, fn[0] );
    LoadMyTextureImage ( tex, 1024, 1024, 512, 0, fn[1] );
    LoadMyTextureImage ( tex, 1024, 1024, 0, 512, fn[2] );
    LoadMyTextureImage ( tex, 1024, 1024, 512, 512, fn[3] );
    SetupMyTextureMipmaps ( tex );
  }
  return tex;
} /*LoadMyTextures*/

static char ConstructMyTeapot ( kl_linkage *lkg, kl_object *obj )
{
  const int r0[198] =
    {  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
      16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
      32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
      48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
      64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
      80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95,
      96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,
     112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,
     128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
     144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
     160,161,162,163,164,165,166,167,168,177,178,179,180,269,274,275,
     276,277,278,279,280,281,285,286,287,288,289,290,294,295,296,297,
     298,299,302,303,304,305};
  const int r1[30] =
    {169,170,171,172,173,174,175,176,181,182,183,184,185,186,187,
     188,189,190,191,192,193,194,195,196,197,198,199,200,201,202};
  const int r2[62] =
    {203,206,207,208,209,210,211,212,213,214,216,217,218,219,220,221,
     223,224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,
     239,240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,
     255,256,257,258,259,260,261,262,263,264,265,266,267,268};
  const GLfloat MyColour[4] = { 1.0, 1.0, 1.0, 1.0 };
  const GLfloat diffr[4] = { 0.75, 0.65, 0.3, 1.0 };
  const GLfloat specr[4] = { 0.7, 0.7, 0.6, 1.0 };
  const GLfloat shn = 60.0, wa = 5.0, we = 5.0;
  const GLfloat txc[32][4] =
    {{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.-1,0.0,0.0},
     {0.5,0.5,0.0,0.0},{0.5,1.0,0.0,0.5},{1.0,0.5,0.5,0.0},{1.0,1.0,0.5,0.5},
     {-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},
     {-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},
     {-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},
     {-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},
     {-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},
     {-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0}};
  AppData      *ad;
  KLBezPatches *bezp;
  GLuint       *cpi;
  int          on, rn, i;

  ad = (AppData*)lkg->usrdata;
  bezp = (KLBezPatches*)obj->usrdata;
  bezp->bpatches[0] = ConstructTheTeapot ( MyColour );
  bezp->bpatches[1] = malloc ( sizeof(BezierPatchObjf) );
  cpi = malloc ( obj->nvert*sizeof(GLuint) );
  if ( bezp->bpatches[0] && bezp->bpatches[1] && cpi ) {
    on = obj - lkg->obj;
    bezp->ntr = -1;
    memset ( cpi, 0, obj->nvert*sizeof(GLuint) );
    rn = kl_NewObjRef ( lkg, 2, on, 198, NULL );
    for ( i = 0; i < 198; i++ ) cpi[r0[i]] = rn;
    rn = kl_NewObjRef ( lkg, 4, on,  30, NULL );
    for ( i = 0; i <  30; i++ ) cpi[r1[i]] = rn;
    rn = kl_NewObjRef ( lkg, 6, on,  62, NULL );
    for ( i = 0; i <  62; i++ ) cpi[r2[i]] = rn;
    bezp->tribuf = NewStorageBuffer ( obj->nvert*sizeof(GLuint),
                                      ad->artprog.ctribp );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, 0, obj->nvert*sizeof(GLuint), cpi );
    free ( cpi );
    bezp->mtn = SetupMaterial ( &ad->mat, -1, diffr, specr, shn, wa, we );
    if ( !GenBezierPatchTextureBlock ( bezp->bpatches[0], &txc[0][0] ) )
      ExitOnError ( "ConstructMyTeapot 0" );
    SetBezierPatchTessLevel ( bezp->bpatches[0], ad->TessLevel );
    SetBezierPatchNVS ( bezp->bpatches[0], ad->BezNormals );
    memcpy ( bezp->bpatches[1], bezp->bpatches[0], sizeof(BezierPatchObjf) );
    glGenBuffers ( 1, &bezp->bpatches[1]->buf[1] );
    glBindBuffer ( GL_SHADER_STORAGE_BUFFER, bezp->bpatches[1]->buf[1] );
    glBufferData ( GL_SHADER_STORAGE_BUFFER, TEAPOT_NPOINTS*3*sizeof(GLfloat),
                   NULL, GL_DYNAMIC_DRAW );
    bezp->texture = LoadMyTextures ();
    ExitIfGLError ( "ConstructMyTeapot" );
  }
  else
    ExitOnError ( "ConstructMyTeapot" );
  return true;
} /*ConstructMyTeapot*/

static char ConstructMyTorus ( kl_linkage *lkg, kl_object *obj )
{
  GLfloat MyColour[4] = { 0.2, 0.3, 1.0, 1.0 };
  const GLfloat diffr[4] = { 0.0, 0.4, 1.0, 1.0 };
  const GLfloat specr[4] = { 0.7, 0.7, 0.7, 1.0 };
  const GLfloat shn = 20.0, wa = 2.0, we = 5.0;
  AppData      *ad;
  KLBezPatches *bezp;

  ad = (AppData*)lkg->usrdata;
  bezp = (KLBezPatches*)obj->usrdata;
  bezp->bpatches[0] = EnterTorus ( 1.0, 0.5, MyColour );
  bezp->bpatches[1] = malloc ( sizeof(BezierPatchObjf) );
  if ( bezp->bpatches[0] && bezp->bpatches[1] ) {
    bezp->tribuf = 0;
    bezp->ntr = kl_NewObjRef ( lkg, 9, obj - lkg->obj, 49, NULL );
    bezp->mtn = SetupMaterial ( &ad->mat, -1, diffr, specr, shn, wa, we );
    SetBezierPatchTessLevel ( bezp->bpatches[0], ad->TessLevel );
    SetBezierPatchNVS ( bezp->bpatches[0], ad->BezNormals );
    memcpy ( bezp->bpatches[1], bezp->bpatches[0], sizeof(BezierPatchObjf) );
    glGenBuffers ( 1, &bezp->bpatches[1]->buf[1] );
    glBindBuffer ( GL_SHADER_STORAGE_BUFFER, bezp->bpatches[1]->buf[1] );
    glBufferData ( GL_SHADER_STORAGE_BUFFER, 49*4*sizeof(GLfloat),
                   NULL, GL_DYNAMIC_DRAW );
    bezp->texture = 0;
    ExitIfGLError ( "ConstructMyTorus" );
  }
  else
    ExitOnError ( "ConstructMyTorus" );
  return true;
} /*ConstructMyTorus*/

static void KLTransformBP ( kl_linkage *lkg, kl_object *obj,
                            int refn, GLfloat *tr, int nv, int *vn )
{
  AppData *ad;

  ad = (AppData*)lkg->usrdata;
  glBindBuffer ( GL_SHADER_STORAGE_BUFFER, ad->lktrbuf );
  glBufferSubData ( GL_SHADER_STORAGE_BUFFER, refn*16*sizeof(GLfloat),
                    16*sizeof(GLfloat), tr );
  ExitIfGLError ( "KLTransformBP" );
} /*KLTransformBP*/

static void KLPostprocessBP ( kl_linkage *lkg, kl_object *obj )
{
  AppData               *ad;
  KLArticulationProgram *prog;
  KLBezPatches          *bpobj;
  BezierPatchObjf       **bp;

  ad = (AppData*)lkg->usrdata;
  prog = &ad->artprog;
  bpobj = (KLBezPatches*)obj->usrdata;
  glUseProgram ( prog->program_id );
  bp = (BezierPatchObjf**)bpobj->bpatches;
  glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, prog->ctrbp, ad->lktrbuf );
  glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, prog->cpibp, bp[0]->buf[1] );
  glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, prog->cpobp, bp[1]->buf[1] );
  if ( bpobj->ntr == -1 )
    glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, prog->ctribp, bpobj->tribuf );
  glUniform1i ( prog->trnum_loc, bpobj->ntr );
  glUniform1i ( prog->dim_loc, bp[0]->dim );
  glDispatchCompute ( obj->nvert, 1, 1 );
  glMemoryBarrier ( GL_SHADER_STORAGE_BARRIER_BIT );
  ExitIfGLError ( "KLPostprocessBP" );
} /*KLPostprocessBP*/

static void KLRedrawBezPatches ( kl_linkage *lkg, kl_object *obj )
{
  AppData      *ad;
  KLBezPatches *bezp;
  GLint        cs;

  ad = (AppData*)lkg->usrdata;
  bezp = (KLBezPatches*)obj->usrdata;
  if ( ad->skeleton )
    glPolygonMode ( GL_FRONT_AND_BACK, GL_LINE );
  else
    glPolygonMode ( GL_FRONT_AND_BACK, GL_FILL );
  if ( ad->final ) {
    glUseProgram ( ad->brprog.program_id[1] );
    if ( (cs = ad->brprog.colour_source) == 2 ) {
      if ( bezp->bpatches[0]->buf[3] ) {
        BindBezPatchTextureBuffer ( bezp->bpatches[0] );
        glActiveTexture ( GL_TEXTURE0 );
        glBindTexture ( GL_TEXTURE_2D, bezp->texture );
      }
      else
        cs = 1;
    }
    glUniform1i ( ad->brprog.ColourSourceLoc, cs );
    glUniform1i ( ad->brprog.NormalSourceLoc, ad->brprog.normal_source );
    glUniform1i ( ad->brprog.ModifyDepthLoc, ad->brprog.modify_depth );
    ChooseMaterial ( &ad->mat, bezp->mtn );
  }
  else
    glUseProgram ( ad->brprog.program_id[0] );
  DrawBezierPatches ( bezp->bpatches[1] );
  if ( ad->cnet ) {
    glUseProgram ( ad->brprog.program_id[2] );
    DrawBezierNets ( bezp->bpatches[1] );
  }
} /*KLRedrawBezPatches*/

void KLDeleteBezPatches ( kl_linkage *lkg, kl_object *obj )
{
  KLBezPatches *bezp;

  bezp = (KLBezPatches*)obj->usrdata;
  DeleteBezierPatches ( bezp->bpatches[0] );
  glDeleteBuffers ( 1, &bezp->bpatches[1]->buf[1] );
  if ( bezp->tribuf ) glDeleteBuffers ( 1, &bezp->tribuf );
  if ( bezp->texture ) glDeleteTextures ( 1, &bezp->texture );
  free ( bezp->bpatches[1] );
} /*KLDeleteBezPatches*/

static char ConstructParticleGun ( kl_linkage *lkg, kl_object *obj )
{
  AppData    *ad;
  PartSystem *ps;

  ad = (AppData*)lkg->usrdata;
  ps = (PartSystem*)obj->usrdata;
  InitParticleSystem ( &ad->psprog, ps );
  kl_NewObjRef ( lkg, 4, obj - lkg->obj, 0, NULL );
  return true;
} /*ConstructParticleGun*/

static void KLTransformParticles ( kl_linkage *lkg, kl_object *obj,
                                   int refn, GLfloat *tr, int nv, int *vn )
{
  const GLfloat p0[3] = {3.083333,0.0,2.4875},
                v0[3] = {0.8,0.0,0.6};
  PartSystem *ps;

  ps = (PartSystem*)obj->usrdata;
  memcpy ( ps->pos0, ps->pos1, 3*sizeof(GLfloat) );
  M4x4MultMP3f ( ps->pos1, tr, p0 );
  memcpy ( ps->vel0, ps->vel1, 3*sizeof(GLfloat) );
  M4x4MultMV3f ( ps->vel1, tr, v0 );
  V3Normalisef ( ps->vel1 );
} /*KLTransformParticles*/

static void KLPostprocessParticles ( kl_linkage *lkg, kl_object *obj )
{
  AppData *ad;

  ad = (AppData*)lkg->usrdata;
  if ( ad->particles )
    MoveParticles ( &ad->psprog, &ad->ps, app_time-ad->hold_time );
} /*KLPostprocessParticles*/

static void KLRedrawParticles ( kl_linkage *lkg, kl_object *obj )
{
  AppData *ad;

  ad = (AppData*)lkg->usrdata;
  if ( ad->particles ) {
    if ( !ad->final )
      glColorMask ( GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE );
    DrawParticles ( &ad->psprog, &ad->ps, &ad->camera, ad->final );
  }
} /*KLRedrawParticles*/

static void KLDeleteParticles ( kl_linkage *lkg, kl_object *obj )
{
  AppData *ad;

  ad = (AppData*)lkg->usrdata;
  DeleteParticleSystem ( &ad->ps );
} /*KLDeleteParticles*/

kl_linkage *ConstructMyLinkage ( AppData *ad )
{
#define SCF (1.0/3.0)
  kl_linkage *lkg;
  int        l[10], j[9];
  int        i;
  GLfloat    tra[16], trb[16];

  if ( (lkg = kl_NewLinkage ( 3, 10, 5, 9, 9, (void*)ad )) ) {
    ad->linkage = lkg;
    for ( i = 0; i < 10; i++ )
      l[i] = kl_NewLink ( lkg );
    M4x4Scalef ( tra, SCF, SCF, SCF*4.0/3.0 );
    kl_NewObject ( lkg, 0, 3, TEAPOT_NPOINTS, tra, (void*)&ad->bezp[0],
                   ConstructMyTeapot, KLTransformBP, KLPostprocessBP,
                   KLRedrawBezPatches, KLDeleteBezPatches );
    M4x4RotateXf ( trb, 0.5*PI );
    M4x4MScalef ( trb, 0.1, 0.1, 0.1 );
    kl_NewObject ( lkg, 0, 4, 49, trb, (void*)&ad->bezp[1],
                   ConstructMyTorus, KLTransformBP, KLPostprocessBP,
                   KLRedrawBezPatches, KLDeleteBezPatches );
    kl_NewObject ( lkg, 0, 3, 0, tra, (void*)&ad->ps, ConstructParticleGun,
                   KLTransformParticles, KLPostprocessParticles,
                   KLRedrawParticles, KLDeleteParticles );
    j[0] = kl_NewJoint ( lkg, l[0], l[1], KL_ART_ROT_Z, 0 );
    j[1] = kl_NewJoint ( lkg, l[1], l[2], KL_ART_ROT_Y, 1 );
    j[2] = kl_NewJoint ( lkg, l[2], l[3], KL_ART_ROT_Y, 2 );
    j[3] = kl_NewJoint ( lkg, l[3], l[4], KL_ART_ROT_Y, 3 );
    j[4] = kl_NewJoint ( lkg, l[2], l[5], KL_ART_ROT_Y, 4 );
    j[5] = kl_NewJoint ( lkg, l[5], l[6], KL_ART_ROT_Y, 5 );
    j[6] = kl_NewJoint ( lkg, l[2], l[7], KL_ART_ROT_Y, 6 );
    j[7] = kl_NewJoint ( lkg, l[7], l[8], KL_ART_ROT_Y, 7 );
    j[8] = kl_NewJoint ( lkg, l[8], l[9], KL_ART_ROT_Z, 8 );
    M4x4Translatef ( lkg->current_root_tr, 0.0, 0.0, -0.6 );
    M4x4Translatef ( tra, -0.43, 0.0, 0.92 );
    kl_SetJointFtr ( lkg, j[1], tra, true );
    M4x4Translatef ( tra, 0.78, 0.0, 0.59 );
    kl_SetJointFtr ( lkg, j[2], tra, true );
    M4x4Translatef ( tra, 0.78, 0.0, 0.91 );
    kl_SetJointFtr ( lkg, j[3], tra, true );
    M4x4Translatef ( tra, 0.6, 0.0, 1.0 );
    kl_SetJointFtr ( lkg, j[4], tra, true );
    M4x4Translatef ( tra, -0.6, 0.0, 1.0 );
    kl_SetJointFtr ( lkg, j[5], tra, true );
    M4x4Translatef ( tra, 0.52, 0.0, 0.97 );
    kl_SetJointFtr ( lkg, j[6], tra, false );
    M4x4Translatef ( tra, 0.52, 0.0, 0.0 );
    kl_SetJointFtr ( lkg, j[7], tra, false );
    glGenBuffers ( 1, &ad->lktrbuf );
    glBindBuffer ( GL_SHADER_STORAGE_BUFFER, ad->lktrbuf );
    glBufferData ( GL_SHADER_STORAGE_BUFFER, lkg->norefs*16*sizeof(GLfloat),
                   NULL, GL_DYNAMIC_DRAW );
  }
  else
    ExitOnError ( "ConstructMyLinkage" );
  return lkg;
#undef SCF
} /*ConstructMyLinkage*/

double TeapotRotAngle2 ( double time )
{
  double s;

  s = sin ( 2.0*PI*time/8.0 );
  return s > 0.5 ? 3.0*(s-0.5) : 0.0;
} /*TeapotRotAngle2*/

double SpoutAngle ( double time )
{
  double s;

  s = sin ( 2.0*PI*time/8.0 );
  return s > 0.25 ? 2.0*(s-0.25) : 0.0;
} /*SpoutAngle*/

double LidAngle1 ( double time )
{
  double s;

  s = sin ( 2.0*PI*time/8.0 );
  if ( s <= 0.8 ) return 0.0;
  s = 6.0*(s-0.8);
  return s < 0.4 ? s : 0.4;
} /*LidAngle1*/

double LidAngle2 ( double time )
{
  double s;

  s = sin ( 2.0*PI*time/8.0 );
  return s < 0.0 ? s : 0.0;
} /*LidAngle2*/

double TorusRotAngle1 ( double time, char particles )
{
  double s;

  if ( particles )
    return 4.6*0.4;
  else {
    s = sin ( 2.0*PI*time/8.0 );
    return s < -0.2 ? 4.6*(0.2-s) : 4.6*0.4;
  }
} /*TorusRotAngle1*/

void ArticulateMyLinkage ( kl_linkage *lk )
{
  AppData *ad;
  double  at, dt, par[9];

  ad = (AppData*)lk->usrdata;
  dt = TimerTocTic ();
  at = app_time - ad->hold_time;
  if ( ad->animate ) {
    if ( (ad->teapot_rot_angle += ANGULAR_VELOCITY1 * dt) >= PI )
      ad->teapot_rot_angle -= 2.0*PI;
  }
  par[0] = ad->teapot_rot_angle;
  par[1] = TeapotRotAngle2 ( at );
  par[2] = SpoutAngle ( at );
  par[3] = -0.5*par[2];
  par[4] = LidAngle1 ( at );
  par[5] = LidAngle2 ( at );
  par[7] = -(par[6] = TorusRotAngle1 ( at, ad->particles ));
  par[8] = -2.0*PI*at;
  kl_SetArtParam ( lk, 0, 9, par );
  kl_Articulate ( lk );
} /*ArticulateMyLinkage*/

void DrawMyLinkage ( AppData *ad, char final )
{
  ad->final = final;
  kl_Redraw ( ad->linkage );
} /*DrawMyLinkage*/

