#define USE_HIFRCLOCK
/* If the macrodefinition above is removed or commented out, */
/* measured time accuracy is only up to 0.01s.               */

#ifdef __linux__  /* Linux */
#ifdef USE_HIFRCLOCK
#include <sys/time.h>
#ifndef __USE_POSIX199309
#define __USE_POSIX199309 1
#endif
#include <time.h>
#else
#include <unistd.h>
#include <sys/times.h>
#endif
#endif
#ifdef _WIN32
#define WIN32_LEAN_AND_MEAN
#include <Windows.h>
#ifdef USE_HIFRCLOCK
#include <profileapi.h>
#include <timeapi.h>
#else
#include <time.h>
#endif
#endif

#include "openglheader.h"
#include "utilities.h"

double app_time, tic_time, toc_time;

/* ////////////////////////////////////////////////////////////////////////// */
/* Linux implementation */
#ifdef __linux__
#ifdef USE_HIFRCLOCK
static clockid_t       clockid;
static struct timespec tick0, tick1, tick2;
#else
static struct tms      clk;
static clock_t         tick0, tick1, tick2;
static double          ticks_per_sec = 100.0;
#endif

#ifdef USE_HIFRCLOCK
#define TDIFF(t1,t0) \
  ((double)(t1.tv_sec-t0.tv_sec) + 1.0e-9*(double)(t1.tv_nsec-t0.tv_nsec))
#else
#define TDIFF(t1,t0) ((double)(t1-t0)/ticks_per_sec)
#endif

void TimerInit ( void )
{
#ifdef USE_HIFRCLOCK
  struct timespec res;

  if ( clock_getres ( (clockid = CLOCK_MONOTONIC), &res ) )
    if ( clock_getres ( (clockid = CLOCK_REALTIME), &res ) )
      ExitOnError ( "TimerInit" );
  clock_gettime ( clockid, &tick0 );
  tick1 = tick2 = tick0;
#else
  ticks_per_sec = (double)sysconf ( _SC_CLK_TCK );
  tick0 = tick1 = tick2 = times ( &clk );
#endif
  app_time = tic_time = toc_time = 0.0;
} /*TimerInit*/

double TimerTic ( void )
{
#ifdef USE_HIFRCLOCK
  clock_gettime ( clockid, &tick1 );
  tick2 = tick1;
#else
  tick1 = tick2 = times ( &clk );
#endif
  toc_time = 0.0;
  return app_time = tic_time = TDIFF ( tick2, tick0 );
} /*TimerTic*/

double TimerToc ( void )
{
#ifdef USE_HIFRCLOCK
  clock_gettime ( clockid, &tick2 );
#else
  tick2 = times ( &clk );
#endif
  app_time = TDIFF ( tick2, tick0 );
  return toc_time = TDIFF ( tick2, tick1 );
} /*TimerToc*/

double TimerTocTic ( void )
{
#ifdef USE_HIFRCLOCK
  struct timespec tick;

  tick = tick1;
  clock_gettime ( clockid, &tick1 );
  tick2 = tick1;
#else
  clock_t tick;

  tick = tick1;  tick1 = tick2 = times ( &clk );
#endif
  app_time = TDIFF ( tick2, tick0 );
  toc_time = 0.0;
  return toc_time = TDIFF ( tick2, tick );
} /*TimerTocTic*/
#endif

/* ////////////////////////////////////////////////////////////////////////// */
/* Windows implementation */
#ifdef _WIN32
#ifdef USE_HIFRCLOCK
static char          performance;
static double        frequency;
static LARGE_INTEGER tick0, tick1, tick2;
#else
static clock_t       tick0, tick1, tick2;
#endif

#ifdef USE_HIFRCLOCK
#define TDIFF(t1,t0) ((double)(t1.QuadPart-t0.QuadPart)/frequency)
#else
#define TDIFF(t1,t0) ((double)(t1-t0)/(double)CLOCKS_PER_SEC)
#endif

void TimerInit ( void )
{
#ifdef USE_HIFRCLOCK
  LARGE_INTEGER fv;

  if ( (performance = (QueryPerformanceFrequency ( &fv ) != 0)) ) {
    frequency = (double)fv.QuadPart;
    QueryPerformanceCounter ( &tick2 );
  }
  else {
    frequency = 1000;
    tick2.QuadPart = timeGetTime ();
  }
  tick0 = tick1 = tick2;
#else
  tick0 = tick1 = tick2 = clock ();
#endif
  app_time = toc_time = 0.0;
} /*TimerInit*/

double TimerTic ( void )
{
#ifdef USE_HIFRCLOCK
  if ( performance )
    QueryPerformanceCounter ( &tick2 );
  else
    tick2.QuadPart = timeGetTime ();
  tick1 = tick2;
#else
  tick1 = tick2 = clock ();
#endif
  toc_time = 0.0;
  return app_time = tic_time = TDIFF ( tick2, tick0 );
} /*TimerTic*/

double TimerToc ( void )
{
#ifdef USE_HIFRCLOCK
  if ( performance )
    QueryPerformanceCounter ( &tick2 );
  else
    tick2.QuadPart = timeGetTime ();
#else
  tick2 = clock ();
#endif
  app_time = TDIFF ( tick2, tick0 );
  return toc_time = TDIFF ( tick2, tick1 );
} /*TimerToc*/

double TimerTocTic ( void )
{
#ifdef USE_HIFRCLOCK
  LARGE_INTEGER tick;

  tick = tick1;
  if ( performance )
    QueryPerformanceCounter ( &tick2 );
  else
    tick2.QuadPart = timeGetTime ();
  tick1 = tick2;
#else
  clock_t tick;

  tick = tick1;  tick1 = tick2 = clock ();
#endif
  app_time = TDIFF ( tick2, tick0 );
  toc_time = 0.0;
  return toc_time = TDIFF ( tick2, tick );
} /*TimerTocTic*/
#endif

