#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "openglheader.h"

#include "utilities.h"
#include "lights.h"
#include "trans.h"
#include "bezpatches.h"
#include "teapot.h"
#include "app2a.h"
#include "app2astruct.h"
#include "app2aproc.h"

static AppData appdata;

void LoadBPShaders ( GLuint program_id[2] )
{
  static const char *filename[] =
    { "app2.vert.glsl", "app2.tesc.glsl", "app2.tese.glsl",
      "app2.geom.glsl", "app2.frag.glsl",
      "app2a1.vert.glsl", "app2a1.frag.glsl" };
  static const GLuint shtype[7] =
    { GL_VERTEX_SHADER, GL_TESS_CONTROL_SHADER, GL_TESS_EVALUATION_SHADER,
      GL_GEOMETRY_SHADER, GL_FRAGMENT_SHADER,
      GL_VERTEX_SHADER, GL_FRAGMENT_SHADER };
  GLuint shader_id[7];
  int    i;

  for ( i = 0; i < 7; i++ )
    shader_id[i] = CompileShaderFiles ( shtype[i], 1, &filename[i] );
  program_id[0] = LinkShaderProgram ( 5, shader_id, "0" );
  program_id[1] = LinkShaderProgram ( 2, &shader_id[5], "1" );
  GetAccessToTransBlockUniform ( program_id[0] );
  GetAccessToLightBlockUniform ( program_id[0] );
  GetAccessToBezPatchStorageBlocks ( program_id[0], false );
  AttachUniformTransBlockToBP ( program_id[1] );
  for ( i = 0; i < 7; i++ )
    glDeleteShader ( shader_id[i] );
  ExitIfGLError ( "LoadBPShaders" );
} /*LoadBPShaders*/

void SetupModelMatrix ( AppData *ad, float axis[3], float angle )
{
#define SCF 0.33
  GLfloat ms[16], mr[16], mt[16], ma[16];

  memcpy ( ad->model_rot_axis, axis, 3*sizeof(float) );
  ad->model_rot_angle = angle;
  M4x4Scalef ( ms, SCF, SCF, SCF*4.0/3.0 );
  M4x4Translatef ( mt, 0.0, 0.0, -0.6 );
  M4x4Multf ( ma, mt, ms );
  M4x4RotateVf ( mr, axis[0], axis[1], axis[2], angle );
  M4x4Multf ( ms, mr, ma );
  LoadMMatrix ( ad->trbuf, &ad->trans, ms );
} /*SetupModelMatrix*/

static void LoadViewMatrix ( AppData *ad )
{
  GLfloat tm[16], rm[16];

  M4x4TranslateNfv ( tm, ad->camera.viewer_pos0 );
  M4x4RotateVfv ( rm, ad->camera.viewer_rvec, -ad->camera.viewer_rangle );
  M4x4Multf ( ad->trans.vm, tm, rm );
  M4x4Transposef ( tm, rm );
  M4x4MultMVf ( ad->trans.eyepos, tm, ad->camera.viewer_pos0 );
  LoadVPMatrix ( ad->trbuf, &ad->trans );
} /*LoadViewMatrix*/

static const float zenith[3] = {0.0,0.0,1.0};

void Verticalise ( float vk[3], double *angk )
{
  float   R[16], s, c, vr[3];
  double  theta, ang;

  M4x4RotateVfv ( R, vk, *angk );
  s = -V3DotProductf ( &R[0], zenith );
  c = V3DotProductf ( &R[4], zenith );
  theta = atan2 ( s, c );
  V3CompRotationsf ( vr, &ang, &R[8], theta, vk, *angk );
  memcpy ( vk, vr, 3*sizeof(float) );
  *angk = ang;
} /*Verticalise*/

void _RotateViewer ( AppData *ad, double delta_xi, double delta_eta )
{
  float   vi[3], lgt, vk[3];
  double  angi, angk;

  vi[0] = (float)delta_eta*ad->camera.tb/(float)ad->camera.win_height;
  vi[1] = (float)delta_xi*ad->camera.rl/(float)ad->camera.win_width;
  vi[2] = 0.0;
  lgt = sqrt ( V3DotProductf ( vi, vi ) );
  vi[0] /= lgt;
  vi[1] /= lgt;
  angi = -0.052359878;  /* -3 stopnie */
  if ( ad->vertical )
    angi *= 0.01 + 0.99*fabs ( ad->trans.vm[1]*zenith[0] +
             ad->trans.vm[5]*zenith[1] + ad->trans.vm[9]*zenith[2] );
  V3CompRotationsf ( vk, &angk, ad->camera.viewer_rvec,
                     ad->camera.viewer_rangle, vi, angi );
  if ( ad->vertical )
    Verticalise ( vk, &angk );
  memcpy ( ad->camera.viewer_rvec, vk, 3*sizeof(float) );
  ad->camera.viewer_rangle = angk;
  LoadViewMatrix ( ad );
} /*_RotateViewer*/

void RotateViewer ( double delta_xi, double delta_eta )
{
  if ( delta_xi == 0.0 && delta_eta == 0.0 )
    return;  /* natychmiast uciekamy - nie chcemy dzielic przez 0 */
  _RotateViewer ( &appdata, delta_xi, delta_eta );
} /*RotateViewer*/

void InitLights ( AppData *ad )
{
  GLfloat amb0[3] = { 0.2, 0.2, 0.3 };
  GLfloat dir0[3] = { 0.8, 0.8, 0.8 };
  GLfloat pos0[4] = { 0.0, 1.0, 1.0, 0.0 };
  GLfloat atn0[3] = { 1.0, 0.0, 0.0 };

  SetLightAmbient ( ad->lsbuf, &ad->light, 0, amb0 );
  SetLightDirect ( ad->lsbuf, &ad->light, 0, dir0 );
  SetLightPosition ( ad->lsbuf, &ad->light, 0, pos0 );
  SetLightAttenuation ( ad->lsbuf, &ad->light, 0, atn0 );
  SetLightOnOff ( ad->lsbuf, &ad->light, 0, 1 );
} /*InitLights*/

void ConstructMyTeapot ( AppData *ad )
{
  const GLfloat MyColour[4] = { 1.0, 1.0, 1.0, 1.0 };

  ad->myteapot = ConstructTheTeapot ( MyColour );
  SetBezierPatchTessLevel ( ad->myteapot, ad->TessLevel );
  SetBezierPatchNVS ( ad->myteapot, ad->BezNormals );
} /*ConstructMyTeapot*/

static void _ResizeMyWorld ( AppData *ad, int width, int height )
{
  float lr;

  glViewport ( 0, 0, ad->camera.win_width = width,
               ad->camera.win_height = height );
  lr = 0.5533*(float)width/(float)height;
  M4x4Frustumf ( ad->trans.pm, NULL, ad->camera.left = -lr, ad->camera.right = lr,
                 ad->camera.bottom = -0.5533, ad->camera.top = 0.5533,
                 ad->camera.near = 5.0, ad->camera.far = 15.0 );
  ad->camera.rl = 2.0*lr;  ad->camera.tb = 2.0*0.5533;
  LoadVPMatrix ( ad->trbuf, &ad->trans );
} /*_ResizeMyWorld*/

void ResizeMyWorld ( int width, int height )
{
  _ResizeMyWorld ( &appdata, width, height );
} /*ResizeMyWorld*/

void InitCamera ( AppData *ad, int width, int height )
{
  static const float viewer_rvec[3] = {1.0,0.0,0.0};
  static const float viewer_pos0[4] = {0.0,0.0,10.0,1.0};

  memcpy ( ad->camera.viewer_rvec, viewer_rvec, 3*sizeof(float) );
  memcpy ( ad->camera.viewer_pos0, viewer_pos0, 4*sizeof(float) );
  memcpy ( ad->trans.eyepos, viewer_pos0, 4*sizeof(GLfloat) );
  ad->camera.viewer_rangle = 0.0;
  M4x4Translatef ( ad->trans.vm, -viewer_pos0[0],
                   -viewer_pos0[1], -viewer_pos0[2] );
  _ResizeMyWorld ( ad, width, height );
} /*InitCamera*/

void InitMyWorld ( int argc, char *argv[], int width, int height )
{
  float axis[4] = {0.0,0.0,1.0};

  memset ( &appdata, 0, sizeof(AppData) );
  LoadBPShaders ( appdata.program_id );
  appdata.trbuf = NewUniformTransBlock ();
  appdata.lsbuf = NewUniformLightBlock ();
  TimerInit ();
  SetupModelMatrix ( &appdata, axis, 0.0 );
  InitCamera ( &appdata, width, height );
  appdata.TessLevel = 10;
  appdata.BezNormals = GL_TRUE;
  appdata.cnet = appdata.skeleton = appdata.vertical = false;
  ConstructMyTeapot ( &appdata );
  InitLights ( &appdata );
} /*InitMyWorld*/

void DrawMyTeapot ( AppData *ad )
{
  glUseProgram ( ad->program_id[0] );
  if ( ad->skeleton )
    glPolygonMode ( GL_FRONT_AND_BACK, GL_LINE );
  else
    glPolygonMode ( GL_FRONT_AND_BACK, GL_FILL );
  DrawBezierPatches ( ad->myteapot );
} /*DrawMyTeapot*/

void DrawMyTeapotCNet ( AppData *ad )
{
  glUseProgram ( ad->program_id[1] );
  DrawBezierNets ( ad->myteapot );   
} /*DrawMyTeapotCNet*/

void RedrawMyWorld ( void )
{
  glClearColor ( 1.0, 1.0, 1.0, 1.0 );
  glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
  glEnable ( GL_DEPTH_TEST );
  DrawMyTeapot ( &appdata );
  if ( appdata.cnet )
    DrawMyTeapotCNet ( &appdata );
  glUseProgram ( 0 );
  glFlush ();
  ExitIfGLError ( "RedrawMyWorld" );
} /*RedrawMyWorld*/

void DeleteMyWorld ( void )
{
  glUseProgram ( 0 );
  glDeleteProgram ( appdata.program_id[0] );
  glDeleteProgram ( appdata.program_id[1] );
  glDeleteBuffers ( 1, &appdata.trbuf );
  glDeleteBuffers ( 1, &appdata.lsbuf );
  DeleteBezierPatches ( appdata.myteapot );
  DeleteEmptyVAO ();
} /*DeleteMyWorld*/

char ProcessCharCommand ( char charcode )
{
  switch ( toupper ( charcode ) ) {
case '+':
    if ( appdata.TessLevel < MAX_TESS_LEVEL ) {
      SetBezierPatchTessLevel ( appdata.myteapot, ++appdata.TessLevel );
      return true;
    }
    break;
case '-':
    if ( appdata.TessLevel > MIN_TESS_LEVEL ) {
      SetBezierPatchTessLevel ( appdata.myteapot, --appdata.TessLevel );
      return true;
    }
    break;
case 'N':
    SetBezierPatchNVS ( appdata.myteapot,
                        appdata.BezNormals = appdata.BezNormals == 0 );
    return true;
case 'C':
    appdata.cnet = !appdata.cnet;
    return true;
case 'S':
    appdata.skeleton = !appdata.skeleton;
    return true;
case 'V':
    if ( (appdata.vertical = !appdata.vertical) ) {
      Verticalise ( appdata.camera.viewer_rvec,
                    &appdata.camera.viewer_rangle );
      LoadViewMatrix ( &appdata );
      return true;
    }
    else
      return false;
default:
    break;
  }
  return false;
} /*ProcessCharCommand*/

char MoveOn ( void )
{
  if ( (appdata.model_rot_angle += ANGULAR_VELOCITY * TimerTocTic ()) >= PI )
    appdata.model_rot_angle -= 2.0*PI;
  SetupModelMatrix ( &appdata, appdata.model_rot_axis,
                     appdata.model_rot_angle );
  return true;
} /*MoveOn*/

