#include <stdlib.h>
#include <string.h>
#include <stdio.h>    
#include <math.h>   
#include "openglheader.h"

#include "utilities.h"
#include "bezpatches.h"


#define NBEZPATCHUOFFS 13

static GLuint bezpbbp = GL_INVALID_INDEX, cpbbp = GL_INVALID_INDEX,
              cpibbp = GL_INVALID_INDEX, txcbbp = GL_INVALID_INDEX,
              tesspbbp = GL_INVALID_INDEX;
static GLint  bezpbsize;
static GLint  bezpbofs[NBEZPATCHUOFFS];

static const GLchar *UCPNames[] = { "CPoints" };
static const GLchar *UCPINames[] = { "CPIndices" };
static const GLchar *UBezPatchNames[NBEZPATCHUOFFS+1] =
  { "BezPatch", "BezPatch.npatches", "BezPatch.dim", "BezPatch.udeg",
    "BezPatch.vdeg", "BezPatch.stride_u", "BezPatch.stride_v",
    "BezPatch.stride_p", "BezPatch.stride_q", "BezPatch.nq",
    "BezPatch.use_ind", "BezPatch.Colour",
    "BezPatch.TessLevel", "BezPatch.BezNormals" };
static const GLchar *UTexCoordNames[] = { "BezPatchTexCoord" };
static const GLchar *UTessBlockNames[] = { "BezPatchTessParams" };

void GetAccessToBezPatchStorageBlocks ( GLuint program_id,
                                        char txt, char tess )
{
  GLint size, ofs;

  if ( cpbbp == GL_INVALID_INDEX )
    GetAccessToStorageBlock ( program_id, 0, &UCPNames[0],
                              &size, &ofs, &cpbbp );
  if ( cpibbp == GL_INVALID_INDEX )
    GetAccessToStorageBlock ( program_id, 0, &UCPINames[0],
                              &size, &ofs, &cpibbp );
  if ( bezpbbp == GL_INVALID_INDEX )
    GetAccessToStorageBlock ( program_id, NBEZPATCHUOFFS, &UBezPatchNames[0],
                              &bezpbsize, bezpbofs, &bezpbbp );
  if ( txt && txcbbp == GL_INVALID_INDEX )
    GetAccessToStorageBlock ( program_id, 0, &UTexCoordNames[0],
                              &size, &ofs, &txcbbp );
  if ( tess && tesspbbp == GL_INVALID_INDEX )
    GetAccessToStorageBlock ( program_id, 0, &UTessBlockNames[0],
                              &size, &ofs, &tesspbbp );
} /*GetAccessToBezPatchStorageBlocks*/

BezierPatchObjf* EnterBezierPatches ( GLint udeg, GLint vdeg, GLint dim,
                      GLint np, GLint nq, GLint ncp, const GLfloat *cp,
                      GLint stride_p, GLint stride_q, GLint stride_u, GLint stride_v,
                      const GLfloat *colour )
{
  BezierPatchObjf *bp;
  GLint           size;
  int             zero = 0, one = 1, tl = 10;

  if ( dim < 2 || dim > 4 || np*nq < 1 ||
       udeg < 1 || udeg > MAX_PATCH_DEGREE ||
       vdeg < 1 || vdeg > MAX_PATCH_DEGREE )
    return NULL;
  bp = malloc ( sizeof(BezierPatchObjf) );
  if ( bp ) {
    memset ( bp, 0, sizeof(BezierPatchObjf) );
    bp->udeg = udeg;  bp->vdeg = vdeg;  bp->dim = dim;
    bp->npatches = np*nq;
    bp->stride_p = stride_p;  bp->stride_q = stride_q;
    bp->stride_u = stride_u;  bp->stride_v = stride_v;

    bp->buf[0] = NewStorageBuffer ( bezpbsize, bezpbbp );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[0], sizeof(GLint), &bp->npatches );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[1], sizeof(GLint), &bp->dim );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[2], sizeof(GLint), &bp->udeg );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[3], sizeof(GLint), &bp->vdeg );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[4], sizeof(GLint), &bp->stride_u );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[5], sizeof(GLint), &bp->stride_v );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[6], sizeof(GLint), &bp->stride_p );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[7], sizeof(GLint), &bp->stride_q );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[8], sizeof(GLint), &nq );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[9], sizeof(GLint), &zero );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[10], 3*sizeof(GLfloat), colour );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[11], sizeof(GLint), &tl );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[12], sizeof(GLint), &one );

    size = ncp*dim*sizeof(GLfloat);
    bp->buf[1] = NewStorageBuffer ( size, cpbbp );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, 0, size, cp );
    ExitIfGLError ( "EnterBezierPatches" );
    ConstructEmptyVAO ();
  }
  return bp;
} /*EnterBezierPatches*/

BezierPatchObjf* EnterBezierPatchesElem ( GLint udeg, GLint vdeg, GLint dim,
                      int npatches, int ncp,
                      const GLfloat *cp, const GLint *ind,
                      const GLfloat *colour )
{
  BezierPatchObjf *bp;
  GLint           size, one = 1, tl = 10;

  if ( dim < 2 || dim > 4 || npatches < 1 ||
       udeg < 1 || udeg > MAX_PATCH_DEGREE ||
       vdeg < 1 || vdeg > MAX_PATCH_DEGREE )
    return NULL;
  bp = malloc ( sizeof(BezierPatchObjf) );
  if ( bp ) {
    memset ( bp, 0, sizeof(BezierPatchObjf) );
    bp->udeg = udeg;  bp->vdeg = vdeg;  bp->dim = dim;
    bp->npatches = npatches;
    bp->buf[0] = NewStorageBuffer ( bezpbsize, bezpbbp );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[0], sizeof(GLint), &bp->npatches );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[1], sizeof(GLint), &bp->dim );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[2], sizeof(GLint), &bp->udeg );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[3], sizeof(GLint), &bp->vdeg );
    bp->stride_p = (udeg+1)*(vdeg+1);
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[6], sizeof(GLint), &bp->stride_p );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[7], sizeof(GLint), &one );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[9], sizeof(GLint), &one );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[10], 3*sizeof(GLfloat), colour );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[11], sizeof(GLint), &tl );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[12], sizeof(GLint), &one );

    size = ncp*dim*sizeof(GLfloat);
    bp->buf[1] = NewStorageBuffer ( size, cpbbp );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, 0, size, cp );

    size = (udeg+1)*(vdeg+1)*npatches*sizeof(GLint);
    bp->buf[2] = NewStorageBuffer ( size, cpibbp );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, 0, size, ind );
    ExitIfGLError ( "EnterBezierPatchesElem" );
    ConstructEmptyVAO ();
  }
  return bp;
} /*EnterBezierPatchesElem*/

void SetBezierPatchTessLevel ( BezierPatchObjf *bp, GLint tesslevel )
{
  glBindBuffer ( GL_SHADER_STORAGE_BUFFER, bp->buf[0] );
  glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[11], sizeof(GLint), &tesslevel );
  ExitIfGLError ( "SetBezierPatchTessLvel" );
} /*SetBezierPatchTessLevel*/

void SetBezierPatchNVS ( BezierPatchObjf *bp, GLint nvs )
{
  glBindBuffer ( GL_SHADER_STORAGE_BUFFER, bp->buf[0] );
  glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[12], sizeof(GLint), &nvs );
  ExitIfGLError ( "SetBezierPatchNVS" );
} /*SetBezierPatchNVS*/

void SetBezierPatchColour ( BezierPatchObjf *bp, GLfloat *colour )
{
  glBindBuffer ( GL_SHADER_STORAGE_BUFFER, bp->buf[0] );
  glBufferSubData ( GL_SHADER_STORAGE_BUFFER, bezpbofs[10], 3*sizeof(GLfloat), colour );
  ExitIfGLError ( "SetBezierPatchColour" );
} /*SetBezierPatchColour*/

char GenBezierPatchTextureBlock ( BezierPatchObjf *bp, const GLfloat *data )
{
  GLint size;

  size = bp->npatches*4*sizeof(GLfloat);
  if ( !(bp->buf[3] = NewStorageBuffer ( size, txcbbp )) )
    return false;
  glBufferSubData ( GL_SHADER_STORAGE_BUFFER, 0, size, data );
  ExitIfGLError ( "GenBezierPatchTextureBlock" );
  return true;
} /*GenBezierPatchTextureBlock*/

void BindBezPatchTextureBuffer ( BezierPatchObjf *bp )
{
  glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, txcbbp, bp->buf[3] );
} /*BindBezPatchTextureBuffer*/

char GenBezierPatchTessParamBlock ( BezierPatchObjf *bp )
{
  GLint size;

  size = bp->npatches*12*sizeof(GLfloat);
  if ( !(bp->buf[4] = NewStorageBuffer ( size, tesspbbp )) )
    return false;
  ExitIfGLError ( "GenBezierPatchTessParamBlock" );
  return true;
} /*GenBezierPatchTessParamBlock*/

void BindBezPatchTessParamBuffer ( BezierPatchObjf *bp )
{
  glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, tesspbbp, bp->buf[4] );
} /*BindBezPatchTessParamBuffer*/

void BindBezierPatchBuffers ( BezierPatchObjf *bp )
{
  if ( bp ) {
    glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, bezpbbp, bp->buf[0] );
    glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, cpbbp, bp->buf[1] );
    if ( bp->buf[2] )
      glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, cpibbp, bp->buf[2] );
    if ( bp->buf[3] )
      glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, txcbbp, bp->buf[3] );
    if ( bp->buf[4] )
      glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, tesspbbp, bp->buf[4] );
    ExitIfGLError ( "BindBezierPatchBuffers" );
  }
} /*BindBezierPatchBuffers*/

void DrawBezierPatches ( BezierPatchObjf *bp )
{
  if ( bp ) {
    BindBezierPatchBuffers ( bp );
    glBindVertexArray ( empty_vao );
    glPatchParameteri ( GL_PATCH_VERTICES, 1 );
    glDrawArraysInstanced ( GL_PATCHES, 0, 1, bp->npatches );
    glBindVertexArray ( 0 );
    ExitIfGLError ( "DrawBezierPatches" );
  }
} /*DrawBezierPatch*/

void DrawBezierNets ( BezierPatchObjf *bp )
{
  int nlines;

  if ( bp ) {
    BindBezierPatchBuffers ( bp );
    glBindVertexArray ( empty_vao );
    nlines = 2*bp->udeg*bp->vdeg + bp->udeg + bp->vdeg;
    glDrawArraysInstanced ( GL_LINES, 0, 2, bp->npatches*nlines );
    glBindVertexArray ( 0 );
    ExitIfGLError ( "DrawBezierNets" );
  }
} /*DrawBezierNets*/

void DeleteBezierPatches ( BezierPatchObjf *bp )
{
  if ( bp ) {
    glDeleteBuffers ( 5, bp->buf );
    free ( bp );
  }
} /*DeleteBezierPatches*/
