#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include "../utilities/openglheader.h"

#include "../utilities/utilities.h"
#include "trans.h"
#include "lights.h"

static GLuint lsbbp = GL_INVALID_INDEX;
static GLint  lsbsize, lsbofs[NLSUOFFS];

static GLuint matbbp = GL_INVALID_INDEX;
static GLint  matbsize, matbofs[NMATUOFFS];

static const GLchar *ULSNames[] =
  { "LSBlock", "LSBlock.nls", "LSBlock.mask", "LSBlock.slmask",
    "LSBlock.shmask", "LSBlock.cshmask", "LSBlock.max_depth",
    "LSBlock.lightn",
    "LSBlock.ls[0].ambient", "LSBlock.ls[0].direct",
    "LSBlock.ls[0].position", "LSBlock.ls[0].attenuation",
    "LSBlock.ls[0].shadow_vpm", "LSBlock.ls[1].ambient" };

static const GLchar *UMatNames[] =
  { "MatBlock", "MatBlock.mtn", "MatBlock.mat[0].emission0",
    "MatBlock.mat[0].emission1", "MatBlock.mat[0].diffref",
    "MatBlock.mat[0].specref", "MatBlock.mat[0].shininess",
    "MatBlock.mat[0].wa", "MatBlock.mat[0].we",
    "MatBlock.mat[0].txtnum", "MatBlock.mat[1].emission0" };

GLuint GetAccessToLightBlockUniform ( GLuint program_id )
{
  if ( lsbbp == GL_INVALID_INDEX )
    GetAccessToUniformBlock ( program_id, NLSUOFFS, &ULSNames[0],
                              &lsbsize, lsbofs, &lsbbp );
  else
    AttachUniformBlockToBP ( program_id, ULSNames[0], lsbbp );
  return lsbbp;
} /*GetAccessToLightBlockUniform*/

GLuint NewUniformLightBlock ( void )
{
  return NewUniformBuffer ( lsbsize, lsbbp );
} /*NewUniformLightBlock*/

void AttachUniformLightBlockToBP ( GLuint program_id )
{
  AttachUniformBlockToBP ( program_id, ULSNames[0], lsbbp );
} /*AttachUniformLightBlockToBP*/

void SetLightSurfBit ( LightBl *light, int l, char bit )
{
  GLuint mask;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  mask = 0x00000001 << l;
  light->slmask = bit ? light->slmask | mask : light->slmask & ~mask;
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, lsbofs[2], sizeof(GLuint), &light->slmask );
  ExitIfGLError ( "SetLightSurfBit" );
} /*SetLightSurfBit*/

void SetLightAmbient ( LightBl *light, int l, GLfloat amb[4] )
{
  GLint ofs;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  memcpy ( light->ls[l].ambient, amb, 4*sizeof(GLfloat) );
  ofs = l*(lsbofs[NLSUOFFS-1]-lsbofs[7]) + lsbofs[7];
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, ofs, 4*sizeof(GLfloat), amb );
  ExitIfGLError ( "SetLightAmbient" );
} /*SetLightAmbient*/

void SetLightDiffuse ( LightBl *light, int l, GLfloat dif[4] )
{
  GLint ofs;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  memcpy ( light->ls[l].diffuse, dif, 4*sizeof(GLfloat) );
  ofs = l*(lsbofs[NLSUOFFS-1]-lsbofs[7]) + lsbofs[8];
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, ofs, 4*sizeof(GLfloat), dif );
  ExitIfGLError ( "SetLightDiffuse" );
} /*SetLightDiffuse*/

void SetLightPosition ( LightBl *light, int l, GLfloat pos[4] )
{
  GLint   ofs;
  GLfloat w, *p;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  memcpy ( p = light->ls[l].position, pos, 4*sizeof(GLfloat) );
  w = p[3];
  if ( w != 0.0 && w != 1.0 )
    p[0] /= w, p[1] /= w, p[2] /= w, p[3] = 1.0;
  ofs = l*(lsbofs[NLSUOFFS-1]-lsbofs[7]) + lsbofs[9];
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, ofs, 4*sizeof(GLfloat), p );
  ExitIfGLError ( "SetLightPosition" );
} /*SetLightPosition*/

void SetLightAttenuation ( LightBl *light, int l, GLfloat atn[3] )
{
  GLint ofs;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  memcpy ( light->ls[l].attenuation, atn, 3*sizeof(GLfloat) );
  ofs = l*(lsbofs[NLSUOFFS-1]-lsbofs[7]) + lsbofs[10];
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, ofs, 3*sizeof(GLfloat), atn );
  ExitIfGLError ( "SetLightAttenuation" );
} /*SetLightAttenuation*/

void SetLightMaxDepth ( LightBl *light, GLfloat max_depth )
{
  light->max_depth = max_depth;
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, lsbofs[5], sizeof(GLfloat), &max_depth );
  ExitIfGLError ( "SetLightMaxDepth" );
} /*SetLightMaxDepth*/

void SetLightOnOff ( LightBl *light, int l, char on )
{
  GLuint mask;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  mask = 0x00000001 << l;
  if ( on ) {
    light->mask |= mask;
    if ( l >= light->nls )
      light->nls = l+1;
  }
  else {
    light->mask &= ~mask;
    for ( mask = 0x00000001 << (light->nls-1); mask; mask >>= 1 ) {
      if ( light->mask & mask )
        break;
      else
        light->nls --;
    }
  }
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, lsbofs[0], sizeof(GLuint), &light->nls );
  glBufferSubData ( GL_UNIFORM_BUFFER, lsbofs[1], sizeof(GLuint), &light->mask );
  ExitIfGLError ( "SetLightOnOff" );
} /*SetLightOnOff*/

/* ////////////////////////////////////////////////////////////////////////// */
void ConstructShadowTxtFBO ( LightBl *light, int l, char cubesh )
{
  GLuint fbo, txt, mask;
  GLenum target;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  glGenTextures ( 1, &txt );
  glGenFramebuffers ( 1, &fbo );
  if ( (light->ls[l].shadow_txt = txt) &&
       (light->ls[l].shadow_fbo = fbo) ) {
    glBindFramebuffer ( GL_FRAMEBUFFER, fbo );
    glActiveTexture ( GL_TEXTURE0+MAX_TEXTURES+l );
    mask = 0x01 << l;
    glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
    if ( cubesh ) {
      light->cshmask |= mask;
      glBufferSubData ( GL_UNIFORM_BUFFER, lsbofs[4], sizeof(GLuint),
                        &light->cshmask );
      target = GL_TEXTURE_CUBE_MAP;
      glBindTexture ( target, txt );
    }
    else {
      light->shmask |= mask;
      glBufferSubData ( GL_UNIFORM_BUFFER, lsbofs[3], sizeof(GLuint),
                        &light->shmask );
      target = GL_TEXTURE_2D;
      glBindTexture ( target, txt );
      glTexParameteri ( target, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
      glTexParameteri ( target, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
    }
    glTexStorage2D ( target, 1, GL_DEPTH_COMPONENT32,
                     SHADOW_MAP_SIZE, SHADOW_MAP_SIZE );
    glTexParameteri ( target, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
    glTexParameteri ( target, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
    glBindTexture ( target, 0 );
    glFramebufferTexture ( GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, txt, 0 );
    glDrawBuffer ( GL_NONE );
    if ( glCheckFramebufferStatus ( GL_FRAMEBUFFER ) != GL_FRAMEBUFFER_COMPLETE )
      ExitOnError ( "ConstructShadowTxtFBO" );
    glBindFramebuffer ( GL_FRAMEBUFFER, 0 );
    ExitIfGLError ( "ConstructShadowTxtFBO" );
  }
} /*ConstructShadowTxtFBO*/

static void SetupShadowViewerTrans ( GLfloat org[3], GLfloat zv[3],
                                     GLfloat vmat[16] )
{
  GLfloat v[3], a[16], g, r, s;
  int     i, j;

        /* skonstruuj odbicie Householdera */
  memcpy ( v, zv, 3*sizeof(GLfloat) );
  r = sqrt ( V3DotProductf ( v, v ) );
  v[2] += v[2] > 0.0 ? r : -r;
  g = 2.0/V3DotProductf ( v, v );
        /* odbij kolumny macierzy jednostkowej */
  M4x4Identf ( a );
  for ( i = 0; i < 2; i++ ) {
    s = v[i]*g;
    for ( j = 0; j < 3; j++ )
      a[4*i+j] -= s*v[j];
  }
  a[8] = zv[0]/r;  a[9] = zv[1]/r;  a[10] = zv[2]/r;
  memcpy ( &a[12], org, 3*sizeof(GLfloat) );
  M4x4InvertAffineIsometryf ( vmat, a );
} /*SetupShadowViewerTrans*/

void UpdateShadowMatrix ( LightBl *light, int l )
{
  int           ofs;
  GLfloat       lvpm[16], a[16];
  const GLfloat b[16] = {0.5,0.0,0.0,0.0,0.0,0.5,0.0,0.0,
                         0.0,0.0,0.5,0.0,0.5,0.5,0.5,1.0};

  if ( light->shmask & (0x01 << l) ) {
    glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
    ofs = l*(lsbofs[NLSUOFFS-1]-lsbofs[7]) + lsbofs[11];
    M4x4Multf ( a, light->ls[l].shadow_proj, light->ls[l].shadow_view );
    M4x4Multf ( lvpm, b, a );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs, 16*sizeof(GLfloat), lvpm );
    ExitIfGLError ( "UpdateShadowMatrix" );
  }
} /*UpdateShadowMatrix*/

void SetupShadowTxtTransformations ( LightBl *light, int l, float sc[3], float R )
{
  GLfloat *lvm, *lpm;
  GLfloat lpos[3], v[3], d, n, s, t;
  int     i;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  if ( light->ls[l].shadow_txt ) {
    lvm = light->ls[l].shadow_view;
    lpm = light->ls[l].shadow_proj;
    if ( light->ls[l].position[3] != 0.0 ) {
            /* zrodlo swiatla w skonczonej odleglosci */
      if ( light->cshmask & (0x01 << l ) ) {
        n = 0.01*R;
        M4x4Frustumf ( lpm, NULL, -n, n, -n, n, n, R );
      }
      else {
        for ( i = 0; i < 3; i++ ) {
          lpos[i] = light->ls[l].position[i]/light->ls[l].position[3];
          v[i] = lpos[i] - sc[i];
        }
        SetupShadowViewerTrans ( lpos, v, lvm );
        d = V3DotProductf ( v, v );
        if ( d > 2.0*R*R ) {
          d = sqrt ( d );  n = d-R;
          s = R/d;                  /* sin(alpha) */
          t = s /sqrt ( 1.0-s*s );  /* tg(alpha)*/
          M4x4Frustumf ( lpm, NULL, -n*t, n*t, -n*t, n*t, n, d+R );
        }
        else {
          n = 0.4142*R;
          M4x4Frustumf ( lpm, NULL, -n, n, -n, n, n, 2.4143*R );
        }
      }
    }
    else {  /* zrodlo swiatla w odleglosci nieskonczonej */
      memcpy ( lpos, sc, 3*sizeof(GLfloat) );
      memcpy ( v, light->ls[l].position, 3*sizeof(GLfloat) );
      SetupShadowViewerTrans ( lpos, v, lvm );
      M4x4Orthof ( lpm, NULL, -R, R, -R, R, -R, R );
    }
  }
} /*SetupShadowTxtTransformations*/

void BindShadowTxtFBO ( LightBl *light, TransBl *trans, int l )
{
  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  if ( (light->ls[l].shadow_txt == 0) )
    return;
  LoadAltVPMatrix ( trans, light->ls[l].shadow_view,
                    light->ls[l].shadow_proj, light->ls[l].position );
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, lsbofs[6], sizeof(GLint), &l );
  glBindFramebuffer ( GL_FRAMEBUFFER, light->ls[l].shadow_fbo );
  ExitIfGLError ( "BindShadowTxtFBO" );
} /*BindShadowTxtFBO*/

void DestroyShadowFBO ( LightBl *light )
{
  int l;

  glBindFramebuffer ( GL_FRAMEBUFFER, 0 );
  for ( l = 0; l < MAX_NLIGHTS; l++ ) {
    if ( light->ls[l].shadow_fbo != 0 )
      glDeleteFramebuffers ( 1, &light->ls[l].shadow_fbo );
    if ( light->ls[l].shadow_txt != 0 )
      glDeleteTextures ( 1, &light->ls[l].shadow_txt );
  }
  ExitIfGLError ( "DestroyShadowFBO" );
} /*DestroyShadowFBO*/

/* ////////////////////////////////////////////////////////////////////////// */
GLuint GetAccessToMatBlockUniform ( GLuint program_id )
{
  if ( matbbp == GL_INVALID_INDEX )
    GetAccessToUniformBlock ( program_id, NMATUOFFS, &UMatNames[0],
                              &matbsize, matbofs, &matbbp );
  else
    AttachUniformBlockToBP ( program_id, UMatNames[0], matbbp );
  return matbbp;
} /*GetAccessToMatBlockUniform*/

GLuint NewUniformMatBlock ( void )
{
  return NewUniformBuffer ( matbsize, matbbp );
} /*NewUniformMatBlock*/

void AttachUniformMatBlockToBP ( GLuint program_id )
{
  AttachUniformBlockToBP ( program_id, UMatNames[0], matbbp );
} /*AttachUniformMatBlockToBP*/

int AllocMaterialID ( MatBl *mat )
{
  if ( mat->nmat >= MAX_MATERIALS )
    ExitOnError ( "AllocMaterialID" );
  return mat->nmat ++;
} /*AllocMaterialID*/

int AllocTextureID ( MatBl *mat )
{
  if ( mat->ntex >= MAX_TEXTURES )
    ExitOnError ( "AllocTextureID" );
  return mat->ntex ++;
} /*AllocTextureID*/

int SetupMaterial ( MatBl *matbl, int m,
                    const GLfloat diffr[4], const GLfloat specr[4],
                    GLfloat shn, GLfloat wa, GLfloat we,
                    GLuint txtid )
{
  GLint   ofs, tn;
  GLfloat em[4] = { 0.0, 0.0, 0.0, 0.0 };

  if ( m < 0 )
    m = AllocMaterialID ( matbl );
  if ( m < MAX_MATERIALS ) {
    tn = txtid != GL_INVALID_INDEX ? AllocTextureID ( matbl ) : -1;
    memcpy ( matbl->mat[m].diffref, diffr, 4*sizeof(GLfloat) );
    memcpy ( matbl->mat[m].specref, specr, 4*sizeof(GLfloat) );
    matbl->mat[m].shininess = shn;
    matbl->mat[m].wa = wa;
    matbl->mat[m].we = we;
    matbl->mat[m].txtid = txtid;
    matbl->mat[m].txtbp = tn;
    ofs = m*(matbofs[NMATUOFFS-1]-matbofs[1]);
    glBindBuffer ( GL_UNIFORM_BUFFER, matbl->matbuf );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[1], 4*sizeof(GLfloat), em );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[2], 4*sizeof(GLfloat), em );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[3], 4*sizeof(GLfloat), diffr );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[4], 4*sizeof(GLfloat), specr );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[5], sizeof(GLfloat), &shn );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[6], sizeof(GLfloat), &wa );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[7], sizeof(GLfloat), &we );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[8], sizeof(GLint), &tn );
    ExitIfGLError ( "SetupMaterial" );
  }
  return m;
} /*SetupMaterial*/

void SetMaterialEmission ( MatBl *matbl, int m, const GLfloat em0[4],
                           const GLfloat em1[4] )
{
  GLint ofs;

  if ( m >= 0 && m < matbl->nmat ) {
    ofs = m*(matbofs[NMATUOFFS-1]-matbofs[1]);
    glBindBuffer ( GL_UNIFORM_BUFFER, matbl->matbuf );
    memcpy ( matbl->mat[m].emission0, em0, 4*sizeof(GLfloat) );
    memcpy ( matbl->mat[m].emission1, em1, 4*sizeof(GLfloat) );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[1], 4*sizeof(GLfloat), em0 );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[2], 4*sizeof(GLfloat), em1 );
    ExitIfGLError ( "SetMaterialEmission" );
  }
} /*SetMaterialEmission*/

void BindMaterialTextures ( MatBl *matbl )
{
  int i;

  glBindBufferBase ( GL_UNIFORM_BUFFER, matbbp, matbl->matbuf );
  for ( i = 0; i < matbl->nmat; i++ )
    if ( matbl->mat[i].txtid != GL_INVALID_INDEX ) {
      glActiveTexture ( GL_TEXTURE0 + matbl->mat[i].txtbp );
      glBindTexture ( GL_TEXTURE_2D, matbl->mat[i].txtid );
    }
} /*BindMaterialTextures*/

void ChooseMaterial ( MatBl *matbl, GLint m )
{
  Material *mat;

  if ( m < 0 || m >= matbl->nmat )
    m = 0;
  glBindBuffer ( GL_UNIFORM_BUFFER, matbl->matbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, matbofs[0], sizeof(GLint), &m );
  mat = &matbl->mat[m];
  if ( mat->txtbp != GL_INVALID_INDEX ) {
    glActiveTexture ( GL_TEXTURE0+mat->txtbp );
    glBindTexture ( GL_TEXTURE_2D, mat->txtid );
  }
  ExitIfGLError ( "ChooseMaterial" );
} /*ChooseMaterial*/

void DeleteMaterials ( MatBl *matbl )
{
  int i;

  for ( i = 0; i < matbl->nmat; i++ ) {
    if ( matbl->mat[i].txtid != GL_INVALID_INDEX ) {
      if ( matbl->mat[i].txtbp != GL_INVALID_INDEX ) {
        glActiveTexture ( GL_TEXTURE0+matbl->mat[i].txtbp );
        glBindTexture ( GL_TEXTURE_2D, 0 );
      }
      glDeleteTextures ( 1, &matbl->mat[i].txtid );
    }
  }
  glDeleteBuffers ( 1, &matbl->matbuf );
  memset ( matbl, 0, sizeof(MatBl) );
} /*DeleteMaterials*/

