#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include "../utilities/openglheader.h"

#include "../utilities/utilities.h"
#include "trans.h"
#include "lights.h"

#define NLSOFFS   9
#define NMATUOFFS 9

static GLuint lsbbp = GL_INVALID_INDEX, matbbp = GL_INVALID_INDEX;
static GLint  lsbsize, lsbofs[NLSOFFS], matbsize, matbofs[NMATUOFFS];

static const GLchar *ULSNames[] =
  { "LSBlock", "LSBlock.nls", "LSBlock.mask", "LSBlock.shmask",
    "LSBlock.ls[0].ambient", "LSBlock.ls[0].direct", "LSBlock.ls[0].position",
    "LSBlock.ls[0].attenuation", "LSBlock.ls[0].shadow_vpm",
    "LSBlock.ls[1].ambient" };
static const GLchar *UMatNames[] =
  { "MatBlock", "MatBlock.mtn", "MatBlock.mat[0].emission0",
    "MatBlock.mat[0].emission1", "MatBlock.mat[0].diffref",
    "MatBlock.mat[0].specref", "MatBlock.mat[0].shininess",
    "MatBlock.mat[0].wa", "MatBlock.mat[0].we",
    "MatBlock.mat[1].emission0" };

void GetAccessToLightMatUniformBlocks ( GLuint program_id )
{
  if ( lsbbp == GL_INVALID_INDEX )
    GetAccessToUniformBlock ( program_id, NLSOFFS, &ULSNames[0],
                              &lsbsize, lsbofs, &lsbbp );
  else
    AttachUniformBlockToBP ( program_id, ULSNames[0], lsbbp );
  if ( matbbp == GL_INVALID_INDEX )
    GetAccessToUniformBlock ( program_id, NMATUOFFS, &UMatNames[0],
                              &matbsize, matbofs, &matbbp );
  else
    AttachUniformBlockToBP ( program_id, UMatNames[0], matbbp );
} /*GetAccessToLightMatUniformBlocks*/

void AttachUniformLightMatBlockToBP ( GLuint program_id )
{
  AttachUniformBlockToBP ( program_id, ULSNames[0], lsbbp );
  AttachUniformBlockToBP ( program_id, UMatNames[0], matbbp );
} /*AttachUniformLightMatBlockToBP*/

GLuint NewUniformLightBlock ( void )
{
  return NewUniformBuffer ( lsbsize, lsbbp );
} /*NewUniformLightBlock*/

void SetLightAmbient ( LightBl *light, int l, GLfloat amb[3] )
{
  GLint ofs;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  memcpy ( light->ls[l].ambient, amb, 3*sizeof(GLfloat) );
  ofs = l*(lsbofs[8]-lsbofs[3]) + lsbofs[3];
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, ofs, 3*sizeof(GLfloat), amb );
  ExitIfGLError ( "SetLightAmbient" );
} /*SetLightAmbient*/

void SetLightDirect ( LightBl *light, int l, GLfloat dir[3] )
{
  GLint ofs;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  memcpy ( light->ls[l].direct, dir, 3*sizeof(GLfloat) );
  ofs = l*(lsbofs[8]-lsbofs[3]) + lsbofs[4];
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, ofs, 3*sizeof(GLfloat), dir );
  ExitIfGLError ( "SetLightDirect" );
} /*SetLightDirect*/

void SetLightPosition ( LightBl *light, int l, GLfloat pos[4] )
{
  GLint   ofs;
  GLfloat w, *p;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  memcpy ( p = light->ls[l].position, pos, 4*sizeof(GLfloat) );
  w = p[3];
  if ( w != 0.0 && w != 1.0 )
    p[0] /= w, p[1] /= w, p[2] /= w, p[3] = 1.0;
  ofs = l*(lsbofs[8]-lsbofs[3]) + lsbofs[5];
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, ofs, 4*sizeof(GLfloat), p );
  ExitIfGLError ( "SetLightPosition" );
} /*SetLightPosition*/

void SetLightAttenuation ( LightBl *light, int l, GLfloat atn[3] )
{
  GLint ofs;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  memcpy ( light->ls[l].attenuation, atn, 3*sizeof(GLfloat) );
  ofs = l*(lsbofs[8]-lsbofs[3]) + lsbofs[6];
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, ofs, 3*sizeof(GLfloat), atn );
  ExitIfGLError ( "SetLightAttenuation" );
} /*SetLightAttenuation*/

void SetLightOnOff ( LightBl *light, int l, char on )
{
  GLuint mask;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  mask = 0x01 << l;
  if ( on ) {
    light->mask |= mask;
    if ( l >= light->nls )
      light->nls = l+1;
  }
  else {
    light->mask &= ~mask;
    for ( mask = 0x01 << (light->nls-1); mask; mask >>= 1 ) {
      if ( light->mask & mask )
        break;
      else
        light->nls --;
    }
  }
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, lsbofs[0], sizeof(GLuint), &light->nls );
  glBufferSubData ( GL_UNIFORM_BUFFER, lsbofs[1], sizeof(GLuint), &light->mask );
  ExitIfGLError ( "SetLightOnOff" );
} /*SetLightOnOff*/

/* ////////////////////////////////////////////////////////////////////////// */
void ConstructShadowTxtFBO ( LightBl *light, int l )
{
  GLuint fbo, txt;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  glGenTextures ( 1, &txt );
  glGenFramebuffers ( 1, &fbo );
  if ( (light->ls[l].shadow_txt = txt) &&
       (light->ls[l].shadow_fbo = fbo) ) {
    glActiveTexture ( GL_TEXTURE2+l );
    glBindTexture ( GL_TEXTURE_2D, txt );
    glTexImage2D ( GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT32,
                   SHADOW_MAP_SIZE, SHADOW_MAP_SIZE, 0,
                   GL_DEPTH_COMPONENT, GL_FLOAT, NULL );
    glTexParameteri ( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
    glTexParameteri ( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
    glTexParameteri ( GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE,
                      GL_COMPARE_REF_TO_TEXTURE );
    glTexParameteri ( GL_TEXTURE_2D, GL_TEXTURE_COMPARE_FUNC, GL_LEQUAL );
    glTexParameteri ( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
    glTexParameteri ( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
    glBindTexture ( GL_TEXTURE_2D, 0 );
    glBindFramebuffer ( GL_FRAMEBUFFER, fbo );
    glFramebufferTexture ( GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT,
                           light->ls[l].shadow_txt, 0 );
    glDrawBuffer ( GL_NONE );
    light->shmask |= 0x01 << l;
    glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
    glBufferSubData ( GL_UNIFORM_BUFFER, lsbofs[2], sizeof(GLuint), &light->shmask );
    glBindFramebuffer ( GL_FRAMEBUFFER, 0 );
    ExitIfGLError ( "ConstructShadowTxtFBO" );
  }
} /*ConstructShadowTxtFBO*/

void UpdateShadowMatrix ( LightBl *light, int l )
{
  int           ofs;
  GLfloat       lvpm[16], a[16];
  const GLfloat b[16] = {0.5,0.0,0.0,0.0,0.0,0.5,0.0,0.0,
                         0.0,0.0,0.5,0.0,0.5,0.5,0.5,1.0};

  M4x4Multf ( a, light->ls[l].shadow_proj, light->ls[l].shadow_view );
  M4x4Multf ( lvpm, b, a );
  ofs = l*(lsbofs[8]-lsbofs[3]) + lsbofs[7];
  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, ofs, 16*sizeof(GLfloat), lvpm );
  ExitIfGLError ( "UpdateShadowMatrix" );
} /*UpdateShadowMatrix*/

static void SetupShadowViewerTrans ( GLfloat org[3], GLfloat zv[3],
                                     GLfloat vmat[16] )
{
  GLfloat v[3], a[16], g, r, s;
  int     i, j;

        /* skonstruuj odbicie Householdera */
  memcpy ( v, zv, 3*sizeof(GLfloat) );
  r = sqrt ( V3DotProductf ( v, v ) );
  v[2] += v[2] > 0.0 ? r : -r;
  g = 2.0/V3DotProductf ( v, v );
        /* odbij kolumny macierzy jednostkowej */
  M4x4Identf ( a );
  for ( i = 0; i < 2; i++ ) {
    s = v[i]*g;
    for ( j = 0; j < 3; j++ )
      a[4*i+j] -= s*v[j];
  }
  a[8] = zv[0]/r;  a[9] = zv[1]/r;  a[10] = zv[2]/r;
  memcpy ( &a[12], org, 3*sizeof(GLfloat) );
  M4x4InvertAffineIsometryf ( vmat, a );
} /*SetupShadowViewerTrans*/

void SetupShadowTxtTransformations ( LightBl *light, int l, float sc[3], float R )
{
  GLfloat *lvm, *lpm;
  GLfloat lpos[3], v[3], d, n, s, t;
  int     i;

  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  if ( light->ls[l].shadow_txt ) {
    lvm = light->ls[l].shadow_view;
    lpm = light->ls[l].shadow_proj;
    if ( light->ls[l].position[3] != 0.0 ) {
            /* zrodlo swiatla w skonczonej odleglosci */
      for ( i = 0; i < 3; i++ ) {
        lpos[i] = light->ls[l].position[i]/light->ls[l].position[3];
        v[i] = lpos[i] - sc[i];
      }
      SetupShadowViewerTrans ( lpos, v, lvm );
      d = V3DotProductf ( v, v );
      if ( d > 2.0*R*R ) {
        d = sqrt ( d );  n = d-R;
        s = R/d;                  /* sin(alpha) */
        t = s /sqrt ( 1.0-s*s );  /* tg(alpha)*/
        M4x4Frustumf ( lpm, NULL, -n*t, n*t, -n*t, n*t, n, d+R );
      }
      else {
        n = 0.4142*R;
        M4x4Frustumf ( lpm, NULL, -n, n, -n, n, n, 2.4143*R );
      }
    }
    else {  /* zrodlo swiatla w odleglosci nieskonczonej */
      memcpy ( lpos, sc, 3*sizeof(GLfloat) );
      memcpy ( v, light->ls[l].position, 3*sizeof(GLfloat) );
      SetupShadowViewerTrans ( lpos, v, lvm );
      M4x4Orthof ( lpm, NULL, -R, R, -R, R, -R, R );
    }
  }
} /*SetupShadowTxtTransformations*/

void BindShadowTxtFBO ( TransBl *trans, LightBl *light, int l )
{
  if ( l < 0 || l >= MAX_NLIGHTS )
    return;
  if ( (light->ls[l].shadow_txt == 0) )
    return;
  LoadShTrans ( trans, light->ls[l].shadow_view,
                light->ls[l].shadow_proj, light->ls[l].position );
  glBindFramebuffer ( GL_FRAMEBUFFER, light->ls[l].shadow_fbo );
  ExitIfGLError ( "BindShadowTxtFBO" );
} /*BindShadowTxtFBO*/

void DeleteShadowFBO ( LightBl *light )
{
  int l;

  glBindFramebuffer ( GL_FRAMEBUFFER, 0 );
  for ( l = 0; l < MAX_NLIGHTS; l++ ) {
    if ( light->ls[l].shadow_fbo != 0 )
      glDeleteFramebuffers ( 1, &light->ls[l].shadow_fbo );
    if ( light->ls[l].shadow_txt != 0 )
      glDeleteTextures ( 1, &light->ls[l].shadow_txt );
  }
  ExitIfGLError ( "DeleteShadowFBO" );
} /*DeleteShadowFBO*/

void SetShadowsOnOff ( LightBl *light, char on )
{
  GLuint z = 0;

  glBindBuffer ( GL_UNIFORM_BUFFER, light->lsbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, lsbofs[2], sizeof(GLuint),
                    on ? &light->shmask : &z );
} /*SetShadowsOnOff*/

/* ////////////////////////////////////////////////////////////////////////// */
GLuint NewUniformMatBlock ( void ) 
{
  return NewUniformBuffer ( matbsize, matbbp );
} /*NewUniformMatBlock*/

int AllocMaterialID ( MatBl *mat )
{
  if ( mat->nmat >= MAX_MATERIALS )
    ExitOnError ( "AllocMaterialID" );
  return mat->nmat ++;
} /*AllocMaterialID*/

int SetupMaterial ( MatBl *matbl, int m, const GLfloat diffr[3],
                    const GLfloat specr[3], GLfloat shn, GLfloat wa, GLfloat we )
{
  GLint ofs;
  GLfloat em[4] = { 0.0, 0.0, 0.0, 1.0 };

  if ( m < 0 )
    m = AllocMaterialID ( matbl );
  if ( m < MAX_MATERIALS ) {
    memcpy ( matbl->mat[m].diffref, diffr, 4*sizeof(GLfloat) );
    memcpy ( matbl->mat[m].specref, specr, 4*sizeof(GLfloat) );
    matbl->mat[m].shininess = shn;
    matbl->mat[m].wa = wa;
    matbl->mat[m].we = we;
    ofs = m*(matbofs[NMATUOFFS-1]-matbofs[1]);
    glBindBuffer ( GL_UNIFORM_BUFFER, matbl->matbuf );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[1], 4*sizeof(GLfloat), em );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[2], 4*sizeof(GLfloat), em );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[3], 3*sizeof(GLfloat), diffr );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[4], 3*sizeof(GLfloat), specr );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[5], sizeof(GLfloat), &shn );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[6], sizeof(GLfloat), &wa );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[7], sizeof(GLfloat), &we );
    ExitIfGLError ( "SetupMaterial" );
  }
  return m;
} /*SetupMaterial*/

void SetMaterialEmission ( MatBl *matbl, int m,
                           const GLfloat em0[4], const GLfloat em1[4] )
{
  GLint ofs;

  if ( m >= 0 && m < matbl->nmat ) {
    ofs = m*(matbofs[NMATUOFFS-1]-matbofs[1]);
    glBindBuffer ( GL_UNIFORM_BUFFER, matbl->matbuf );
    memcpy ( matbl->mat[m].emission0, em0, 4*sizeof(GLfloat) );
    memcpy ( matbl->mat[m].emission1, em1, 4*sizeof(GLfloat) );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[1], 4*sizeof(GLfloat), em0 );
    glBufferSubData ( GL_UNIFORM_BUFFER, ofs+matbofs[2], 4*sizeof(GLfloat), em1 );
    ExitIfGLError ( "SetMaterialEmission" );
  }
} /*SetMaterialEmission*/

void ChooseMaterial ( MatBl *matbl, GLint m )
{
  if ( m < 0 || m >= matbl->nmat )
    m = 0;
  glBindBuffer ( GL_UNIFORM_BUFFER, matbl->matbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, matbofs[0], sizeof(GLint), &m );
  ExitIfGLError ( "ChooseMaterial" );
} /*ChooseMaterial*/

