#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include "../utilities/openglheader.h"

#include "app3c.h"
#include "../utilities/utilities.h"
#include "../utilities/linkage.h"
#include "../utilities/bezpatches.h"
#include "../utilities/GPUsparsemat.h"
#include "../utilities/meshes.h"
#include "trans.h"
#include "lights.h"
#include "palm.h"
#include "app3cproc.h"
#include "app3cstruct.h"

static float palmartp0[NKLARTPARAMS] =
    {0.0,0.0,0.0,0.0,0.2,0.0,0.0,0.0,0.5,0.0,0.0,
     0.0,0.5,0.0,0.0,0.0,0.8,0.0,0.0,0.0};

static const float palmartprange[NKLARTPARAMS][2] =
  {{0.0,-0.5*PI},{0.0,-0.5},{0.0,0.4*PI},{0.0,0.4*PI},
   {0.025*PI,-0.1*PI},{0.0,0.5*PI},{0.0,0.5*PI},{0.0,0.5*PI},
   {0.025*PI,-0.025*PI},{0.0,0.5*PI},{0.0,0.5*PI},{0.0,0.5*PI},
   {0.025*PI,-0.025*PI},{0.0,0.5*PI},{0.0,0.5*PI},{0.0,0.5*PI},
   {0.12*PI,-0.03*PI},{0.0,0.5*PI},{0.0,0.5*PI},{0.0,0.5*PI},
   {0.0,0.2}};


void LoadLinkageArticulationProgram ( KLArticulationProgram *prog )
{
  static const char *filename[] = { "app2h.comp.glsl" };
  static const GLchar *CPINames[] = { "CPoints" };
  static const GLchar *CPONames[] = { "CPointsOut" };
  static const GLchar *CompTrNames[] =  { "Tr", "Tr.tr" };
  static const GLchar *CompTrIndNames[] = { "TrInd", "TrInd.ind" };
  static const GLchar DimName[] = "dim";
  static const GLchar NcpName[] = "ncp";
  static const GLchar TrNumName[] = "trnum";
  GLuint shader_id;
  GLint  i;

  shader_id = CompileShaderFiles ( GL_COMPUTE_SHADER, 1, &filename[0] );
  prog->progid = LinkShaderProgram ( 1, &shader_id, "articulate" );
  prog->cpibp = prog->cpobp = prog->ctrbp = prog->ctribp = GL_INVALID_INDEX;
  GetAccessToStorageBlock ( prog->progid, 0, &CPINames[0], &i, &i, &prog->cpibp );
  GetAccessToStorageBlock ( prog->progid, 0, &CPONames[0], &i, &i, &prog->cpobp );
  GetAccessToStorageBlock ( prog->progid, 1, &CompTrNames[0],
                            &i, &i, &prog->ctrbp );
  GetAccessToStorageBlock ( prog->progid, 1, &CompTrIndNames[0],
                            &i, &i, &prog->ctribp );
  prog->dim_loc = glGetUniformLocation ( prog->progid, DimName );
  prog->ncp_loc = glGetUniformLocation ( prog->progid, NcpName );
  prog->trnum_loc = glGetUniformLocation ( prog->progid, TrNumName );
  glDeleteShader ( shader_id );
  ExitIfGLError ( "LoadLinkageArticulationProgram" );
} /*LoadLinkageArticulationProgram*/

void DeleteLinkageArticulationProgram ( KLArticulationProgram *prog )
{
  glUseProgram ( 0 );
  glDeleteProgram ( prog->progid );
} /*DeleteLinkageArticulationProgram*/

static char KLInitPalmMesh ( kl_linkage *lkg, kl_object *obj )
{
  static const GLfloat edges_colour[3] = {0.0,0.5,0.7};
  static const GLfloat facets_colour[3] = {0.91,0.65,0.5};
  static GLint r0[144] =
   {0,1,2,3,6,7,8,9,10,13,16,17,135,138,139,140,141,142,143,
     4,11,134,136,137, 5,12,14,15,18,19,20,21, 22,23,24,25,26,27,28,29,
    30,31,32,33,34,35,36,37, 38,39,40,41,42,43,44,45, 46,47,48,49,50,51,52,53,
    54,55,56,57,58,59,60,61, 62,63,64,65,66,67,68,69, 70,71,72,73,74,75,76,77,
    78,79,80,81,82,83,84,85, 86,87,88,89,90,91,92,93, 94,95,96,97,98,99,100,101,
    102,103,104,105,106,107,108,109, 110,111,112,113,114,115,116,117,
    118,119,120,121,122,123,124,125, 126,127,128,129,130,131,132,133};
  static const int r1[17] = {19,5,8,8,8,8,8,8, 8, 8, 8, 8, 8, 8, 8, 8, 8};
  static const int r2[17] = { 0,1,2,3,4,6,7,8,10,11,12,14,15,16,18,19,20};
  static const GLfloat diffr[4] = { 0.91, 0.65, 0.5, 1.0 };
  static const GLfloat specr[4] = { 0.15, 0.1, 0.12, 1.0 };
  static const GLfloat shn = 60.0, wa = 5.0, we = 5.0;
  AppData *ad;
  KLMesh  *md;
  GPUmesh **palms;
  GLuint  *cpi;
  int     on, rn, nv, i, j, k;

  ad = (AppData*)lkg->usrdata;
  on = obj - lkg->obj;
  nv = obj->nvert;
  if ( (cpi = malloc ( nv*sizeof(GLuint) )) ) {
    memset ( cpi, 0, nv*sizeof(GLuint) );
    md = (KLMesh*)obj->usrdata;
    palms = md->mesh;
    palms[0] = EnterPalmToGPU ();
    for ( j = k = 0;  j < 17;  k += r1[j++] ) {
      rn = kl_NewObjRef ( lkg, r2[j], on, 0, NULL );
      for ( i = 0; i < r1[j]; i++ )
        cpi[r0[k+i]] = rn;
    }
    md->tribuf = NewStorageBuffer ( nv*sizeof(GLuint), ad->artprog.ctribp );
    glBufferSubData ( GL_SHADER_STORAGE_BUFFER, 0, nv*sizeof(GLuint), cpi );
    free ( cpi );
    for ( i = 1; i <= NPALMMESHES+1; i++ ) {
      if ( !(palms[i] = malloc ( sizeof(GPUmesh) )) )
        ExitOnError ( "KLInitPalmMesh" );
      memset ( palms[i], 0, sizeof(GPUmesh) );
    }
    memcpy ( palms[1], palms[0], sizeof(GPUmesh) );
    glGenBuffers ( 1, &palms[1]->mbuf[2] );
    glBindBuffer ( GL_SHADER_STORAGE_BUFFER, palms[1]->mbuf[2] );
    glBufferData ( GL_SHADER_STORAGE_BUFFER, PALM_NV*3*sizeof(GLfloat),
                   NULL, GL_DYNAMIC_DRAW );
    md->mtn = SetupMaterial ( &ad->mat, -1, diffr, specr, shn, wa, we );
    memcpy ( md->ecolour, edges_colour, 3*sizeof(GLfloat) );
    memcpy ( md->fcolour, facets_colour, 3*sizeof(GLfloat) );
  }
  else
    ExitOnError ( "KLInitPalmMesh" );
  return true;
} /*KLInitPalmMesh*/

static void KLDeletePalmMesh ( kl_linkage *lkg, kl_object *obj )
{
  KLMesh  *md;
  int     i;

  md = (KLMesh*)obj->usrdata;
  md->mesh[1]->mbuf[0] = md->mesh[1]->mbuf[1] = md->mesh[1]->mbuf[3] = 0;
  for ( i = 0; i <= NPALMMESHES+1; i++ )
    DeleteGPUmesh ( md->mesh[i] );
  glDeleteBuffers ( 1, &md->tribuf );
} /*KLDeletePalmMesh*/

static void KLTransformVertices ( kl_linkage *lkg, kl_object *obj,
                                  int refn, GLfloat *tr, int nv, int *vn )
{
  AppData *ad;

  ad = (AppData*)lkg->usrdata;
  glBindBuffer ( GL_UNIFORM_BUFFER, ad->lktrbuf );
  glBufferSubData ( GL_UNIFORM_BUFFER, refn*16*sizeof(GLfloat),
                    16*sizeof(GLfloat), tr );
  ExitIfGLError ( "KLTransformVertices" );
} /*KLTransformVertices*/

static void KLPostprocessMesh ( kl_linkage *lkg, kl_object *obj )
{
  AppData *ad;
  KLMesh  *md;
  KLArticulationProgram *prog;
  GPUmesh **mesh;
  int     i;

  ad = (AppData*)lkg->usrdata;
  prog = &ad->artprog;
  md = (KLMesh*)obj->usrdata;
  mesh = md->mesh;
  glUseProgram ( prog->progid );
  glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, prog->ctrbp, ad->lktrbuf );
  glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, prog->ctribp, md->tribuf );
  glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, prog->cpibp, mesh[0]->mbuf[2] );
  glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, prog->cpobp, mesh[1]->mbuf[2] );
  glUniform1i ( prog->dim_loc, obj->nvc );
  glUniform1i ( prog->trnum_loc, -1 );
  glUniform1i ( prog->ncp_loc, (GLint)obj->nvert );
  glDispatchCompute ( obj->nvert, 1, 1 );
  glMemoryBarrier ( GL_UNIFORM_BARRIER_BIT );
  if ( ad->lod >= 1 ) {
    for ( i = 1; i <= ad->lod; i++ ) {
      if ( !GPUmeshRefinement ( MESHDEG, mesh[i], mesh[i+1] ) )
        ExitOnError ( "KLPostprocessMesh" );
/*      printf ( "%d: nv = %d, nhe = %d, nfac = %d\n",
               i, mesh[i+1]->nv, mesh[i+1]->nhe, mesh[i+1]->nfac );*/
    }
    ComputeMeshNormalVectors ( mesh[ad->lod+1], 6, 3 );
  }
  ExitIfGLError ( "KLPostprocessMesh" );
} /*KLPostprocessMesh*/

static void KLRedrawMesh ( kl_linkage *lkg, kl_object *obj )
{
  AppData *ad;
  KLMesh  *md;

  ad = (AppData*)lkg->usrdata;
  md = (KLMesh*)obj->usrdata;
  if ( ad->wdg.sw[0] )
    DrawMeshEdges ( &ad->mrprog, md->mesh[1], md->ecolour );
  if ( ad->lod >= 1 ) {
    if ( ad->edges )
      DrawMeshEdges ( &ad->mrprog, md->mesh[ad->lod+1], md->fcolour );
    else
      DrawMeshFacets ( &ad->mrprog, md->mesh[ad->lod+1],
                       &ad->mat, md->mtn, ad->mnv, ad->final );
  }
} /*KLRedrawMesh*/

static char KLInitFingernails ( kl_linkage *lkg, kl_object *obj )
{
  static GLfloat cnetcolour[3] = { 0.0, 1.0, 0.0 };
  AppData        *ad;
  KLBezPatches   *pd;
  int            i, j, l, m, on, rn;
  GLuint         *cpi;

  ad = (AppData*)lkg->usrdata;
  on = obj - lkg->obj;
  pd = (KLBezPatches*)obj->usrdata;
  if ( (cpi = malloc (FINGERNAIL_NV*sizeof(GLuint))) ) {
    glGenBuffers ( 1, &pd->tribuf );
    glBindBuffer ( GL_SHADER_STORAGE_BUFFER, pd->tribuf );
    for ( l = 0; l < FINGER_NUM; l++ ) {  /* kolejno dla kazdego palca */
      rn = kl_NewObjRef ( lkg, 4*(l+1), on, 0, NULL );
      for ( i = 0;  i <= FINGERNAIL_UDEG;  i++ ) {
        m = i == 0 || i == FINGERNAIL_UDEG ? FINGERNAIL_VDEG : 2;
        for ( j = 0;  j <= m;  j++ )
          cpi[l*FINGERNAIL_NCP + (FINGERNAIL_VDEG+1)*i + j] = rn;
      }
      rn = kl_NewObjRef ( lkg, 21+l, on, 0, NULL );
      for ( i = 1;  i < FINGERNAIL_UDEG;  i++ )
        for ( j = 3;  j <= FINGERNAIL_VDEG;  j++ )
          cpi[l*FINGERNAIL_NCP + (FINGERNAIL_VDEG+1)*i + j] = rn;
    }
    glBufferData ( GL_SHADER_STORAGE_BUFFER, FINGERNAIL_NV*sizeof(GLuint),
                   cpi, GL_STATIC_DRAW );
    free ( cpi );
    EnterFingernailsToGPU ( pd->bpatches, cnetcolour );
    pd->mtn = SetupMaterial ( &ad->mat, -1, cnetcolour, cnetcolour, 1.0, 1.0, 1.0 );
  }
  else
    ExitOnError ( "KLInitFingernails" );
  return true;
} /*KLInitFingernails*/

static void KLDeleteFingernails ( kl_linkage *lkg, kl_object *obj )
{
  KLBezPatches *pd;

  pd = (KLBezPatches*)obj->usrdata;
  DeleteBezierPatches ( pd->bpatches[0] );
  glDeleteBuffers ( 1, &pd->bpatches[1]->buf[1] );
  free ( pd->bpatches[1] );
  glDeleteBuffers ( 1, &pd->tribuf );
} /*KLDeleteFingernails*/

static void V4Interpolatef ( GLfloat v[4],
                             const GLfloat v0[4], const GLfloat v1[4], float t )
{
  int   i;
  float s;

  for ( i = 0, s = 1.0-t;  i < 3;  i++ )
    v[i] = s*v0[i] + t*v1[i];
} /*V4Interpolatef*/

static void KLPostprocessFingernails ( kl_linkage *lkg, kl_object *obj )
{
  const GLfloat diffr0[4] = { 0.91, 0.65, 0.5, 1.0 };
  const GLfloat specr0[4] = { 0.15, 0.1, 0.12, 1.0 };
  const GLfloat diffr1[4] = { 0.7, 0.1, 0.25, 1.0 };
  const GLfloat specr1[4] = { 0.4, 0.4, 0.4, 1.0 };
  const GLfloat shn = 60.0, wa = 5.0, we = 5.0;
  AppData               *ad;
  KLArticulationProgram *prog;
  KLBezPatches          *pd;
  BezierPatchObjf       **nails;
  GLfloat               diffr[4], specr[4];
  float                 t;

  ad = (AppData*)lkg->usrdata;
  prog = &ad->artprog;
  pd = (KLBezPatches*)obj->usrdata;
  nails = pd->bpatches;
  glUseProgram ( ad->artprog.progid );
  glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, prog->ctrbp, ad->lktrbuf );
  glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, prog->ctribp, pd->tribuf );
  glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, prog->cpibp, nails[0]->buf[1] );
  glBindBufferBase ( GL_SHADER_STORAGE_BUFFER, prog->cpobp, nails[1]->buf[1] );
  glUniform1i ( prog->dim_loc, obj->nvc );
  glUniform1i ( prog->trnum_loc, -1 );
  glUniform1i ( prog->ncp_loc, (GLint)obj->nvert );
  glDispatchCompute ( obj->nvert, 1, 1 );
  t = ad->wdg.artp[NKLARTPARAMS-1];
  t = t > 0.2 ? 1 : 5.0*t;
  V4Interpolatef ( diffr, diffr0, diffr1, t );
  V4Interpolatef ( specr, specr0, specr1, t );
  SetupMaterial ( &ad->mat, pd->mtn, diffr, specr, shn, wa, we );
  glMemoryBarrier ( GL_UNIFORM_BARRIER_BIT );
  ExitIfGLError ( "KLPostprocessFingernails" );
} /*KLPostprocessFingernails*/

static void KLRedrawBezPatches ( kl_linkage *lkg, kl_object *obj )
{
  static const int TessLevel[4] = {4,8,16,32};
  AppData      *ad;
  KLBezPatches *bezp;

  ad = (AppData*)lkg->usrdata;
  bezp = (KLBezPatches*)obj->usrdata;
  glPolygonMode ( GL_FRONT_AND_BACK, GL_FILL );
  SetBezierPatchTessLevel ( bezp->bpatches[1], TessLevel[ad->lod-1] );
  SetBezierPatchNVS ( bezp->bpatches[1], (GLint)ad->mnv );
  if ( ad->edges )
    glPolygonMode ( GL_FRONT_AND_BACK, GL_LINE );
  else
    glPolygonMode ( GL_FRONT_AND_BACK, GL_FILL );
  if ( ad->final ) {
    glUseProgram ( ad->brprog.progid[1] );
    ChooseMaterial ( &ad->mat, bezp->mtn );
  }
  else
    glUseProgram ( ad->brprog.progid[2] );
  DrawBezierPatches ( bezp->bpatches[1] );
  if ( ad->wdg.sw[0] ) {
    glUseProgram ( ad->brprog.progid[0] );
    DrawBezierNets ( bezp->bpatches[1] );
  }
  ExitIfGLError ( "KLRedrawBezPatches" );
} /*KLRedrawBezPatches*/

kl_linkage *ConstructPalmLinkage ( AppData *ad )
{
  static int jtype[NKLJOINTS] =
    { KL_ART_ROT_Y, KL_ART_ROT_Z, KL_ART_ROT_X, KL_ART_ROT_X,
      KL_ART_ROT_Z, KL_ART_ROT_X, KL_ART_ROT_X, KL_ART_ROT_X,
      KL_ART_ROT_Z, KL_ART_ROT_X, KL_ART_ROT_X, KL_ART_ROT_X,
      KL_ART_ROT_Z, KL_ART_ROT_X, KL_ART_ROT_X, KL_ART_ROT_X,
      KL_ART_ROT_Z, KL_ART_ROT_X, KL_ART_ROT_X, KL_ART_ROT_X,
      KL_ART_TRANS_Y, KL_ART_TRANS_Y, KL_ART_TRANS_Y, KL_ART_TRANS_Y,
      KL_ART_TRANS_Y };
  static int jpnum[NKLJOINTS] =
    {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,20,20,20,20};
  static float pp0[3] = {-0.2, -0.86, 0.5},
               vp0[3] = {0.0,1.0,0.0}, vp1[3] = {0.22,0.83,0.1};
  kl_linkage *lkg;
  int        lnk[NKLINKS], jnt[NKLJOINTS];
  GLfloat    tra[16];
  int        i, j, k, l;

  if ( (lkg = kl_NewLinkage ( NKLOBJ, NKLINKS, NKLREFS, NKLJOINTS,
                              NKLARTPARAMS, (void*)ad )) ) {
    ad->linkage = lkg;
    memcpy ( ad->wdg.artp, palmartp0, NKLARTPARAMS*sizeof(float) );
    for ( i = 0; i < NKLINKS; i++ )
      lnk[i] = kl_NewLink ( lkg );

    kl_NewObject ( lkg, 0, 3, PALM_NV, NULL, (void*)&ad->palm,
                   KLInitPalmMesh, KLTransformVertices,
                   KLPostprocessMesh, KLRedrawMesh, KLDeletePalmMesh );
    kl_NewObject ( lkg, 0, 3, FINGERNAIL_NV, NULL, (void*)&ad->nails,
                   KLInitFingernails, KLTransformVertices, KLPostprocessFingernails,
                   KLRedrawBezPatches, KLDeleteFingernails );
    for ( k = j = 0;  k < 5;  k++ ) {
      for ( i = 0, l = -1;  i < 4;  i++, l = j++ )
        jnt[j] = kl_NewJoint ( lkg, lnk[l+1], lnk[j+1], jtype[j], jpnum[j] );
    }
    for ( k = 0;  k < 5;  k++, j++ )
      jnt[j] = kl_NewJoint ( lkg, lnk[4*(k+1)], lnk[j+1], jtype[j], jpnum[j] );
        /* kciuk */
    M4x4RotateP2Vf ( tra, pp0, vp0, vp1 );
    kl_SetJointFtr ( lkg, jnt[0], tra, true );
    M4x4Translatef ( tra, 0.125, -0.842, 0.0 );
    kl_SetJointFtr ( lkg, jnt[1], tra, true );
    M4x4Translatef ( tra, 0.329, -0.275, 0.05 );
    M4x4MRotateZf ( tra, -0.1*PI );
    M4x4MRotateYf ( tra, -0.2*PI );
    kl_SetJointFtr ( lkg, jnt[2], tra, true );
    M4x4Translatef ( tra, 0.4, -0.05, 0.05 );
    M4x4MRotateZf ( tra, -0.1*PI );
    M4x4MRotateYf( tra, -0.2*PI );
    kl_SetJointFtr ( lkg, jnt[3], tra, true );
    M4x4RotateZf ( tra, -0.1*PI );
    kl_SetJointFtr ( lkg, jnt[20], tra, true );
        /* wskazujacy */
    M4x4Translatef ( tra, 0.070, 0.05, 0.08 );
    kl_SetJointFtr ( lkg, jnt[4], tra, true );
    kl_SetJointFtr ( lkg, jnt[5], tra, true );
    M4x4Translatef ( tra, 0.094, 0.324, 0.08 );
    kl_SetJointFtr ( lkg, jnt[6], tra, true );
    M4x4Translatef ( tra, 0.1, 0.591, 0.08 );
    kl_SetJointFtr ( lkg, jnt[7], tra, true );
        /* srodkowy */
    M4x4Translatef ( tra, -0.13, 0.08, 0.08 );
    kl_SetJointFtr ( lkg, jnt[8], tra, true );
    kl_SetJointFtr ( lkg, jnt[9], tra, true );
    M4x4Translatef ( tra, -0.13, 0.349, 0.08 );
    kl_SetJointFtr ( lkg, jnt[10], tra, true );
    M4x4Translatef ( tra, -0.13, 0.634, 0.08 );
    kl_SetJointFtr ( lkg, jnt[11], tra, true );
        /* serdeczny */
    M4x4Translatef ( tra, -0.359, 0.05, 0.08 );
    kl_SetJointFtr ( lkg, jnt[12], tra, true );
    kl_SetJointFtr ( lkg, jnt[13], tra, true );
    M4x4Translatef ( tra, -0.374, 0.324, 0.08 );
    kl_SetJointFtr ( lkg, jnt[14], tra, true );
    M4x4Translatef ( tra, -0.392, 0.585, 0.08 );
    kl_SetJointFtr ( lkg, jnt[15], tra, true );
        /* maly */
    M4x4Translatef ( tra, -0.563, 0.0, 0.08 );
    kl_SetJointFtr ( lkg, jnt[16], tra, true );
    kl_SetJointFtr ( lkg, jnt[17], tra, true );
    M4x4Translatef ( tra, -0.59, 0.232, 0.08 );
    kl_SetJointFtr ( lkg, jnt[18], tra, true );
    M4x4Translatef ( tra, -0.605, 0.41, 0.08 );
    kl_SetJointFtr ( lkg, jnt[19], tra, true );
    glGenBuffers ( 1, &ad->lktrbuf );
    glBindBuffer ( GL_SHADER_STORAGE_BUFFER, ad->lktrbuf );
    glBufferData ( GL_SHADER_STORAGE_BUFFER, lkg->norefs*16*sizeof(GLfloat),
                   NULL, GL_DYNAMIC_DRAW );
  }
  return lkg;
} /*ConstructPalmLinkage*/

void SetArticulationParameter ( AppData *ad, int pnum )
{
  double x, par;

  x = ad->wdg.artp[pnum];
  par = (1.0-x)*palmartprange[pnum][0] + x*palmartprange[pnum][1];
  kl_SetArtParam ( ad->linkage, pnum, 1, &par );
} /*SetArticulationParameter*/

void ArticulatePalmLinkage ( AppData *ad )
{
  double x, par[NKLARTPARAMS];
  int    i;

  for ( i = 0; i < NKLARTPARAMS; i++ ) {
    x = ad->wdg.artp[i];
    par[i] = (1.0-x)*palmartprange[i][0] + x*palmartprange[i][1];
  }
  kl_SetArtParam ( ad->linkage, 0, NKLARTPARAMS, par );
  kl_Articulate ( ad->linkage );
} /*ArticulatePalmLinkage*/

