#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include "../utilities/openglheader.h"

#include "../utilities/utilities.h"
#include "accumbuf.h"
#include "../utilities/linkage.h"
#include "../utilities/bezpatches.h"
#include "trans.h"
#include "lights.h"
#include "app2k.h"
#include "app2kstruct.h"

#define UNB GL_UNIFORM_BUFFER
#define SSB GL_SHADER_STORAGE_BUFFER

void LoadParticleShaders ( PartSysPrograms *psprog )
{
  static const char *filename[] =
    { "app2i0.comp.glsl",
      "app2k0.vert.glsl", "app2k0.geom.glsl", "app2k0.frag.glsl" };
  static const GLchar *SBNames[] =
    { "RandomVecPool", "PartPos", "PartVel" };
  static const GLchar *UPNames[] =
    { "PartSys", "PartSys.tablength", "PartSys.iter", "PartSys.dt",
      "PartSys.Tpart", "PartSys.vel", "PartSys.TC", "PartSys.acc",
      "PartSys.divv", "PartSys.divp", "PartSys.pos0", "PartSys.pos1",
      "PartSys.vel0", "PartSys.vel1", "PartSys.conv" };
  static const GLchar PCName[] = "PartColour";
  GLuint shader_id[4];
  GLint  size;
  int    i;

  shader_id[0] = CompileShaderFiles ( GL_COMPUTE_SHADER, 1, &filename[0] );
  psprog->program_id[0] = LinkShaderProgram ( 1, &shader_id[0], "part0" );
  shader_id[1] = CompileShaderFiles ( GL_VERTEX_SHADER, 1, &filename[1] );
  shader_id[2] = CompileShaderFiles ( GL_GEOMETRY_SHADER, 1, &filename[2]);
  shader_id[3] = CompileShaderFiles ( GL_FRAGMENT_SHADER, 1, &filename[3]);
  psprog->program_id[1] = LinkShaderProgram ( 3, &shader_id[1], "part1" );
  GetAccessToUniformBlock ( psprog->program_id[0], 14, &UPNames[0],
                            &psprog->psbsize, psprog->psofs, &psprog->psbp );
  psprog->rvpbp = psprog->posbp = psprog->velbp = GL_INVALID_INDEX;
  GetAccessToStorageBlock ( psprog->program_id[0], 0, &SBNames[0],
                            &size, NULL, &psprog->rvpbp );
  GetAccessToStorageBlock ( psprog->program_id[0], 0, &SBNames[1],
                            &size, NULL, &psprog->posbp );
  GetAccessToStorageBlock ( psprog->program_id[0], 0, &SBNames[2],
                            &size, NULL, &psprog->velbp );
  psprog->pcloc = glGetUniformLocation ( psprog->program_id[1], PCName );
  AttachUniformTransBlockToBP ( psprog->program_id[1] );
  for ( i = 0; i < 3; i++ )
    glDeleteShader ( shader_id[i] );
  ExitIfGLError ( "LoadParticleShaders" );
} /*LoadParticleShaders*/

GLfloat RandomFloat ( void )
{
  float res;
  unsigned int tmp;
  static unsigned int seed = 0xFFFF0C59;

  seed *= 16807;
  tmp = seed ^ (seed >> 4) ^ (seed << 15);
  *((unsigned int *) &res) = (tmp >> 9) | 0x3F800000;
  return res - 1.0;
} /*RandomFloat*/

static int cnt = 0;

void RandomVector ( GLfloat rv[4] )
{
  int i;

  for ( ; ; cnt++ ) {
    for ( i = 0; i < 3; i++ )
      rv[i] = 2.0*RandomFloat() - 1.0;
    if ( V3DotProductf ( rv, rv ) <= 1.0 ) {
      rv[3] = 0.0;
      return;
    }
  }
} /*RandomVector*/

void InitParticleSystem ( PartSysPrograms *psprog, PartSystem *ps )
{
  GLfloat *mbuf;
  int     i;

  glGenVertexArrays ( 1, &ps->vao );
  glGenBuffers ( 3, ps->sbo );
  if ( !(mbuf = (GLfloat*)malloc ( PARTICLE_COUNT*4*sizeof(GLfloat) )) )
    ExitOnError ( "InitParticles" );
  glBindBuffer ( SSB, ps->sbo[0] );
  for ( i = 0; i < PARTICLE_COUNT; i++ ) {
    RandomVector ( &mbuf[4*i] );
    mbuf[4*i+3] = (float)i/(float)PARTICLE_COUNT - 1.0;
  }
printf ( "cnt: %d %d\n", PARTICLE_COUNT, cnt );
  glBufferData ( SSB, PARTICLE_COUNT*4*sizeof(GLfloat), mbuf, GL_STATIC_DRAW );
  glBindVertexArray ( ps->vao );
  glBindBuffer ( GL_ARRAY_BUFFER, ps->sbo[1] );
  glBufferData ( GL_ARRAY_BUFFER, PARTICLE_COUNT*4*sizeof(GLfloat),
                 mbuf, GL_DYNAMIC_COPY );
  glVertexAttribPointer ( 0, 4, GL_FLOAT, GL_FALSE, 0, NULL );
  glEnableVertexAttribArray ( 0 );
  glBindVertexArray ( 0 );
  glBindBuffer ( SSB, ps->sbo[2] );
  glBufferData ( SSB, PARTICLE_COUNT*4*sizeof(GLfloat),
                 NULL, GL_DYNAMIC_COPY );
  free ( mbuf );
  glGenBuffers ( 1, &ps->ubo );
  glBindBufferBase ( UNB, psprog->psbp, ps->ubo );
  glBufferData ( UNB, psprog->psbsize, NULL, GL_DYNAMIC_DRAW );
  ps->tablength = PARTICLE_COUNT;
  glBufferSubData ( UNB, psprog->psofs[0], sizeof(GLint), &ps->tablength );
  ps->iter = 0;
  glBufferSubData ( UNB, psprog->psofs[1], sizeof(GLint), &ps->iter );
  ps->dt = 0.0;
  glBufferSubData ( UNB, psprog->psofs[2], sizeof(GLfloat), &ps->dt );
  ps->Tpart = 1.5;
  glBufferSubData ( UNB, psprog->psofs[3], sizeof(GLfloat), &ps->Tpart );
  ps->TC = 0.4;
  glBufferSubData ( UNB, psprog->psofs[5], sizeof(GLfloat), &ps->TC );
  ps->acc = 0.6;
  glBufferSubData ( UNB, psprog->psofs[6], sizeof(GLfloat), &ps->acc );
  ps->divv = 0.2;
  glBufferSubData ( UNB, psprog->psofs[7], sizeof(GLfloat), &ps->divv );
  ps->divp = 0.04;
  glBufferSubData ( UNB, psprog->psofs[8], sizeof(GLfloat), &ps->divp );
  ps->conv[0] = ps->conv[1] = 0.0;  ps->conv[2] = 0.5;
  glBufferSubData ( UNB, psprog->psofs[13], 3*sizeof(GLfloat), &ps->conv );
  ExitIfGLError ( "InitPartSystem" );
} /*InitPartSystem*/

void MoveParticles ( PartSysPrograms *psprog, PartSystem *ps, double time )
{
  GLfloat deltat;

  if ( (deltat = time-ps->last_time) < 0.01 )
    return;
  ps->last_time = time;
  if ( deltat > 1.0 )
    deltat = 1.0;
  glUseProgram ( psprog->program_id[0] );
  glBindBufferBase ( SSB, psprog->rvpbp, ps->sbo[0] );
  glBindBufferBase ( SSB, psprog->posbp, ps->sbo[1] );
  glBindBufferBase ( SSB, psprog->velbp, ps->sbo[2] );
  time -= ps->time0;
  glBindBufferBase ( UNB, psprog->psbp, ps->ubo );
  ps->iter = (ps->iter+1) % PARTICLE_COUNT;
  glBufferSubData ( UNB, psprog->psofs[1], sizeof(GLint), &ps->iter );
  ps->dt = deltat;
  glBufferSubData ( UNB, psprog->psofs[2], sizeof(GLfloat), &ps->dt );
  ps->vel = 1.0+0.1*sin ( 18.0*PI*time*(1.0+sin(time)) );
  glBufferSubData ( UNB, psprog->psofs[4], sizeof(GLfloat), &ps->vel );
  glBufferSubData ( UNB, psprog->psofs[9], 3*sizeof(GLfloat), ps->pos0 );
  glBufferSubData ( UNB, psprog->psofs[10], 3*sizeof(GLfloat), ps->pos1 );
  glBufferSubData ( UNB, psprog->psofs[11], 3*sizeof(GLfloat), ps->vel0 );
  glBufferSubData ( UNB, psprog->psofs[12], 3*sizeof(GLfloat), ps->vel1 );
  glDispatchCompute ( PARTICLE_COUNT, 1, 1 );
  glMemoryBarrier ( GL_SHADER_IMAGE_ACCESS_BARRIER_BIT );
  ExitIfGLError ( "MoveParticles" );
} /*MoveParticles*/

void DrawParticles ( PartSysPrograms *psprog, PartSystem *ps,
                     Camera *camera, char final )
{
  GLfloat c[4];

  glDepthMask ( GL_FALSE );
  glUseProgram ( psprog->program_id[1] );
  if ( !final ) {
    c[0] = c[1] = c[2] = 0.7;
    c[3] = (float)(SHADOW_MAP_SIZE*SHADOW_MAP_SIZE);
  }
  else {
    c[0] = c[1] = c[2] = 0.85;
    c[3] = (float)(camera->win_height*camera->win_height);
  }
  c[3] = exp ( -0.001*c[3]/(float)PARTICLE_COUNT );
  glUniform4fv ( psprog->pcloc, 1, c );
  glBindVertexArray ( ps->vao );
  glEnable ( GL_DEPTH_CLAMP );
  glEnable ( GL_BLEND );
  glBlendFunc ( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
  glPointSize ( 1.0 );
  glDrawArrays ( GL_POINTS, 0, PARTICLE_COUNT );
  glDisable ( GL_BLEND );
  glDisable ( GL_DEPTH_CLAMP );
  glDepthMask ( GL_TRUE );
  glBindVertexArray ( 0 );
  ExitIfGLError ( "DrawParticles" );
} /*DrawParticles*/

void ResetParticles ( PartSysPrograms *psprog, PartSystem *ps, double time )
{
  ps->time0 = ps->last_time = time;
  glBindBuffer ( GL_COPY_READ_BUFFER, ps->sbo[0] );
  glBindBuffer ( GL_COPY_WRITE_BUFFER, ps->sbo[1] );
  glCopyBufferSubData ( GL_COPY_READ_BUFFER, GL_COPY_WRITE_BUFFER,
                        0, 0, PARTICLE_COUNT*4*sizeof(GLfloat) );
  glBindBufferBase ( UNB, psprog->psbp, ps->ubo );
  ps->iter = 0;
  glBufferSubData ( UNB, psprog->psofs[1], sizeof(GLint), &ps->iter );
  ExitIfGLError ( "ResetParticles" );
} /*ResetParticles*/

void DeleteParticleSystem ( PartSystem *ps )
{
  glDeleteVertexArrays ( 1, &ps->vao );
  glDeleteBuffers ( 3, ps->sbo );
  glDeleteBuffers ( 1, &ps->ubo );
} /*DeleteParticleSystem*/

void DeleteParticleShaders ( PartSysPrograms *psprog )
{
  int i;

  glUseProgram ( 0 );
  for ( i = 0; i < 2; i++ )
    glDeleteProgram ( psprog->program_id[i] );
} /*DeleteParticleShaders*/

