#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "../utilities/openglheader.h"

#include "app2d.h"
#include "../utilities/utilities.h"
#include "../utilities/bezpatches.h"
#include "../utilities/teapot.h"
#include "../utilities/sproduct.h"
#include "lights.h"
#include "trans.h"
#include "app2dstruct.h"
#include "app2dproc.h"

static AppData appdata;

void LoadBPShaders ( BPRenderPrograms *brprog )
{
  static const char *filename[] =
    { "app2.vert.glsl", "app2.tesc.glsl", "app2d.tese.glsl",
      "app2d.geom.glsl", "app2d.frag.glsl",
      "app2a1.vert.glsl", "app2a1.frag.glsl" };
  static const GLuint shtype[7] =
    { GL_VERTEX_SHADER, GL_TESS_CONTROL_SHADER, GL_TESS_EVALUATION_SHADER,
      GL_GEOMETRY_SHADER, GL_FRAGMENT_SHADER,
      GL_VERTEX_SHADER, GL_FRAGMENT_SHADER };
  static const GLchar *UVarNames[] = { "ColourSource", "LightingModel" };
  GLuint shader_id[7];
  int    i;

  for ( i = 0; i < 7; i++ )
    shader_id[i] = CompileShaderFiles ( shtype[i], 1, &filename[i] );
  brprog->program_id[0] = LinkShaderProgram ( 5, shader_id, "0" );
  brprog->program_id[1] = LinkShaderProgram ( 2, &shader_id[5], "1" );
  GetAccessToTransBlockUniform ( brprog->program_id[0] );
  GetAccessToLightMatUniformBlocks ( brprog->program_id[0] );
  GetAccessToBezPatchStorageBlocks ( brprog->program_id[0], true, false );
  brprog->ColourSourceLoc =
      glGetUniformLocation ( brprog->program_id[0], UVarNames[0] );
  brprog->LightingModelLoc =
      glGetUniformLocation ( brprog->program_id[0], UVarNames[1] );
  AttachUniformTransBlockToBP ( brprog->program_id[1] );
  for ( i = 0; i < 7; i++ )
    glDeleteShader ( shader_id[i] );
  ExitIfGLError ( "LoadBPShaders" );
} /*LoadBPShaders*/

void SetupTeapotMatrix ( AppData *ad )
{
#define SCF (1.0/3.0)
  M4x4Translatef ( ad->teapot_mmatrix, 0.0, 0.0, -0.6 );
  M4x4MScalef ( ad->teapot_mmatrix, SCF, SCF, SCF*4.0/3.0 );
  M4x4MRotateVfv ( ad->teapot_mmatrix, ad->model_rot_axis, ad->teapot_rot_angle );
#undef SCF
} /*SetupTeapotMatrix*/

void SetupTorusMatrix ( AppData *ad )
{
  GLfloat p[4] = {3.1,0.0,2.9,1.0}, q[4];

  M4x4RotateZf ( ad->torus_mmatrix, ad->torus_rot_angle );
  M4x4MRotateXf ( ad->torus_mmatrix, 0.5*PI );
  M4x4MScalef ( ad->torus_mmatrix, 0.1, 0.1, 0.1 );
  M4x4MultMVf ( q, ad->teapot_mmatrix, p );
  M4x4TranslateMfv ( ad->torus_mmatrix, q );
} /*SetupTorusMatrix*/

void _RotateViewer ( AppData *ad, double delta_xi, double delta_eta )
{
  float   vi[3], lgt, vk[3];
  double  angi, angk;

  vi[0] = (float)delta_eta*ad->camera.tb/(float)ad->camera.win_height;
  vi[1] = (float)delta_xi*ad->camera.rl/(float)ad->camera.win_width;
  vi[2] = 0.0;
  lgt = sqrt ( V3DotProductf ( vi, vi ) );
  vi[0] /= lgt;  vi[1] /= lgt;
  angi = -0.052359878;  /* -3 stopnie */
  V3CompRotationsf ( vk, &angk, ad->camera.viewer_rvec,
                     ad->camera.viewer_rangle, vi, angi );
  memcpy ( ad->camera.viewer_rvec, vk, 3*sizeof(float) );
  ad->camera.viewer_rangle = angk;
  M4x4RotateVfv ( ad->trans.vm, ad->camera.viewer_rvec,
                  -ad->camera.viewer_rangle );
  M4x4MultMTVf ( ad->trans.eyepos, ad->trans.vm, ad->camera.viewer_pos0 );
  M4x4InvTranslateMfv ( ad->trans.vm, ad->camera.viewer_pos0 );
  LoadVPMatrix ( &ad->trans );
} /*_RotateViewer*/

void RotateViewer ( double delta_xi, double delta_eta )
{
  if ( delta_xi == 0.0 && delta_eta == 0.0 )
    return;  /* natychmiast uciekamy - nie chcemy dzielic przez 0 */
  _RotateViewer ( &appdata, delta_xi, delta_eta );
} /*RotateViewer*/

static GLuint LoadMyTextures ( void )
{
  GLuint tex;
  const char *fn[4] = { "../tiff/jaszczur.tif", "../tiff/salamandra.tif",
                        "../tiff/lis.tif", "../tiff/kwiatki.tif" };

  if ( (tex = CreateMyTexture ( 1024 )) ) {
    LoadMyTextureImage ( tex, 1024, 1024,   0,   0, fn[0] );
    LoadMyTextureImage ( tex, 1024, 1024, 512,   0, fn[1] );
    LoadMyTextureImage ( tex, 1024, 1024,   0, 512, fn[2] );
    LoadMyTextureImage ( tex, 1024, 1024, 512, 512, fn[3] );
    SetupMyTextureMipmaps ( tex );
  }
  return tex;
} /*LoadMyTextures*/

void InitLights ( AppData *ad )
{
  GLfloat amb0[3] = { 0.2, 0.2, 0.3 };
  GLfloat dir0[3] = { 0.8, 0.8, 0.8 };
  GLfloat pos0[4] = { 0.0, 1.0, 1.0, 0.0 };
  GLfloat atn0[3] = { 1.0, 0.0, 0.0 };

  SetLightAmbient ( &ad->light, 0, amb0 );
  SetLightDirect ( &ad->light, 0, dir0 );
  SetLightPosition ( &ad->light, 0, pos0 );
  SetLightAttenuation ( &ad->light, 0, atn0 );
  SetLightOnOff ( &ad->light, 0, 1 );
} /*InitLights*/

void ConstructMyTeapot ( AppData *ad )
{
  const GLfloat MyColour[4] = { 1.0, 1.0, 1.0, 1.0 };
  const GLfloat diffr[4] = { 0.75, 0.6, 0.2, 1.0 };
  const GLfloat specr[4] = { 0.7, 0.7, 0.6, 1.0 };
  const GLfloat shn = 60.0, wa = 5.0, we = 5.0;
  const GLfloat txc[32][4] =
    {{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.-1,0.0,0.0},
     {0.5,0.5,0.0,0.0},{0.5,1.0,0.0,0.5},{1.0,0.5,0.5,0.0},{1.0,1.0,0.5,0.5},
     {-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},
     {-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},
     {-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},
     {-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},
     {-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},
     {-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0},{-1.0,0.0,-1.0,0.0}};

  ad->myteapot = ConstructTheTeapot ( MyColour );
  SetBezierPatchTessLevel ( ad->myteapot, ad->TessLevel );
  SetBezierPatchNVS ( ad->myteapot, ad->BezNormals );
  SetupMaterial ( &ad->mat, -1, diffr, specr, shn, wa, we );
  if ( !GenBezierPatchTextureBlock ( ad->myteapot, &txc[0][0] ) )
    ExitOnError ( "ConstructMyTeapot" );
} /*ConstructMyTeapot*/

void ConstructMyTorus ( AppData *ad )
{
  GLfloat MyColour[4] = { 0.2, 0.3, 1.0, 1.0 };
  const GLfloat diffr[4] = { 0.0, 0.4, 1.0, 1.0 };
  const GLfloat specr[4] = { 0.7, 0.7, 0.7, 1.0 };
  const GLfloat shn = 20.0, wa = 2.0, we = 5.0;

  ad->mytorus = EnterTorus ( 1.0, 0.5, MyColour );
  SetBezierPatchTessLevel ( ad->mytorus, ad->TessLevel );
  SetBezierPatchNVS ( ad->mytorus, ad->BezNormals );
  SetupMaterial ( &ad->mat, -1, diffr, specr, shn, wa, we );
} /*ConstructMyTorus*/

static void _ResizeMyWorld ( AppData *ad, int width, int height )
{
  float lr;

  glViewport ( 0, 0, ad->camera.win_width = width,
               ad->camera.win_height = height );
  lr = 0.5533*(float)width/(float)height;
  M4x4Frustumf ( ad->trans.pm, NULL, ad->camera.left = -lr, ad->camera.right = lr,
                 ad->camera.bottom = -0.5533, ad->camera.top = 0.5533,
                 ad->camera.near = 5.0, ad->camera.far = 15.0 );
  ad->camera.rl = 2.0*lr;  ad->camera.tb = 2.0*0.5533;
  LoadVPMatrix ( &ad->trans );
} /*_ResizeMyWorld*/

void ResizeMyWorld ( int width, int height )
{
  _ResizeMyWorld ( &appdata, width, height );
} /*ResizeMyWorld*/

void InitCamera ( AppData *ad, int width, int height )
{
  static const float viewer_rvec[3] = {1.0,0.0,0.0};
  static const float viewer_pos0[4] = {0.0,0.0,10.0,1.0};

  memcpy ( ad->camera.viewer_rvec, viewer_rvec, 3*sizeof(float) );
  memcpy ( ad->camera.viewer_pos0, viewer_pos0, 4*sizeof(float) );
  memcpy ( ad->trans.eyepos, viewer_pos0, 4*sizeof(GLfloat) );
  ad->camera.viewer_rangle = 0.0;
  M4x4InvTranslatefv ( ad->trans.vm, viewer_pos0 );
  _ResizeMyWorld ( ad, width, height );
} /*InitCamera*/

void InitMyWorld ( int argc, char *argv[], int width, int height )
{
  float axis[4] = {0.0,0.0,1.0};

  memset ( &appdata, 0, sizeof(AppData) );
  LoadBPShaders ( &appdata.brprog );
  appdata.brprog.colour_source = 0;
  appdata.trans.trbuf = NewUniformTransBlock ();
  appdata.light.lsbuf = NewUniformLightBlock ();
  appdata.mat.matbuf = NewUniformMatBlock ();
  TimerInit ();
  memcpy ( appdata.model_rot_axis, axis, 4*sizeof(float) );
  appdata.teapot_rot_angle = appdata.torus_rot_angle = 0.0;
  SetupTeapotMatrix ( &appdata );
  SetupTorusMatrix ( &appdata );
  InitCamera ( &appdata, width, height );
  appdata.TessLevel = 10;
  appdata.BezNormals = GL_TRUE;
  appdata.cnet = appdata.skeleton = appdata.animate = false;
  ConstructMyTeapot ( &appdata );
  ConstructMyTorus ( &appdata );
  InitLights ( &appdata );
  appdata.mytexture = LoadMyTextures ();
} /*InitMyWorld*/

void DrawMyTeapot ( AppData *ad )
{
  if ( ad->skeleton )
    glPolygonMode ( GL_FRONT_AND_BACK, GL_LINE );
  else
    glPolygonMode ( GL_FRONT_AND_BACK, GL_FILL );
  glUseProgram ( ad->brprog.program_id[0] );
  glUniform1i ( ad->brprog.ColourSourceLoc, ad->brprog.colour_source );
  ChooseMaterial ( &ad->mat, 0 );
  if ( ad->brprog.colour_source == 2 ) {
    BindBezPatchTextureBuffer ( ad->myteapot );
    glBindTexture ( GL_TEXTURE_2D, ad->mytexture );
    DrawBezierPatches ( ad->myteapot );
    glBindTexture ( GL_TEXTURE_2D, 0 );
  }
  else
    DrawBezierPatches ( ad->myteapot );
} /*DrawMyTeapot*/

void DrawMyTorus ( AppData *ad )
{
  if ( ad->skeleton )
    glPolygonMode ( GL_FRONT_AND_BACK, GL_LINE );
  else
    glPolygonMode ( GL_FRONT_AND_BACK, GL_FILL );
  glUseProgram ( ad->brprog.program_id[0] );
  glUniform1i ( ad->brprog.ColourSourceLoc,
                ad->brprog.colour_source > 0 ? 1 : 0 );
  ChooseMaterial ( &ad->mat, 1 );
  DrawBezierPatches ( ad->mytorus );
} /*DrawMyTorus*/

void DrawMyCNet ( BezierPatchObjf *bp, GLuint prog_id )
{
  glUseProgram ( prog_id );
  DrawBezierNets ( bp );
} /*DrawMyCNet*/

void RedrawMyWorld ( void )
{
  glClearColor ( 1.0, 1.0, 1.0, 1.0 );
  glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
  glEnable ( GL_DEPTH_TEST );
  LoadMMatrix ( &appdata.trans, appdata.teapot_mmatrix );
  DrawMyTeapot ( &appdata );
  if ( appdata.cnet )
    DrawMyCNet ( appdata.myteapot, appdata.brprog.program_id[1] );
  LoadMMatrix ( &appdata.trans, appdata.torus_mmatrix );
  DrawMyTorus ( &appdata );
  if ( appdata.cnet )
    DrawMyCNet ( appdata.myteapot, appdata.brprog.program_id[1] );
  glUseProgram ( 0 );
  glFlush ();
  ExitIfGLError ( "RedrawMyWorld" );
} /*RedrawMyWorld*/

void DeleteMyWorld ( void )
{
  glUseProgram ( 0 );
  glDeleteProgram ( appdata.brprog.program_id[0] );
  glDeleteProgram ( appdata.brprog.program_id[1] );
  glDeleteBuffers ( 1, &appdata.trans.trbuf );
  glDeleteBuffers ( 1, &appdata.light.lsbuf );
  glDeleteBuffers ( 1, &appdata.mat.matbuf );
  DeleteBezierPatches ( appdata.myteapot );
  DeleteBezierPatches ( appdata.mytorus );
  DeleteEmptyVAO ();
  glDeleteTextures ( 1, &appdata.mytexture );
#ifdef DEBUG_BUFFERS_ALLOCATION
  DumpBufferIdentifiers ();
#endif
} /*DeleteMyWorld*/

char ProcessCharCommand ( char charcode )
{
  switch ( toupper ( charcode ) ) {
case '+':
    if ( appdata.TessLevel < MAX_TESS_LEVEL ) {
      SetBezierPatchTessLevel ( appdata.myteapot, ++appdata.TessLevel );
      SetBezierPatchTessLevel ( appdata.mytorus, appdata.TessLevel );
      return true;
    }
    else return false;
case '-':
    if ( appdata.TessLevel > MIN_TESS_LEVEL ) {
      SetBezierPatchTessLevel ( appdata.myteapot, --appdata.TessLevel );
      SetBezierPatchTessLevel ( appdata.mytorus, appdata.TessLevel );
      return true;
    }
    else return false;
case 'N':
    appdata.BezNormals = appdata.BezNormals == 0;
    SetBezierPatchNVS ( appdata.myteapot, appdata.BezNormals );
    SetBezierPatchNVS ( appdata.mytorus, appdata.BezNormals ); 
    return true;
case 'C':
    appdata.cnet = !appdata.cnet;
    return true;
case 'S':
    appdata.skeleton = !appdata.skeleton;
    return true;
case 'B':
    glUseProgram ( appdata.brprog.program_id[0] );
    glUniform1i ( appdata.brprog.LightingModelLoc,
          appdata.brprog.lighting_model = !appdata.brprog.lighting_model );
    return true;
case ' ':
    if ( (appdata.animate = !appdata.animate) )
      TimerTic ();
    return appdata.animate;
case '0':
    appdata.brprog.colour_source = 0;
    return true;
case '1':
    appdata.brprog.colour_source = 1;
    return true;
case '2':
    appdata.brprog.colour_source = 2;
    return true;
default:
    return false;
  }
} /*ProcessCharCommand*/

char MoveOn ( void )
{
  double dt;

  dt = TimerTocTic ();
  if ( appdata.animate ) {
    if ( (appdata.teapot_rot_angle += ANGULAR_VELOCITY1 * dt) >= PI )
      appdata.teapot_rot_angle -= 2.0*PI;
    SetupTeapotMatrix ( &appdata );
  }
  if ( (appdata.torus_rot_angle += ANGULAR_VELOCITY2 * dt) >= PI )
    appdata.torus_rot_angle -= 2.0*PI;
  SetupTorusMatrix ( &appdata );
  return true;
} /*MoveOn*/

