#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "../utilities/openglheader.h"

#include "../utilities/utilities.h"
#include "../utilities/linkage.h"
#include "lights.h"
#include "trans.h"
#include "../utilities/mygltext.h"
#include "app1e.h"
#include "app1eproc.h"
#include "app1eklinkage.h"

#define MAX_SPEED 100.0
#define NOM_SPEED  10.0
#define MIN_SPEED   1.0
#define SPEED_FCT   1.2

#define MIN_ZOOM 0.2
#define MAX_ZOOM 5.0
#define ZOOM_FCT 1.05

GLuint  program_id[2];
GLuint  icos_vao, icos_vbo[4];
TransBl trans;
LightBl light;
int     win_width, win_height;
float   left, right, bottom, top, near, far;
const float viewer_pos0[4] = {0.0,0.0,10.0,1.0};
float   viewer_rvec[3] = {1.0,0.0,0.0};
double  viewer_rangle = 0.0;
int     speedexp = 0, zoomexp = 0;
double  speed = 10.0, phase = 0.0;
double  zoom = 1.0;
kl_linkage *mylinkage;

GLint tl0loc, tl1loc, csloc, colourloc;

myTextObject *vptext;
myFont  *fonts[2], *font;

void LoadMyShaders ( void )
{
  static const char *filename[6] =
    { "app1e.vert.glsl", "app1e.tesc.glsl", "app1e.tese.glsl",
      "app1e.geom.glsl", "app1e1.frag.glsl", "app1d3.frag.glsl" };
  static const GLuint shtype[6] =
    { GL_VERTEX_SHADER, GL_TESS_CONTROL_SHADER, GL_TESS_EVALUATION_SHADER,
      GL_GEOMETRY_SHADER, GL_FRAGMENT_SHADER, GL_FRAGMENT_SHADER };
  static const GLchar *uvnames[3] = { "tl", "cs", "colour" };
  GLuint shader_id[6], sh[5];
  int    i;

  for ( i = 0; i < 6; i++ )
    shader_id[i] = CompileShaderFiles ( shtype[i], 1, &filename[i] );
  program_id[0] = LinkShaderProgram ( 5, shader_id, "0" );
  memcpy ( sh, shader_id, 4*sizeof(GLuint) );
  sh[4] = shader_id[5];
  program_id[1] = LinkShaderProgram ( 5, sh, "1" );
  tl0loc = glGetUniformLocation ( program_id[0], uvnames[0] );
  tl1loc = glGetUniformLocation ( program_id[1], uvnames[0] );
  csloc = glGetUniformLocation ( program_id[1], uvnames[1] );
  colourloc = glGetUniformLocation ( program_id[1], uvnames[2] );
  GetAccessToTransBlockUniform ( program_id[1] );
  GetAccessToLightBlockUniform ( program_id[1] );
  AttachUniformTransBlockToBP ( program_id[0] );
  for ( i = 0; i < 6; i++ )
    glDeleteShader ( shader_id[i] );
  ExitIfGLError ( "LoadMyShaders" );
} /*LoadMyShaders*/

void NotifyViewerPos ( void )
{
  GLchar s[60];

  sprintf ( s, "x = %5.2f, y = %5.2f, z = %5.2f",
            trans.eyepos[0], trans.eyepos[1], trans.eyepos[2] );
  SetTextObjectContents ( vptext, s, 0, win_height-1, font );
} /*NotifyViewerPos*/

void InitViewMatrix ( void )
{
  memcpy ( trans.eyepos, viewer_pos0, 4*sizeof(GLfloat) );
  M4x4Translatef ( trans.vm, -viewer_pos0[0], -viewer_pos0[1], -viewer_pos0[2] );
  LoadVPMatrix ( &trans );
  NotifyViewerPos ();
} /*InitViewMatrix*/

void RotateViewer ( double delta_xi, double delta_eta )
{
  float   vi[3], lgt, vk[3];
  double  angi, angk;

  if ( delta_xi == 0 && delta_eta == 0 )
    return;  /* natychmiast uciekamy - nie chcemy dzielic przez 0 */
  vi[0] = (float)delta_eta*(top-bottom)/(float)win_height;
  vi[1] = (float)delta_xi*(right-left)/(float)win_width;
  vi[2] = 0.0;
  lgt = sqrt ( V3DotProductf ( vi, vi ) );
  vi[0] /= lgt;  vi[1] /= lgt;
  angi = -0.052359878;  /* -3 stopnie */
  V3CompRotationsf ( vk, &angk, viewer_rvec, viewer_rangle, vi, angi );
  memcpy ( viewer_rvec, vk, 3*sizeof(float) );
  viewer_rangle = angk;
  M4x4RotateVfv ( trans.vm, viewer_rvec, -viewer_rangle );
  M4x4MultMTVf ( trans.eyepos, trans.vm, viewer_pos0 );
  M4x4InvTranslateMfv ( trans.vm, viewer_pos0 );
  LoadVPMatrix ( &trans );
  NotifyViewerPos ();
} /*RotateViewer*/

void ConstructIcosahedronVAO ( void )
{
#define A 0.52573115
#define B 0.85065085
  static const GLfloat vertpos[12][3] =
    {{ -A,0.0, -B},{  A,0.0, -B},{0.0, -B, -A},{ -B, -A,0.0},
     { -B,  A,0.0},{0.0,  B, -A},{  A,0.0,  B},{ -A,0.0,  B},
     {0.0, -B,  A},{  B, -A,0.0},{  B,  A,0.0},{0.0,  B,  A}};
  static const GLubyte vertcol[12][3] =
    {{255,0,0},{255,127,0},{255,255,0},{127,255,0},{0,255,0},{0,255,127},
     {0,255,255},{0,127,255},{0,0,255},{127,0,255},{255,0,255},{255,0,127}};
  static const GLubyte vertind[96] =
     { 0, 1, 2, 0, 3, 4, 0, 5, 1, 9, 2, 8, 3, /* lamana, od 0 */
       7, 4, 11, 5, 10, 9, 6, 8, 7, 6, 11, 7,
       1, 10, 6,                              /* lamana, od 25 */
       2, 3, 4, 5, 8, 9, 10, 11,              /* 4 odcinki, od 28 */
       0, 1, 2, 0, 2, 3, 0, 3, 4, 0, 4, 5,    /* trojkatne platy, od 36 */
       0, 5, 1, 6, 7, 8, 6, 8, 9, 6, 9, 10,
       6, 10, 11, 6, 11, 7, 1, 9, 2, 9, 8, 2,
       2, 8, 3, 8, 7, 3, 3, 7, 4, 7, 11, 4,
       4, 11, 5, 11, 10, 5, 10, 1, 5, 10, 9, 1};

  glGenVertexArrays ( 1, &icos_vao );
  glBindVertexArray ( icos_vao );
  glGenBuffers ( 3, icos_vbo );
  glBindBuffer ( GL_ARRAY_BUFFER, icos_vbo[0] );
  glBufferData ( GL_ARRAY_BUFFER,
                 12*3*sizeof(GLfloat), vertpos, GL_STATIC_DRAW );
  glEnableVertexAttribArray ( 0 );
  glVertexAttribPointer ( 0, 3, GL_FLOAT, GL_FALSE,
                          3*sizeof(GLfloat), (GLvoid*)0 );
  glBindBuffer ( GL_ARRAY_BUFFER, icos_vbo[1] );
  glBufferData ( GL_ARRAY_BUFFER,
                 12*3*sizeof(GLubyte), vertcol, GL_STATIC_DRAW );
  glEnableVertexAttribArray ( 1 );
  glVertexAttribPointer ( 1, 3, GL_UNSIGNED_BYTE, GL_TRUE,
                          3*sizeof(GLubyte), (GLvoid*)0 );
  glBindBuffer ( GL_ELEMENT_ARRAY_BUFFER, icos_vbo[2] );
  glBufferData ( GL_ELEMENT_ARRAY_BUFFER,
                 96*sizeof(GLubyte), vertind, GL_STATIC_DRAW );
  glBindVertexArray ( 0 );
  ExitIfGLError ( "ConstructIcosahedronVAO" );
} /*ConstructIcosahedronVAO*/

void DrawTessIcos ( int prog, GLfloat tl, GLuint cs, GLfloat colour[3] )
{
  glBindVertexArray ( icos_vao );
  glUseProgram ( program_id[prog] );
  switch ( prog ) {
case 0:
    glUniform1f ( tl0loc, tl );
    break;
case 1:
    glUniform1f ( tl1loc, tl );
    glUniform1ui ( csloc, cs );
    glUniform3fv ( colourloc, 1, colour );
    break;
  }
  glPatchParameteri ( GL_PATCH_VERTICES, 3 );
  glDrawElements ( GL_PATCHES, 60, GL_UNSIGNED_BYTE, (GLvoid*)(36*sizeof(GLubyte)) );
  glBindVertexArray ( 0 );
  ExitIfGLError ( "DrawTessIcos" );
} /*DrawTessIcos*/

void InitLights ( void )
{
  GLfloat amb0[3] = { 0.005, 0.005, 0.005 };
  GLfloat dir0[3] = { 1.0, 1.0, 1.0 };
  GLfloat pos0[4] = { 0.0, 0.0, 0.0, 1.0 };
  GLfloat atn0[3] = { 1.0, 0.0, 0.0 };

  SetLightAmbient ( &light, 0, amb0 );
  SetLightDirect ( &light, 0, dir0 );
  SetLightPosition ( &light, 0, pos0 );
  SetLightAttenuation ( &light, 0, atn0 );
  SetLightOnOff ( &light, 0, 1 );
} /*InitLights*/

void SetSunPlanete ( my_object *sp, GLuint prog_id, GLfloat tl,
                     GLfloat colour[3], GLuint cs )
{
  sp->prog = prog_id;
  sp->tl = tl;
  memcpy ( sp->colour, colour, 3*sizeof(GLfloat) );
  sp->cs = cs;
} /*SetSunPlanete*/

void InitMyWorld ( int argc, char *argv[], int width, int height )
{
  static GLfloat fg[4] = { 0.0, 0.0, 1.0, 1.0 };
  static GLfloat bk[4] = { 0.0, 0.0, 0.0, 0.0 };
  static GLfloat colour0[3] = { 1.0, 1.0, 1.0 };
  static GLfloat colour1[3] = { 1.0, 1.0, 1.0 };

  LoadMyShaders ();
  LoadTextShaders ();
  TimerInit ();
  font = fonts[0] = NewFont18x10 ();
  fonts[1] = NewFont12x6 ();
  vptext = NewTextObject ( 60 );
  SetTextForeground ( fg );
  SetTextBackground ( bk );
  memset ( &trans, 0, sizeof(TransBl) );
  memset ( &light, 0, sizeof(LightBl) );
  trans.trbuf = NewUniformTransBlock ();
  light.lsbuf = NewUniformLightBlock ();
  InitViewMatrix ();
  ConstructIcosahedronVAO ();
  InitLights ();
  ResizeMyWorld ( width, height );
  if ( (mylinkage = ConstructMyLinkage ()) ) {
    SetSunPlanete ( &sunplanete[0], 0, 10, colour0, 0 );
    SetSunPlanete ( &sunplanete[1], 1, 5, colour0, 0 );
    SetSunPlanete ( &sunplanete[2], 1, 4, colour1, 1 );
    ArticulateMyLinkage ( mylinkage );
  }
  else
    ExitOnError ( "InitMyWorld" );
} /*InitMyWorld*/

void ResizeMyWorld ( int width, int height )
{
  float lr;

  glViewport ( 0, 0, win_width = width, win_height = height );
  SetupTextFrame ( width, height );
  lr = zoom*0.5533*(float)width/(float)height;  /* przyjmujemy aspekt rowny 1 */
  M4x4Frustumf ( trans.pm, NULL, left = -lr, right = lr,
                 bottom = -0.5533*zoom, top = 0.5533*zoom, near = 5.0, far = 15.0 );
  LoadVPMatrix ( &trans );
  NotifyViewerPos ();
} /*ResizeMyWorld*/

void RedrawMyWorld ( char showpos )
{
  glClearColor ( 0.0, 0.0, 0.0, 1.0 );
  glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
  if ( showpos )
    DisplayTextObject ( vptext );
  glEnable ( GL_DEPTH_TEST );
  glEnable ( GL_CULL_FACE );
  glCullFace ( GL_FRONT );
  glFrontFace ( GL_CW );
  kl_Redraw ( mylinkage );
  glFlush ();
  glUseProgram ( 0 );
} /*RedrawMyWorld*/

void DeleteMyWorld ( void )
{
  int i;

  glUseProgram ( 0 );
  for ( i = 0; i < 2; i++ )
    glDeleteProgram ( program_id[i] );
  glDeleteBuffers ( 1, &trans.trbuf );
  glDeleteBuffers ( 1, &light.lsbuf );
  glDeleteVertexArrays ( 1, &icos_vao );
  glDeleteBuffers ( 3, icos_vbo );
  DeleteTextObject ( vptext );
  for ( i = 0; i < 2; i++ )
    DeleteFontObject ( fonts[i] );
  TextCleanup ();
  DeleteEmptyVAO ();
  kl_DestroyLinkage ( mylinkage );
#ifdef DEBUG_BUFFERS_ALLOCATION
  DumpBufferIdentifiers ();
#endif
  ExitIfGLError ( "DeleteMyWorld" );
} /*DeleteMyWorld*/

char ChangeZoom ( char inczoom )
{
  int   e;
  double z;

  e = zoomexp;
  if ( inczoom ) {
    if ( (z = pow ( ZOOM_FCT, (double)(++e) )) > MAX_ZOOM )
      return false;
  }
  else {
    if ( (z = pow ( ZOOM_FCT, (double)(--e) )) < MIN_ZOOM )
      return false;
  }
  zoomexp = e;  zoom = z;
  ResizeMyWorld ( win_width, win_height );
  return true;
} /*ChangeZoom*/

char ChangeSpeed ( char incspeed )
{
  int    e;
  double s;

  e = speedexp;
  if ( incspeed ) {
    if ( (s = NOM_SPEED*pow ( SPEED_FCT, (double)(++e) )) > MAX_SPEED )
      return false;
  }
  else {
    if ( (s = NOM_SPEED*pow ( SPEED_FCT, (double)(--e) )) < MIN_SPEED )
      return false;
  }
  speedexp = e;  speed = s;
  return true;
} /*ChangeSpeed*/

char ProcessCharCommand ( char charcode )
{
  switch ( toupper ( charcode ) ) {
case 'M':
    font = font == fonts[0] ? fonts[1] : fonts[0];
    NotifyViewerPos ();
    return true;
case '+':
    ChangeSpeed ( true );
    return false;
case '-':
    ChangeSpeed ( false );
    return false;
default:
    return false;
  }
} /*ProcessCharCommand*/

char MoveOn ( void )
{
  ArticulateMyLinkage ( mylinkage );
  return true;
} /*MoveOn*/

