#include <set>
#include <utility>
#include <vector>
#include <algorithm>
#include <string>
#include <sys/timeb.h>
#include <stdio.h>
#include <cassert>

using namespace std;

const int maxp = 50;
const int max_len = 5;
const int MAX_G = 10000;
const char ID[] = "2006i";

set <pair <int,int> > edges;
set<string> names;
vector<string> dict;
vector <int> vertices;
vector <pair<int,int> > print;

inline void addEdge(set<pair<int,int> > &edges, int a, int b)
{
  if (a<b)
    edges.insert(make_pair(a,b));
  else if (a>b)
    edges.insert(make_pair(b,a));
}

string gen_name()
{
  string s;
  int len = (rand()%max_len)+1;
  while(len--) {
    char a = (rand()&1)?'A':'a';
    s += (char)(a+rand()%26);
  }

  return s;
}

int init(int n, int varn=0)
{
  if (varn) {
    n -= rand()%varn;
    if (n<2) n=2;
  }

  vertices.clear();
  names.clear();
  dict.clear();
  dict.resize(n);
  edges.clear();

  for(int i=0; i<n; vertices.push_back(i++)) {
    while (1) {
      string s = gen_name();
      if( names.find(s) == names.end() ) {
        names.insert(s);
        dict[i] = s;
        break;
      }
    }
  };

  random_shuffle(vertices.begin(),vertices.end());
  return n;
}

void begin_file(const char *przyrostek)
{
  char nazwa[BUFSIZ];
  snprintf(nazwa, sizeof(nazwa), "%s%s.in",ID,przyrostek);
  freopen(nazwa,"w",stdout);
  fprintf(stderr,"%s... ",nazwa);
}

void end_file()
{
  printf("0 0\n");
  fprintf(stderr,"OK\n");
}

void write(int n)
{
  print.clear();
  for(__typeof(edges.begin()) it = edges.begin(); it != edges.end(); ++it) {
    if( rand()&1 ) {
      print.push_back(*it);
    } else {
      print.push_back(make_pair(it->second, it->first));
    }
  }
  random_shuffle(print.begin(), print.end());

  printf("%d %u\n",n, print.size());
  for(unsigned int i=0; i<print.size(); ++i) {
    printf("%s %s%c",dict[print[i].first].c_str(), dict[print[i].second].c_str(),
      (i+1==print.size() ? '\n' : ' ')
      );
  }
  edges.clear();
}

vector<int> gen_tree(int size)
{
  vector<int> tree;
  if (vertices.size() && size--) {
    tree.push_back(vertices.back());
    vertices.pop_back();
    while(size-- && vertices.size()) {
      int w = vertices.back();
      vertices.pop_back();
      addEdge(edges,w,tree[rand()%tree.size()]);
      tree.push_back(w);
    }
  }
  return tree;
}

vector<int> gen_clique(int size)
{
  vector<int> clique;
  while (size-- && vertices.size()) {
    int w = vertices.back();
    vertices.pop_back();
    for(__typeof(clique.begin()) k=clique.begin(); k!=clique.end(); ++k) {
      addEdge(edges,w,*k);
    }
    clique.push_back(w);
  }
  return clique;
}

vector<int> gen_path(int size)
{
  vector<int> path;
  while( size-- && vertices.size() ) {
    int w = vertices.back();
    vertices.pop_back();
    path.push_back(w);
  }
  for(unsigned int i = 1; i<path.size(); ++i) {
    addEdge(edges,path[i-1], path[i]);
  }
  return path;
}

vector<int> gen_cycle(int size)
{
  vector<int> cycle;
  while( size-- && vertices.size() ) {
    int w = vertices.back();
    vertices.pop_back();
    cycle.push_back(w);
  }
  if( cycle.size() ) {
    addEdge(edges, cycle[cycle.size()-1], cycle[0]);
    for(unsigned int i = 1; i<cycle.size(); ++i) {
      addEdge(edges, cycle[i-1], cycle[i]);
    }
  }
  return cycle;
}

int gen_random(const vector<int> &vertices, int g, int varm=0)
{
  set<pair<int,int> > local;
  int n = vertices.size();

  assert(g<=MAX_G);

  int m = n*(n-1)/2 * g/MAX_G;
  if (varm) {
    m -= rand()%varm;
  }

  if (m<1 && edges.size()==0)
    m = 1;

  while ((int) local.size() < m) {
    const int a = vertices[rand()%n];
    const int b = vertices[rand()%n];
    addEdge(local,a,b);
  }
  edges.insert(local.begin(), local.end());

  return m;
}

void test_of_size(int k, int vark)
{
  int n;

  for(int i = 0; i<10; ++i) {
    n = init(k,vark);
    vector<int> tree = gen_tree(n);
    gen_random(tree,i*(MAX_G/10));
    write(n);
  }

  for(int i = 0; i<10; ++i) {
    n = init(k,vark);
    vector<int> path = gen_path(n);
    gen_random(path,i*(MAX_G/20));
    write(n);
  }

  for(int i=1; i<k ; ++i) {
    n = init(k);
    int w = gen_path(i)[0];
    for(unsigned int j=0; j<vertices.size(); ++j) {
      addEdge(edges,w,vertices[j]);
    }
    write(n);
  }

  for(int i =0; i<10; ++i) {
    n = init(k,vark);
    vector<int> subgraph = gen_path(n);
    vertices.insert(vertices.end(), subgraph.begin(), subgraph.end());
    gen_clique(n>>1);
    gen_random(subgraph,i*(MAX_G/30));
    write(n);
  }

  n = init(k);
  int w = vertices.back();
  vertices.pop_back();
  while( vertices.size() ) {
    vector<int> path = gen_path(2);
    assert(path.size());
    addEdge(edges,w,path[0]);
  }
  write(n);

  n = init(k,vark);
  gen_cycle(n);
  write(n);

  for(int i=0; i<k>>1 ; ++i) {
    n = init(k,vark);
    if( i >= n) break;
    gen_clique(n-i);
    write(n);
  }

  for(int i=0; i<100; ++i) {
    n = init(k,vark);
    gen_random(vertices,i*(MAX_G/150));
    write(n);
  }
}


void write_test(const char *suffix, int size)
{
  begin_file(suffix);
  test_of_size(size,3);
  end_file();
}

int main()
{
  srand(920);
  write_test("1",5);

  srand(921);
  write_test("2",10);

  srand(922);
  write_test("3",15);

  srand(923);
  write_test("4",20);

  srand(924);
  write_test("5",25);

  srand(925);
  write_test("6",30);

  srand(926);
  write_test("7",35);

  srand(927);
  write_test("8",40);

  srand(928);
  write_test("9",45);

  srand(929);
  write_test("10",50);
}
