/*
 * Author: Tomasz Czajka, 2002
 * Changes: Aleksander Mielczarek, Jakub Radoszewski, 2006
 */
#include <stdio.h>
#include <limits.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <vector>
using namespace std;

#define TOKEN_STD       0
#define TOKEN_EOF      -1
#define TOKEN_EOLN     -2
#define TOKEN_DOS_EOLN -3
#define TOKEN_SPACE    -4
#define TOKEN_TAB      -5
#define MAX_TOKEN_LEN 512

class TextFile {
public:
  TextFile();
  ~TextFile();
  int init(char *filename);
  int compare(TextFile *t);
  int getToken(char *token, int max_len);
  int eof();  
  int readInt(int min,int max);
  int readEof();
  int readSpace();  
  int readEoln();
  int readEolnOrEof();
  int error(char *message);
  /* JRAD: */
  void spr();
     
private:
  FILE *f;
  int line;
  int tno;
  char *token2str(int token, char *t_str);
  int err(char *expectedtoken, char *readtoken);
};


#define WHITE(ch) (ch==' ' || ch=='\t' || ch==13 || ch==EOF)
#define WHITE_OR_EOLN(ch) (ch==' ' || ch=='\t' || ch==13 || ch=='\n' || ch==EOF)

// ----------------------------------------------------------------------

class InputChecker : public TextFile {
public:
  int check();
};


int main(int argc, char *argv[]) {
    InputChecker *ic;


    ic=new InputChecker;

    if (argc==2) {
      if (ic->init(argv[1])) {
	fprintf(stderr,"can't open: %s\n",argv[1]);
	return 1;
      }
    } else {
      if (ic->init("")) {
	fprintf(stderr,"can't open: stdin\n");	
      }
    }

    ic->check();
    return 0;
}

//----
/**
 * Klasa wspomagajca obsug plikw tekstowych
 */

/**
 * Konstruktor, inicjalizacja wewntrznych struktur
 */
TextFile::TextFile() {
  f=NULL;
  line=1;
  tno=0;
}

/**
 * Destruktor, zamknicie pliku
 */
TextFile::~TextFile() {
  if (f!=NULL) fclose(f);
}

/**
 * Otworzenie pliku
 * @param filename nazwa pliku
 */
int TextFile::init(char *filename) {
  if (strlen(filename)) {
    f=fopen(filename,"r");
    if (f==NULL) return -1;
  } else {
    f=stdin;
  }
  return 0;
}

/**
 * Pobranie pojedynczego tokena z pliku
 * @param token wskanik do miejsca na wynik
 * @param max_len maksymalna dugo wyniku
 */
int TextFile::getToken(char *token,int max_len) {
  int i;
  int ch;

  token[0]=0;

  // omijanie bialych znakow
  ch=EOF;
  if (!feof(f)) ch=fgetc(f);

  if (ch==EOF)   return TOKEN_EOF;
  if (ch==' ')   return TOKEN_SPACE;
  if (ch=='\t')  return TOKEN_TAB;
  if (ch=='\n')  return TOKEN_EOLN;
  if (ch==13)    return TOKEN_DOS_EOLN;

  i=0;
  while (!WHITE_OR_EOLN(ch)) {
    token[i]=ch;
    i++;
    if (feof(f)) break;
    if (i==max_len-1) break;
    ch=fgetc(f);
  }
  token[i]=0;

  if (WHITE_OR_EOLN(ch)) ungetc(ch,f);
  
  tno++;
  return TOKEN_STD;
}

int TextFile::readInt(int min,int max) {
  char buf[10];  
  int token=getToken(buf,sizeof(buf));
  if (token!=TOKEN_STD) {
    err("Int",token2str(token,buf));
  }
  // teraz trzeba policzyc wartosc
  long long value=0;
  int minus=0;
  int i,n;
  i=0;n=strlen(buf);
  if (buf[0]=='-') { i++; minus=1; }
  while (i<n) {
    if (buf[i]<'0' || buf[i]>'9')        err("Int",buf);
    if (buf[i]=='0' && value==0 && n!=1) err("Int",buf);
    value=value*10+(buf[i]-'0');
    if (value>LONG_MAX)  err("Int",buf);
    i++;
  }
  if (minus) value=-value;
  char int_desc[100];
  snprintf(int_desc,sizeof(int_desc),"Int[%d,%d]",min,max);
  if (value>max || value<min) err(int_desc,buf);
  return value;
}

int TextFile::readEoln() {  
  char buf[100];
  int token=getToken(buf,sizeof(buf));
  if (token!=TOKEN_EOLN) err("EOLN",token2str(token,buf));
  line++;tno=0;
  return 0;
}

int TextFile::readEof() {  
  char buf[100];
  int token=getToken(buf,sizeof(buf));
  if (token!=TOKEN_EOF) err("EOF",token2str(token,buf));
  return 0;
}

int TextFile::readSpace() {  
  char buf[100];
  int token=getToken(buf,sizeof(buf));
  if (token!=TOKEN_SPACE) err("SPACE",token2str(token,buf));
  return 0;
}

int TextFile::readEolnOrEof() {
  char buf[100];
  int token=getToken(buf,sizeof(buf));
  if (token!=TOKEN_EOLN && token!=TOKEN_EOF) 
    err("EOLN_OR_EOF",token2str(token,buf));
  if (token==TOKEN_EOLN) {
    line++;tno=0;
  }
  return token;
}

/**
 * Czy osignito koniec pliku
 */
int TextFile::eof() {
  return feof(f);
}

char *TextFile::token2str(int token, char *t_str) {
  static char buf[1024];
  bzero(&buf,sizeof(buf));
  switch(token) {
  case TOKEN_STD:   strncpy(buf,t_str,sizeof(buf)-1); break;
  case TOKEN_EOF:   sprintf(buf,"EOF");   break;
  case TOKEN_EOLN:  sprintf(buf,"EOLN");  break;
  case TOKEN_SPACE: sprintf(buf,"SPACE"); break;
  case TOKEN_TAB:   sprintf(buf,"TAB");   break;
  case TOKEN_DOS_EOLN: sprintf(buf,"DOS_EOLN");  break;
  default:
    sprintf(buf,"UNKNOWN");
  }
  return (char *)&buf;
}

int TextFile::err(char *expectedtoken, char *readtoken) {  
  fprintf(stdout,"ERROR[line:%d,token:%d] expected: %s, read: %s\n",
	  line,tno,expectedtoken,readtoken);
  exit(1);
}

int TextFile::error(char *message) 
{
   
     fprintf(stdout,"ERROR[line:%d] %s\n",
	               line,message);
     exit(1);
}


// Author: Aleksander Mielczarek, Jakub Radoszewski

#define MAX_N 50

int n, k;
char name[9];

int InputChecker::check()
{
  while (1)
  {
    n = readInt(0, MAX_N);
    readEoln();
    if (!n)
      break;
    
    for (int i = 0; i < n; i++)
    {
      getToken(name, 8);
      readSpace();
      k = readInt(1, 2000);
      readEoln();

      if (!strcmp(name, "IN") || !strcmp(name, "OUT"))
      {
        if (k > 20)
          error("k is too big");
      }
      else if (!strcmp(name, "COLLECT"))
      {
        if (k > 200)
          error("k is too big");
      }
      else if (strcmp(name, "PAY"))
        error("bad keyword");        
    }
  }
  readEof();
  fprintf(stdout, "OK\n");
  return 0;
}
