/*
 * Przykadowy weryfikator danych wejciowych
 * (naley zmieni funkcj InputChecker::check())
 *
 * Autor: Tomasz Czajka, 2002
 * Modyfikacje: Tomasz Wale, 2003.01
 */
#include <stdio.h>
#include <limits.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#define TOKEN_STD       0
#define TOKEN_EOF      -1
#define TOKEN_EOLN     -2
#define TOKEN_DOS_EOLN -3
#define TOKEN_SPACE    -4
#define TOKEN_TAB      -5
#define MAX_TOKEN_LEN 512

class TextFile {
public:
  TextFile();
  ~TextFile();
  int init(char *filename);
  int compare(TextFile *t);
  int getToken(char *token, int max_len);
  int eof();
  int readInt(int min,int max);
  int readFloat(int min,int max, int nfrac);
  int readEof();
  int readSpace();
  int readEoln();
  int readEolnOrEof();
  int error(char *message);

private:
  FILE *f;
  int line;
  int tno;
  char *token2str(int token, char *t_str);
  int err(char *expectedtoken, char *readtoken);
};


#define WHITE(ch) (ch==' ' || ch=='\t' || ch==13 || ch==EOF)
#define WHITE_OR_EOLN(ch) (ch==' ' || ch=='\t' || ch==13 || ch=='\n' || ch==EOF)

// ----------------------------------------------------------------------

class InputChecker : public TextFile {
public:
  int check();
};


int main(int argc, char *argv[]) {
    InputChecker *ic;


    ic=new InputChecker;

    if (argc==2) {
      if (ic->init(argv[1])) {
        fprintf(stderr,"can't open: %s\n",argv[1]);
        return 1;
      }
    } else {
      if (ic->init("")) {
        fprintf(stderr,"can't open: stdin\n");
      }
    }

    ic->check();
    return 0;
}

//----
/**
 * Klasa wspomagajca obsug plikw tekstowych
 */

/**
 * Konstruktor, inicjalizacja wewntrznych struktur
 */
TextFile::TextFile() {
  f=NULL;
  line=1;
  tno=0;
}

/**
 * Destruktor, zamknicie pliku
 */
TextFile::~TextFile() {
  if (f!=NULL) fclose(f);
}

/**
 * Otworzenie pliku
 * @param filename nazwa pliku
 */
int TextFile::init(char *filename) {
  if (strlen(filename)) {
    f=fopen(filename,"r");
    if (f==NULL) return -1;
  } else {
    f=stdin;
  }
  return 0;
}

/**
 * Pobranie pojedynczego tokena z pliku
 * @param token wskanik do miejsca na wynik
 * @param max_len maksymalna dugo wyniku
 */
int TextFile::getToken(char *token,int max_len) {
  int i;
  int ch;

  token[0]=0;

  // omijanie bialych znakow
  ch=EOF;
  if (!feof(f)) ch=fgetc(f);

  if (ch==EOF)   return TOKEN_EOF;
  if (ch==' ')   return TOKEN_SPACE;
  if (ch=='\t')  return TOKEN_TAB;
  if (ch=='\n')  return TOKEN_EOLN;
//  if (ch=='\r')  return getToken(token,max_len);
  if (ch==13)    return TOKEN_DOS_EOLN;

  i=0;
  while (!WHITE_OR_EOLN(ch)) {
    token[i]=ch;
    i++;
    if (feof(f)) break;
    if (i==max_len-1) break;
    ch=fgetc(f);
  }
  token[i]=0;

  if (WHITE_OR_EOLN(ch)) ungetc(ch,f);

  tno++;
  return TOKEN_STD;
}

int TextFile::readInt(int min,int max) {
  char buf[15];
  int token=getToken(buf,sizeof(buf));
  if (token!=TOKEN_STD) {
    err("Int",token2str(token,buf));
  }
  // teraz trzeba policzyc wartosc
  long long value=0;
  int minus=0;
  int i,n;
  i=0;n=strlen(buf);
  if (buf[0]=='-') { i++; minus=1; }
  while (i<n) {
    if (buf[i]<'0' || buf[i]>'9')        err("Int",buf);
    if (buf[i]=='0' && value==0 && n!=1) err("Int",buf);
    value=value*10+(buf[i]-'0');
    if (value>LONG_MAX)  err("Int",buf);
    i++;
  }
  if (minus) value=-value;
  if (value>max || value<min){
    char int_desc[100];
    snprintf(int_desc,sizeof(int_desc),"Int[%d,%d]",min,max);
    err(int_desc,buf);
  }
  return value;
}

int TextFile::readFloat(int min,int max, int nfrac) {
  char buf[16];
  int token=getToken(buf,sizeof(buf));
  if (token!=TOKEN_STD) {
    err("Float",token2str(token,buf));
  }
  // teraz trzeba policzyc wartosc
  long long value=0;
  int minus=0;
  int zero=0;
  int kropka=0;
  int n=strlen(buf);
  int i=0;
  if (buf[0]=='-') { i++; minus=1; }
  for(;i<n;++i) {
    if (buf[i]=='.' && zero==0 && value==0) err("Float",buf);
    if (buf[i]=='.') {
      ++kropka;
      continue;
    }
    if (buf[i]<'0' || buf[i]>'9')        err("Float",buf);
    if (buf[i]=='0') ++zero;
    if (buf[i]=='0' && value==0 && zero>1 && kropka==0) err("Float1",buf);
    if( kropka && nfrac <=0 ) err("Float", buf);
    if( kropka ) --nfrac;
    value=value*10+(buf[i]-'0');
    if (value>LONG_MAX)  err("Float",buf);
  }
  while (nfrac--) value*=10;
  if (kropka > 1) err("Float", buf);
  if (minus) value=-value;
  if (value>max || value<min) {
    char int_desc[100];
    snprintf(int_desc,sizeof(int_desc),"Float[%d,%d]",min,max);
    err(int_desc,buf);
  }
//  fprintf(stderr, "Buf: %s, val: %d\n",buf, value);
  return value;
}

int TextFile::readEoln() {
  char buf[100];
  int token=getToken(buf,sizeof(buf));
  if (token!=TOKEN_EOLN) err("EOLN",token2str(token,buf));
  line++;tno=0;
  return 0;
}

int TextFile::readEof() {
  char buf[100];
  int token=getToken(buf,sizeof(buf));
  if (token!=TOKEN_EOF) err("EOF",token2str(token,buf));
  return 0;
}

int TextFile::readSpace() {
  char buf[100];
  int token=getToken(buf,sizeof(buf));
  if (token!=TOKEN_SPACE) err("SPACE",token2str(token,buf));
  return 0;
}

int TextFile::readEolnOrEof() {
  char buf[100];
  int token=getToken(buf,sizeof(buf));
  if (token!=TOKEN_EOLN && token!=TOKEN_EOF)
    err("EOLN_OR_EOF",token2str(token,buf));
  if (token==TOKEN_EOLN) {
    line++;tno=0;
  }
  return token;
}

/**
 * Czy osignito koniec pliku
 */
int TextFile::eof() {
  return feof(f);
}

char *TextFile::token2str(int token, char *t_str) {
  static char buf[1024];
  bzero(&buf,sizeof(buf));
  switch(token) {
  case TOKEN_STD:   strncpy(buf,t_str,sizeof(buf)-1); break;
  case TOKEN_EOF:   sprintf(buf,"EOF");   break;
  case TOKEN_EOLN:  sprintf(buf,"EOLN");  break;
  case TOKEN_SPACE: sprintf(buf,"SPACE"); break;
  case TOKEN_TAB:   sprintf(buf,"TAB");   break;
  case TOKEN_DOS_EOLN: sprintf(buf,"DOS_EOLN");  break;
  default:
    sprintf(buf,"UNKNOWN");
  }
  return (char *)&buf;
}

int TextFile::err(char *expectedtoken, char *readtoken) {
  fprintf(stdout,"ERROR[line:%d,token:%d] expected: %s, read: %s\n",
          line,tno,expectedtoken,readtoken);
  exit(1);
}

int TextFile::error(char *message)
{

     fprintf(stdout,"ERROR[line:%d] %s\n",
                       line,message);
     exit(1);
}


// ----------------------------------------------------------------------
// NALEZY ZMIENIC TA FUNKCJE!
// ----------------------------------------------------------------------


#include<string>
#include<vector>

#define MAXNUM    50
#define MAXAMOUNT 100
#define MINVALUE  -100
#define MAXVALUE  1000
#define MAXFRAC   2
#define REP(I,N)  for(int I=0;I<(N);++I)
#define PB        push_back
#define INFTY     1000000000

using namespace std;

int P,I;
vector<vector<int> > v;
int amounts[2][MAXNUM];


/* Wierzcholki 0..n-1 */
#define MAX_N 202

struct kraw
{
  int v1,v2,w,c; /* wierzch, waga, przepustowosc */
  int n,wh; /* sasiad */
};

int n,m,ost;
vector<kraw> t[MAX_N];
int d[MAX_N];
kraw par[MAX_N]; /* ojcowie */

/* Dodaje krawedz v1->v2 o wadze w i przepustowosci c. */
void add_edge(int v1,int v2,int w,int c)
{
  kraw e1;
  e1.v1=v1; e1.v2=v2; e1.w=w; e1.c=c;
  e1.wh = t[v1].size();
  e1.n =-1;
  t[v1].PB(e1);
}

void Bellman_Ford(int pocz,int kon)
{
  bool relaxed[MAX_N];
  REP(i,n) { d[i]=INFTY; par[i].v1=-1; relaxed[i]=false; }
  d[pocz]=0; relaxed[pocz]=true;
  bool dorel=true;
  while (dorel && d[kon]>ost)
  {
    dorel=false;
    REP(i,n) if (relaxed[i])
    {
      REP(j,(int)t[i].size()) if (t[i][j].c && d[t[i][j].v2]>d[i]+t[i][j].w)
      {
        relaxed[t[i][j].v2]=true;
        dorel=true;
        d[t[i][j].v2]=d[i]+t[i][j].w;
        par[t[i][j].v2]=t[i][j];
      }
      relaxed[i]=false;
    }
  }
}

/* Przeksztalca graf. */
int transform(int pocz,int kon)
{
  kraw e1;
  int v=kon,mi=INFTY;
  while (v!=pocz) /* o ile przeplyw wzrosnie */
  {
    mi=min(mi,par[v].c);
    v=par[v].v1;
  }
  v=kon;
  while (v!=pocz) /* przesylamy przeplyw, ew. dodajemy krawedzie */
  {
    kraw e=par[v];
    t[e.v1][e.wh].c-=mi;
    if (e.n!=-1) t[e.v2][e.n].c+=mi;
    else
    {
      e1.v1=e.v2; e1.v2=e.v1; e1.w=-e.w; e1.c=mi;
      e1.wh=t[e1.v1].size(); e1.n=e.wh;
      t[e.v1][e.wh].n=e1.wh;
      t[e1.v1].PB(e1);
    }
    v=e.v1;
  }
  return mi;
}

/* Zwraca wartosc przeplywu. */
int Busacker_Gowen(int pocz,int kon)
{
  bool poss=true;
  int wyn=0;
  ost=-1000000000;
  while (poss)
  {
    Bellman_Ford(pocz,kon);
    if (par[kon].v1==-1) poss=false;
    else
    {
      wyn+=transform(pocz,kon)*d[kon];
      ost=d[kon];
    }
  }
//  fprintf(stderr,"%.2f ",(float)wyn/100);
  return wyn;
}


int istnieje_pelne(void)
{
  int pocz = 0;
  int kon = P+I+1;
  n = kon+1;
  REP(i,n) t[i].clear();
  REP(i,P)
    add_edge(pocz,i+1, 0, amounts[0][i]);
  REP(i,I)
    add_edge(i+P+1,kon, 0, amounts[1][i]);

  REP(i,P)
    REP(j,I) {
      if (v[i][j] >0)
        add_edge(i+1,j+P+1,v[i][j], INFTY);
    }

  Busacker_Gowen(pocz,kon);
  for(unsigned i=0; i<t[0].size(); ++i)
    if (t[0][i].c != 0) return 0;  
  return 1;
}


int InputChecker::check() {
  int num=0;

  while(1) {
    P = readInt(0,MAXNUM);
    readSpace();
    I = readInt(0,MAXNUM);
    int pies = 0;
    int ices = 0;
    readEoln();
    if (P==0 && I==0) break;
    ++num;
    v.clear();
    for(int i=0; i<P; ++i) {
      int pom = readInt(1, MAXAMOUNT);
      amounts[0][i] = pom;
      pies += pom;
      if (i+1<P) readSpace();
    }
    readEoln();

    for(int i=0; i<I; ++i) {
      int pom = readInt(1, MAXAMOUNT);
      amounts[1][i] = pom;
      ices += pom;
      if (i+1<I) readSpace();
    }
    readEoln();

    if (ices != pies) error("Liczby ciastek i lodow nie sa sobie rowne");

    for(int i=0; i<P; ++i) {
      vector<int> row;
      for(int j=0; j<I; ++j) {
        int val = readFloat(MINVALUE, MAXVALUE, MAXFRAC);
        if( val <= 0 && val != -100) error("Niewlasciwy koszt krawedzi");
        if( val == -100) val = -1;
        row.push_back(val);
        if (j+1<I) readSpace();
      }
      readEoln();
      v.push_back(row);
    }
    if( !istnieje_pelne() ) error("Nie ma pelnego skojarzenia");
  }


  /* nowe zakoczenie: */
  readEof();
  /*------------------*/

   fprintf(stdout, "OK (%d tests)\n",num);
   return 0;
}
