#include <iostream>
#include <algorithm>

#define MAXN 50

class Edge {
public:
   char direction;
   int length;
   friend std::istream& operator>> (std::istream& i, Edge& edge);
   void neg();
   bool operator==(const Edge &e) const;
   bool operator!=(const Edge &e) const;
   bool operator<(const Edge &e) const;
   bool operator>(const Edge &e) const;
};

class Polyline {
private:
   bool check(int p1, int p2, int p3, int p4) const;
public:
   int  n;
   int  length;
   int  dist[MAXN + 1];
   Edge edges[MAXN + 1];
   friend std::istream& operator>> (std::istream& i, Polyline& poly);
   int find (int dist) const;
   void extractRange (int from, int to, Polyline &dest) const;
   void neg ();
   void rev ();
   void fix ();
   bool operator==(const Polyline &p) const;
   bool operator!=(const Polyline &p) const;
   bool equiv(const Polyline &p) const;
   bool canTile () const;
};

int caseId;
Polyline polygon;
Polyline part1;
Polyline part2;

std::istream& operator>> (std::istream& is, Edge& edge){
   is >> edge.direction >> edge.length;
   return is;
}

void Edge::neg (){
   switch (direction){
      case 'N': direction = 'S'; break;
      case 'E': direction = 'W'; break;
      case 'W': direction = 'E'; break;
      case 'S': direction = 'N'; break;
   }
}

bool Edge::operator==(const Edge &e) const {
   return (direction == e.direction) && (length == e.length);
}

bool Edge::operator!=(const Edge &e) const {
   return ! (*this == e);
}

bool Edge::operator<(const Edge &e) const {
   return (direction < e.direction) ||
      ((direction == e.direction) && (length < e.length));
}

bool Edge::operator>(const Edge &e) const {
   return (direction > e.direction) ||
      ((direction == e.direction) && (length > e.length));
}

std::istream& operator>> (std::istream& is, Polyline& poly){
   int i;
   is >> poly.n;
   poly.length = 0;
   for (i=0; i < poly.n; i++){
      poly.dist[i] = poly.length;
      is >> poly.edges[i];
      poly.length += poly.edges[i].length;
   }
   return is;
}

int Polyline::find (int d) const {
   int r;
   r = std::lower_bound (dist, dist + n, d) - dist;
   if (r < n) {
      return dist[r] == d ? r : r - 1;
   } else {
      return n - 1;
   }
}

void Polyline::extractRange(int from,
   int to, Polyline &dest) const {

   int p1, p2;
   
   p1 = find (from);
   p2 = find (to);
   dest.n = 0;
   if (from == to) return;
   if (dist[p1] != from){
      dest.n = 1;
      if ((p1 == p2) && (from <= to)){
         dest.edges[0].direction = edges[p1].direction;
         dest.edges[0].length    = to - from;
         return;
      }
      dest.edges[0].direction = edges[p1].direction;
      p1 = (p1 + 1) % n;
      dest.edges[0].length    = p1 ? dist[p1] - from : length - from;
   }
   while (p1 != p2) {
      dest.edges[dest.n] = edges[p1];
      p1 = (p1 + 1) % n;
      dest.n++;
   }
   if (dist[p2] != to) {
      dest.edges[dest.n].direction = edges[p2].direction;
      dest.edges[dest.n].length    = to - dist[p2];
      dest.n++;
   }
}

void Polyline::neg (){
   int i;
   for (i=0; i < n; i++){
      edges[i].neg ();
   }
}

void Polyline::rev (){
   std::reverse (edges, edges + n);
}

bool Polyline::operator==(const Polyline &p) const {
   int i;
   if (n != p.n) return false;
   for (i=0; i < n; i++) {
      if (edges[i] != p.edges[i]) {
         return false;
      }
   }
   return true;
}

bool Polyline::operator!=(const Polyline &p) const {
   return ! (*this == p);
}

void Polyline::fix () {
   if (n > 1) {
      if (edges[0].direction == edges[n - 1].direction) {
         edges[0].length += edges[n-1].length;
         n--;
      }
   }
}

template<class T> bool cyclicEquiv (T *v1, T *v2, int n) {
   int i, j, k;

   i = 0; j = 0; k = 0;
   while ((i < n) && (j < n)) {
      while (v1[(i + k) % n] == v2[(j + k) % n]) {
         k++;
         if (k == n) return true;
      }
      if (v1[i + k] > v2[j + k]){
         i = i + k + 1;
      } else {
         j = j + k + 1;
      }
      k = 0;
   }
   return false;
}


bool Polyline::equiv(const Polyline &p) const {
   if (n != p.n) {
      return false;
   }
   return cyclicEquiv(edges, p.edges, n);
}

bool Polyline::check(int p1, int p2, int p3, int p4) const {
   extractRange (p1, p2, part1);
   extractRange (p3, p4, part2);
   part2.rev ();
   part2.neg ();
   if (part1 != part2) {
      return false;
   }
   extractRange (p2, p3, part1);
   extractRange (p4, p1, part2);
   
   part1.fix ();
   
   part2.rev ();
   part2.neg ();
   part2.fix ();
   return part1.equiv (part2);
}

bool Polyline::canTile () const {
   int i, j, p1, p2, p3, p4;
   
   for (i = 0; i < n; i++) {
      for (j = 0; j < n; j++) {
         p1 = dist[i];
         p2 = dist[j];
         p3 = (p1 + length / 2) % length;
         p4 = (p2 + length / 2) % length;
         if (check(p1, p2, p3, p4) ||
             check(p3, p2, p1, p4) ||
             check(p1, p4, p3, p2) ||
             check(p3, p4, p1, p2)
            ) {
            return true;
         }
      }
   }
   return false;
}

int main (int argc, char **argv){
   for (caseId = 1; ; caseId++){
      std::cin >> polygon;
      if (polygon.n == 0){
         return 0;
      }
      std::cout << "Polygon " << caseId << ": "
         << (polygon.canTile() ? "Possible" : "Impossible") << "\n";
   }
   return 1;
}
