type tDenote = 
  | Den_none
(* _  (_, _)  *)
  | Den_one of int
(* x *)
  | Den_prod of (int * tDenote) list;;
(* (_, x, (_, _), (y, z)) *)

type tVal_prim = Primitives.tVal_prim;;
type tFun_prim =  int * Primitives.tVal_prim list * Primitives.tFun_prim;;

type tValue =
  | Val_prim  of tVal_prim
    (* wartosc prymitywna *)
  | Fun_prim  of tFun_prim
    (* funkcja prymitywna *)
  | Val_oper  of tValue * Primitives.tOper_prim * tValue
    (* operator primitywny *)
  | Val_if    of tValue * tValue * tValue
    (* warunek * if_true * if_false *)
  | Val_raise
    (* raise *)
  | Val_match of tValue * tDenote array * tValue array
    (* what_to_match * denotations * values *)
  | Val_fappl of tValue * tValue
    (* function * argument *)
  | Val_cappl of (int * string) * tValue
    (* konstruktor * wyliczony argument *)
  | Val_fun   of tDenote * tValue
    (* denote * cialo *)
  | Val_rec   of tValue
    (* Y ( val ) *)
  | Val_ident of int
    (* numer id *)
  | Val_cons  of int * string
    (* numer cons * nazwa *)
  | Val_ref   of tValue
    (* ref do wartosci *)
  | Val_prod  of tValue array;;
    (* produkt w tabeli *)

type tEnvi = (string * tValue) list;;

let is_lazy value =
  match value with
  | Val_rec _ -> true
  | _ -> false;;

(* ========================================================================== *)

let rec str_of_value value =
  match value with
  | Val_prim v -> Primitives.str_of_val v
  | Val_cappl ((_, n), x) -> n ^ " " ^ (str_of_value x)
  | Val_prod vl ->  "(" ^ (String.concat ", " (Array.to_list (Array.map str_of_value vl))) ^ ")"
  | Val_cons (_, _) -> " FUN "  
  | Fun_prim _ -> " FUN "
  | Val_raise -> " FUN "
  | Val_fun _ -> " FUN "
  | _ -> assert false;;


(* ========================================================================== *)

module Int =
struct
  type t = int

  let compare i1 i2 =
    compare i1 i2
end;;

module Int_map = Map.Make (Int);;

(* ========================================================================== *)

let comp_list l =
  let one_comp act_comp act_el =
    if act_comp = 0 then
      act_el
    else
      act_comp
  in
  List.fold_left one_comp 0 l;;

let rec val_comp v1 v2 =
  match (v1, v2) with
  | (Val_prim vp1, Val_prim vp2) -> Primitives.compare vp1 vp2
  | (Val_cappl ((h1, _), t1), Val_cappl ((h2, _), t2)) ->
    if compare h1 h2 <> 0 then
      compare h1 h2
    else
      val_comp t1 t2
  | (Val_prod a1, Val_prod a2) ->
    let l1 = Array.to_list a1 in
    let l2 = Array.to_list a2 in
    assert (List.length l1 = List.length l2);
    let comp = List.map2 val_comp l1 l2 in
      comp_list comp
  | _ -> assert false;;

(* ========================================================================== *)

let value_null  = Val_prim (Primitives.VP_unit);;
let value_false = Val_prim (Primitives.VP_bool false);;
let value_true  = Val_prim (Primitives.VP_bool true);;

(* ========================================================================== *)

let rec beta_reduct new_idents value =
  let reduct = beta_reduct new_idents in
  match value with
  | Val_prim _ ->
    value
  | Fun_prim _ ->
    value
  | Val_oper (e1, op, e2) ->
    Val_oper (reduct e1, op, reduct e2)
  | Val_if (w, t, f) ->
    Val_if (reduct w, reduct t, reduct f)
  | Val_raise ->
    value
  | Val_match (v, d, vs) ->
    Val_match (reduct v, d, Array.map reduct vs)
  | Val_fappl (f, x) ->
    Val_fappl (reduct f, reduct x)
  | Val_cappl (_, _) ->
    value
  | Val_fun (d, v) ->
    Val_fun (d, reduct v)
  | Val_rec _ ->
    value
  | Val_ident i ->
    if Int_map.mem i new_idents then
      Val_ref (Int_map.find i new_idents)
    else
      value
  | Val_cons (_, _) ->
    value
  | Val_ref _ ->
    value
  | Val_prod vs ->
    Val_prod (Array.map reduct vs);;

let rec get_one_ident value act_map den =
  match den with
  | Den_none -> act_map
  | Den_one i -> Int_map.add i value act_map
  | Den_prod dl ->
    match value with
    | Val_prod vs ->
      let add_prod_elem new_map (i, new_den) =
        get_one_ident (Array.get vs i) new_map new_den
      in
        List.fold_left add_prod_elem act_map dl
    | _ -> assert false;;

(* zwraca mape identyfikator -> wartosc *)
let get_all_idents value denote =
  get_one_ident value Int_map.empty denote

(* ========================================================================== *)

let rec eval_equal v1 v2 =
  if val_comp v1 v2 = 0 then
    value_true
  else
    value_false

and eval_scolon v1 v2=
  ignore (eval v2);
  eval v1

and eval_ge v1 v2 =
  if val_comp v1 v2 > 0 then
    value_true
  else
    value_false

and eval_le v1 v2 =
  if val_comp v1 v2 < 0 then
    value_true
  else
    value_false

and eval_geq v1 v2 =
  if val_comp v1 v2 >= 0 then
    value_true
  else
    value_false

and eval_leq v1 v2 =
  if val_comp v1 v2 <= 0 then
    value_true
  else
    value_false

and eval_neq v1 v2 =
  if val_comp v1 v2 <> 0 then
    value_true
  else
    value_false

and eval_or the_v1 v2 =
  if the_v1 = value_false then
    eval v2
  else
    value_true

and eval_and the_v1 v2 =
  if the_v1 = value_true then
    eval v2
  else
    value_false

and eval_xor v1 v2 =
  if v1 = value_true then
    v2
  else
    if v2 = value_true then
      value_false
    else
      value_true
    
and eval_add v1 v2 =
  match (v1, v2) with
  | (Val_prim (Primitives.VP_int i1), Val_prim (Primitives.VP_int i2)) ->
     Val_prim (Primitives.VP_int (Big_int.add_big_int i1 i2))
  | (Val_prim (Primitives.VP_rat i1), Val_prim (Primitives.VP_rat i2)) ->
     Val_prim (Primitives.VP_rat (Num.add_num i1 i2))
  | (Val_prim (Primitives.VP_int i1), Val_prim (Primitives.VP_rat i2)) ->
     Val_prim (Primitives.VP_rat (Num.add_num (Num.num_of_big_int i1) i2))
  | (Val_prim (Primitives.VP_rat i1), Val_prim (Primitives.VP_int i2)) ->
     Val_prim (Primitives.VP_rat (Num.add_num i1 (Num.num_of_big_int i2)))
  | _ -> assert false

and eval_sub v1 v2 =
  match (v1, v2) with
  | (Val_prim (Primitives.VP_int i1), Val_prim (Primitives.VP_int i2)) ->
     Val_prim (Primitives.VP_int (Big_int.sub_big_int i1 i2))
  | (Val_prim (Primitives.VP_rat i1), Val_prim (Primitives.VP_rat i2)) ->
     Val_prim (Primitives.VP_rat (Num.sub_num i1 i2))
  | (Val_prim (Primitives.VP_int i1), Val_prim (Primitives.VP_rat i2)) ->
     Val_prim (Primitives.VP_rat (Num.sub_num (Num.num_of_big_int i1) i2))
  | (Val_prim (Primitives.VP_rat i1), Val_prim (Primitives.VP_int i2)) ->
     Val_prim (Primitives.VP_rat (Num.sub_num i1 (Num.num_of_big_int i2)))     
  | _ -> assert false

and eval_mul v1 v2 =
  match (v1, v2) with
  | (Val_prim (Primitives.VP_int i1), Val_prim (Primitives.VP_int i2)) ->
     Val_prim (Primitives.VP_int (Big_int.mult_big_int i1 i2))
  | (Val_prim (Primitives.VP_rat i1), Val_prim (Primitives.VP_rat i2)) ->
     Val_prim (Primitives.VP_rat (Num.mult_num i1 i2))
  | (Val_prim (Primitives.VP_int i1), Val_prim (Primitives.VP_rat i2)) ->
     Val_prim (Primitives.VP_rat (Num.mult_num (Num.num_of_big_int i1) i2))
  | (Val_prim (Primitives.VP_rat i1), Val_prim (Primitives.VP_int i2)) ->
     Val_prim (Primitives.VP_rat (Num.mult_num i1 (Num.num_of_big_int i2)))     
  | _ -> assert false

and eval_div v1 v2 =
  match (v1, v2) with
  | (Val_prim (Primitives.VP_int i1), Val_prim (Primitives.VP_int i2)) ->
     Val_prim (Primitives.VP_int (Big_int.div_big_int i1 i2))
  | (Val_prim (Primitives.VP_rat i1), Val_prim (Primitives.VP_rat i2)) ->
     Val_prim (Primitives.VP_rat (Num.div_num i1 i2))
  | (Val_prim (Primitives.VP_int i1), Val_prim (Primitives.VP_rat i2)) ->
     Val_prim (Primitives.VP_rat (Num.div_num (Num.num_of_big_int i1) i2))
  | (Val_prim (Primitives.VP_rat i1), Val_prim (Primitives.VP_int i2)) ->
     Val_prim (Primitives.VP_rat (Num.div_num i1 (Num.num_of_big_int i2)))     
  | _ -> assert false

and eval_mod v1 v2 =
  match (v1, v2) with
  | (Val_prim (Primitives.VP_int i1), Val_prim (Primitives.VP_int i2)) ->
     Val_prim (Primitives.VP_int (Big_int.mod_big_int i1 i2))
  | (Val_prim (Primitives.VP_rat i1), Val_prim (Primitives.VP_rat i2)) ->
     Val_prim (Primitives.VP_rat (Num.mod_num i1 i2))
  | (Val_prim (Primitives.VP_int i1), Val_prim (Primitives.VP_rat i2)) ->
     Val_prim (Primitives.VP_rat (Num.mod_num (Num.num_of_big_int i1) i2))
  | (Val_prim (Primitives.VP_rat i1), Val_prim (Primitives.VP_int i2)) ->
     Val_prim (Primitives.VP_rat (Num.mod_num i1 (Num.num_of_big_int i2)))     
  | _ -> assert false

and eval_pow v1 v2 =
  match (v1, v2) with
  | (Val_prim (Primitives.VP_rat i1), Val_prim (Primitives.VP_rat i2)) ->
     Val_prim (Primitives.VP_rat (Num.power_num i1 i2))
  | (Val_prim (Primitives.VP_int i1), Val_prim (Primitives.VP_int i2)) ->
     Val_prim (Primitives.VP_rat (Num.power_num (Num.num_of_big_int i1) (Num.num_of_big_int i2)))
  | (Val_prim (Primitives.VP_int i1), Val_prim (Primitives.VP_rat i2)) ->
     Val_prim (Primitives.VP_rat (Num.power_num (Num.num_of_big_int i1) i2))
  | (Val_prim (Primitives.VP_rat i1), Val_prim (Primitives.VP_int i2)) ->
     Val_prim (Primitives.VP_rat (Num.power_num i1 (Num.num_of_big_int i2)))
  | (Val_prim (Primitives.VP_str i1), Val_prim (Primitives.VP_str i2)) ->
     Val_prim (Primitives.VP_str (i1 ^ i2))
  | _ -> assert false

(* ========================================================================== *)

and eval_oper (e1, op, e2) =
try
  match op with
  | Primitives.Op_equal ->
    eval_equal   (eval e1) (eval e2)
  | Primitives.Op_scolon ->
    eval_scolon  (eval e2) (eval e1)
  | Primitives.Op_ge ->
    eval_ge      (eval e1) (eval e2)
  | Primitives.Op_le ->
    eval_le      (eval e1) (eval e2)
  | Primitives.Op_geq ->
    eval_geq     (eval e1) (eval e2)
  | Primitives.Op_leq ->
    eval_leq     (eval e1) (eval e2)
  | Primitives.Op_neq ->
    eval_neq     (eval e1) (eval e2)
  | Primitives.Op_or ->
    eval_or      (eval e1) e2
  | Primitives.Op_and ->
    eval_and     (eval e1) e2
  | Primitives.Op_xor ->
    eval_xor     (eval e1) (eval e2)
  | Primitives.Op_add ->
    eval_add     (eval e1) (eval e2)
  | Primitives.Op_sub ->
    eval_sub     (eval e1) (eval e2)
  | Primitives.Op_mul ->
    eval_mul     (eval e1) (eval e2)
  | Primitives.Op_div ->
    eval_div     (eval e1) (eval e2)
  | Primitives.Op_mod ->
    eval_mod     (eval e1) (eval e2)
  | Primitives.Op_pow ->
    eval_pow     (eval e1) (eval e2)
with
| Primitives.Error s -> raise (Primitives.Error s)
| Primitives.Run_time_error s -> raise (Primitives.Run_time_error s)
(* | _ -> raise (Primitives.Run_time_error "Wrong infix-op arguments") *)

and eval_if (w, t, f) =
  let the_w = eval w in
  if the_w = value_false then
    eval f
  else
    eval t

and eval_raise vs =
  match vs with
  | Val_prim (Primitives.VP_str s) ->
    raise (Primitives.Run_time_error s)
  | _ -> assert false

and eval_match (value, ds, vs) =
  match eval value with
  | Val_cappl ((act_i, _), dx) ->
      eval_fun (Array.get ds act_i, Array.get vs act_i) dx
  | _ -> assert false

and eval_fun (d, v) x =
  let new_idents = get_all_idents x d in
  eval (beta_reduct new_idents v)

and eval_prim_fun (i, l, f) x =
  match x with
  | Val_prim v ->
    assert (i >= 1);
    let new_i = i-1 in
    let new_l = v :: l in
      if new_i = 0 then
        try
          Val_prim (f new_l)
	with
	| Primitives.Error s -> raise (Primitives.Error s)	
	| Primitives.Run_time_error s -> raise (Primitives.Run_time_error s)
	| _ -> raise (Primitives.Run_time_error "Wrong primitive-fun arguments")
      else
        Fun_prim (new_i, new_l, f)
  | _ -> assert false

and eval_fappl f x =
  let the_f = eval f in
  let the_x = 
    if is_lazy x then x else eval x
  in
  match the_f with
  | Val_fun (d, v) ->
    eval_fun (d, v) the_x
  | Val_cons (i, n) ->
    Val_cappl ((i, n), the_x)
  | Fun_prim (i, l, f) ->
    eval_prim_fun (i, l, f) the_x
  | Val_raise ->
    eval_raise the_x
  | _ -> assert false

and eval_ref v =
  if is_lazy v then
    eval v
  else
    v
  
and eval value =
  match value with
  | Val_prim _ -> value
  | Fun_prim _ -> value
  | Val_oper (e1, op, e2) -> eval_oper (e1, op, e2)
  | Val_if (w, t, f) -> eval_if (w, t, f)
  | Val_raise -> value
  | Val_match (v, d, m) -> eval_match (v, d, m)
  | Val_fappl (f, x) -> eval_fappl f x
  | Val_cappl (_, _) -> value
  | Val_fun _ -> value
  | Val_rec v -> eval_fappl v value
  | Val_ident _ -> value
  | Val_cons (_, _) -> value
  | Val_ref v -> eval_ref v
  | Val_prod vs -> Val_prod (Array.map eval vs);;

(* ========================================================================== *)

let rec trans_den_prod dl ind =
  match dl with
  | [] -> []
  | g::o ->
    (ind, trans_den g) :: (trans_den_prod o (ind+1))

and trans_den denote =
  match denote with
  | Semantic.Den_none -> Den_none
  | Semantic.Den_unit -> Den_none
  | Semantic.Den_ident (_, i, _) -> Den_one i
  | Semantic.Den_prod dl ->
    let res = trans_den_prod dl 0 in
    let is_not_none (_, d) = d <> Den_none in
    let new_res = List.filter is_not_none res in
      match new_res with
      | [] -> Den_none
      | l -> Den_prod l

let rec trans_match envi (e, m_list) =
  let the_val = trans_expr envi e in
  let simply ((_, i, _), d, e) = (i, d, e) in
  let cmp (i1, _, _) (i2, _, _) = compare i1 i2 in
  let get_den (_, d, _) = trans_den d in
  let get_val (_, _, e) = trans_expr envi e in
  let new_m_list = List.sort cmp (List.map simply m_list) in
  let (last, _, _) = List.hd (List.rev new_m_list) in
  let dens = Array.of_list (List.map get_den new_m_list) in
  let vals = Array.of_list (List.map get_val new_m_list) in
    assert (last = List.length m_list - 1);
    Val_match (the_val, dens, vals)

and trans_ref envi name =
  if List.mem_assoc name envi then
    let the_expr = List.assoc name envi in
      Val_ref the_expr
  else
    assert false

and trans_expr envi expr =
  let trans = trans_expr envi in
  match expr with
  | Semantic.Expr_val_prim e ->
    Val_prim e
  | Semantic.Expr_fun_prim (n, a, f) ->
    Fun_prim (n, a, f)
  | Semantic.Expr_oper_prim (e1, op, e2) ->
    Val_oper (trans e1, op, trans e2)
  | Semantic.Expr_if (w, t, f) ->
    Val_if (trans w, trans t, trans f)
  | Semantic.Expr_raise ->
    Val_raise
  | Semantic.Expr_match (e, m_list) ->
    trans_match envi (e, m_list)
  | Semantic.Expr_appl (f, x) ->
    Val_fappl (trans f, trans x)
  | Semantic.Expr_fun (d, v) ->
    Val_fun (trans_den d, trans v)
  | Semantic.Expr_rec v ->
    Val_rec (trans v)
  | Semantic.Expr_ident (_, i, _) ->
    Val_ident i
  | Semantic.Expr_cons (n, i, _) ->
    Val_cons (i, n)
  | Semantic.Expr_ref name ->
    trans_ref envi name
  | Semantic.Expr_prod el ->
    Val_prod (Array.of_list (List.map trans el))
  | Semantic.Expr_letin (d, e1 ,e2) ->
    Val_fappl (Val_fun (trans_den d, trans e2), trans e1);;


(* ========================================================================== *)

let make old_envi =
  let vals = Semantic.get_values old_envi in
  let step (name, (expr, (p_args, p_type), is_embed)) act_envi =
    let res_val = eval (trans_expr act_envi expr) in  
    if (!Primitives.show_full_val_info && not is_embed) then
    (
      print_string ("# " ^ name ^ (Semantic.str_of_poly_args p_args)^ " : " ^
       (Semantic.str_of_type p_type) ^ " = " ^ (str_of_value res_val) ^ "\n")
    );
    (name, res_val) :: act_envi;    
  in
    let res = List.fold_right step vals [] in
      if (!Primitives.show_full_val_info) then
        print_newline ();
      res;;

let run envi =
  let main_fun = List.assoc "main" envi in
    eval (Val_fappl (main_fun, value_null));;