(* typ *)
type tType =
  | Type_prim     of Primitives.tType_prim
  | Type_prod     of tType list
  | Type_fun      of tType * tType
  | Type_alg      of string * tType list * (string list)
  | Type_poly_alg of string * (string list)
  | Type_ident    of string;;

(* poly type args *)
type tDen_typ = string list;;

(* poly type *)
type tPoly_type = tDen_typ * tType;;

(* konstruktor w algebrze *)
type tCons = string * int * tPoly_type;;
(* nazwa * numer * typ *)

(* identyfikator *)
type tIdent = string * int * tType;;
(* nazwa * numer * typ *)

type tDenote =
  | Den_none
  | Den_unit
  | Den_ident of tIdent
  | Den_prod of tDenote list;;

(* czyste wyrazenie - bez typu *)
type tExpr =
  | Expr_val_prim  of Primitives.tVal_prim
  | Expr_fun_prim  of int * Primitives.tVal_prim list * Primitives.tFun_prim
  | Expr_oper_prim of tExpr * Primitives.tOper_prim * tExpr
  | Expr_if        of tExpr * tExpr * tExpr
  | Expr_raise
  | Expr_match     of tExpr * (tCons * tDenote * tExpr) list
  | Expr_appl      of tExpr * tExpr
  | Expr_fun       of tDenote * tExpr
  | Expr_rec       of tExpr
  | Expr_ident     of tIdent
  | Expr_cons      of tCons
  | Expr_ref       of string
  | Expr_prod      of tExpr list
  | Expr_letin     of tDenote * tExpr * tExpr;;

exception Semantic_error of string

exception Ident_error      of string * string * string
exception Ident_error_pure of string * string
(* tytul * ident *)

exception SType_error      of string * tType * string
exception SType_error_pure of string * tType
(* tytul * typ *)

exception Type_error      of string * tType * tType * string
exception Type_error_pure of string * tType * tType
(* tytul * typ * typ *)

(* ========================================================================== *)

type tEnvi =
{
  mutable values : (string * (tExpr * tPoly_type * bool)) list;
  (* nazwa * (wyrazenie * poly typ * czy_wbudowane) *)
  mutable types  : (string * tPoly_type) list;
  mutable fun_alg_types : string list;
  mutable conss  : (string * tCons) list;
  mutable idents : (string * tIdent) list
};;

let get_values envi = envi.values;;

(* ========================================================================== *)

let unit_type = Type_prim (Primitives.TP_unit);;
let bool_type = Type_prim (Primitives.TP_bool);;
let  int_type = Type_prim (Primitives.TP_int );;
let  rat_type = Type_prim (Primitives.TP_rat );;
let char_type = Type_prim (Primitives.TP_char);;
let  str_type = Type_prim (Primitives.TP_str );;

let typ_of_ptyp t = Type_prim t;;

let rec typ_fun_of_list l =
  match l with
  | [] -> assert false
  | [g] -> g
  | g::o -> Type_fun (g, typ_fun_of_list o);;

let make_val_of_prim (n, (v, t)) =
  (n, (Expr_val_prim v, ([], typ_of_ptyp t), true));;

let make_fun_of_prim (n, (f, ts)) =
  let ntity = List.length ts - 1 in
  let the_types = List.map typ_of_ptyp ts in
  let type_fun = typ_fun_of_list the_types in
    (n, (Expr_fun_prim (ntity, [], f), ([], type_fun), true));;

let make_typ_of_prim (n, t) =
  (n, ([], typ_of_ptyp t));;

let starting_values =
[
  ("raise", (Expr_raise,(["a"], Type_fun (Type_prim Primitives.TP_str, Type_ident "a")), true))
];;

let starting_envi =
{
  values = 
    (List.map make_val_of_prim Primitives.all_prim_vals) @
    (List.map make_fun_of_prim Primitives.all_prim_funs) @
    starting_values;

  types =
     List.map make_typ_of_prim Primitives.all_prim_typs;
     
  fun_alg_types = [];
  conss = [];
  idents = []
};;

(* ========================================================================== *)

let the_envi = starting_envi;;

let initialize_envi =
  the_envi.values <- starting_envi.values;
  the_envi.types  <- starting_envi.types;
  the_envi.conss  <- starting_envi.conss;
  the_envi.fun_alg_types <- starting_envi.fun_alg_types;
  the_envi.idents <- starting_envi.idents;;

(* czy dany obiekt jest znany juz *)
let known_value nazwa = List.mem_assoc nazwa the_envi.values;;
let known_type  nazwa = List.mem_assoc nazwa the_envi.types;;
let known_cons  nazwa = List.mem_assoc nazwa the_envi.conss;;

let known_ident nazwa idents =
                        List.mem_assoc nazwa idents;;

(* pobranie znanej wartosci obiektu *)
let get_value nazwa = List.assoc nazwa the_envi.values;;
let get_type  nazwa = List.assoc nazwa the_envi.types;;
let get_cons  nazwa = List.assoc nazwa the_envi.conss;;

let get_ident nazwa idents =
                      List.assoc nazwa idents;;

let add_value_to_envi n v t =
  if known_value n || known_cons n then
    raise (Ident_error_pure ("Redefinition of value",n))
  else
    the_envi.values <- (n, (v, t, false)) :: the_envi.values;;

let add_type_to_envi n t =
  if known_type n then
    raise (Ident_error_pure ("Redefinition of type",n))
  else
    the_envi.types <- (n, t) :: the_envi.types;;

let get_fun_alg_type name =
  List.exists (fun n -> n = name) the_envi.fun_alg_types;;

let set_fun_alg_type name =
  if not (get_fun_alg_type name) then
    the_envi.fun_alg_types <- name :: the_envi.fun_alg_types;;

let add_cons_to_envi name numer typ =
  if known_value name || known_cons name then
    raise (Ident_error_pure ("Redefinition of cons",name))
  else
    let new_cons = (name, numer, typ) in
      the_envi.conss <- (name, new_cons) :: the_envi.conss;;

let add_ident_to_envi n t =
  if known_value n || known_cons n then
    raise (Ident_error_pure ("Redefinition of ident",n))
  else
    let i = List.length the_envi.idents in
    let new_ident = (n, i, t) in
      the_envi.idents <- (n, new_ident) :: the_envi.idents;
      new_ident;;

(* ========================================================================== *)
(* ========================================================================== *)

let rec str_of_type t_in =
  match t_in with
  | Type_prim t ->
    Primitives.str_of_type t
  | Type_prod tl ->
    if tl = [] then
      "unit"
    else
      "(" ^ (String.concat " * " (List.map str_of_type tl)) ^ ")"
  | Type_fun (a,b) ->
      "(" ^ (str_of_type a) ^ " => " ^ (str_of_type b) ^ ")"
  | Type_alg (n, a, _) ->
      n ^ " [" ^ (String.concat " , " (List.map str_of_type a)) ^ "]"
  | Type_poly_alg (n, _) ->
      n
  | Type_ident s -> s;;
  
let str_of_poly_args args =  
    if args = [] then
      ""
    else
      " [" ^ (String.concat " , " args) ^ "]";;
      
let str_of_sem_error t = t;;

let str_of_ident_error (t, i, g) =
  t ^ ": '" ^ i ^ "', in " ^ g ^ ".";;

let str_of_stype_error (t, y, g) =
  t ^ ":\n" ^ (str_of_type y) ^ "\nin " ^ g ^ ".";;

let str_of_type_error (t, y, p, g) =
  t ^ ":\n" ^ (str_of_type y) ^ "\n  !=\n" ^ (str_of_type p) ^ "\nin " ^ g ^ ".";;


(* ========================================================================== *)
(* ========================================================================== *)

let rec is_fun_type typ =
  match typ with
  | Type_prim     _  -> false
  | Type_prod     tl -> List.exists is_fun_type tl
  | Type_fun      (_, _) -> true
  | Type_alg      (name, ptypes, _) -> get_fun_alg_type name || (List.exists is_fun_type ptypes)
  | Type_poly_alg (name, _) -> get_fun_alg_type name
  | Type_ident    _ -> false;;

(* zamienia 'a list option w 'a list *)
let simply_option list_option =
  match list_option with
  | None -> []
  | Some lis -> lis;;

let rec rec_reduct_type assoc typ =
  let reduct = rec_reduct_type assoc in
    match typ with
    | Type_prim _ -> typ
    | Type_prod tl -> Type_prod (List.map reduct tl)
    | Type_fun (a, b) -> Type_fun (reduct a, reduct b)
    | Type_alg (n, a, c) -> Type_alg (n, List.map reduct a, c)
    | Type_poly_alg (n, c) -> assert false
    | Type_ident n ->
      assert (List.mem_assoc n assoc);
      List.assoc n assoc;;

(* applikuje do typu polimorficznego, konkretne wartosci *)
let reduct_type (args, typ) appl =
  if List.length args <> List.length appl then
    raise (SType_error_pure ("Wrong number of poly-args in application to type",typ))
  else
    match typ with
    | Type_poly_alg (n, c) -> Type_alg (n, appl, c)
    | _ ->
      let assoc = List.combine args appl in
      rec_reduct_type assoc typ;;
  
(* trans type_cons *)
let rec trans_type typ_idents typ =
  let trans = trans_type typ_idents in
  match typ with
  | Syntax_tree.Type_inny (n, a) ->
    if known_type n then
      let the_appl = List.map trans (simply_option a) in
        reduct_type (get_type n) the_appl
    else
    if List.mem n typ_idents then
      if a = None then
        Type_ident n
      else
        raise (Ident_error_pure ("Poly application to ident",n))
    else
      raise (Ident_error_pure ("Unknown type name",n))

  | Syntax_tree.Type_fun (a, b) ->
      Type_fun (trans a, trans b)

  | Syntax_tree.Type_prod t_list -> 
    let subtypes = List.map trans t_list in
      if List.length subtypes = 1 then
        List.hd subtypes
      else
        Type_prod subtypes;;

(* ========================================================================== *)

(* trans denote, z dorzuceniem identyfikatorow do envi *)
let rec trans_den den typ_in =
  match den with
  | Syntax_tree.Den_none    -> Den_none
  | Syntax_tree.Den_unit    -> 
    if typ_in = unit_type then
      Den_unit
    else
      raise (SType_error_pure ("Unit denoting with non-unit type",typ_in))
  | Syntax_tree.Den_arg n   -> Den_ident (add_ident_to_envi n typ_in)
  | Syntax_tree.Den_prod dl -> 
    if List.length dl = 1 then
      trans_den (List.hd dl) typ_in
    else
      match typ_in with
      | Type_prod tl ->
        if List.length tl = List.length dl then
	  Den_prod (List.map2 trans_den dl tl)
	else
          raise (SType_error_pure ("Product denoting with wrong number of arguments",typ_in))
      | _ -> raise (SType_error_pure ("Product denoting of non-produkt type",typ_in));;

(* dorzuca identyfikatory w denote do act_idents *)
let rec accumulate_idents val_ids den =
  match den with
  | Den_none -> val_ids
  | Den_unit -> val_ids
  | Den_ident i ->
    let (n, _, _) = i in
      if List.mem_assoc n val_ids then
        raise (Ident_error_pure ("Duplicate identifier",n))
      else
        (n, i) :: val_ids
  | Den_prod dl ->
    List.fold_left accumulate_idents val_ids dl;;

(* ========================================================================== *)
(* ========================================================================== *)

let get_oper_equal (e1, t1) op (e2, t2) =
  if t1 = t2 then
    if is_fun_type t1 then
      raise (SType_error_pure ("Comparing fun-type",t1))
    else
      (Expr_oper_prim (e1, op, e2), bool_type)
  else
    raise (Type_error_pure ("Different compare subtypes",t1,t2));;

let get_oper_scolon (e1, t1) op (e2, t2) =
  if t1 <> unit_type then
    raise (SType_error_pure ("Non-unit type of pre-scolon",t1))
  else
    (Expr_oper_prim (e1, op, e2), t2);;

let get_oper_ge (e1, t1) op (e2, t2) =
  if t1 = t2 then
    match t1 with
    | Type_fun _ ->
      raise (SType_error_pure ("Comparing fun-type",t1))
    | _ -> (Expr_oper_prim (e1, op, e2), bool_type)
  else
    raise (Type_error_pure ("Different compare subtypes",t1,t2));;

let get_oper_le (e1, t1) op (e2, t2) =
  if t1 = t2 then
    match t1 with
    | Type_fun _ ->
      raise (SType_error_pure ("Comparing fun-type",t1))
    | _ -> (Expr_oper_prim (e1, op, e2), bool_type)
  else
    raise (Type_error_pure ("Different compare subtypes",t1,t2));;

let get_oper_geq (e1, t1) op (e2, t2) =
  if t1 = t2 then
    match t1 with
    | Type_fun _ ->
      raise (SType_error_pure ("Comparing fun-type",t1))
    | _ -> (Expr_oper_prim (e1, op, e2), bool_type)
  else
    raise (Type_error_pure ("Different compare subtypes",t1,t2));;

let get_oper_leq (e1, t1) op (e2, t2) =
  if t1 = t2 then
    match t1 with
    | Type_fun _ ->
      raise (SType_error_pure ("Comparing fun-type",t1))
    | _ -> (Expr_oper_prim (e1, op, e2), bool_type)
  else
    raise (Type_error_pure ("Different compare subtypes",t1,t2));;

let get_oper_neq (e1, t1) op (e2, t2) =
  if t1 = t2 then
    match t1 with
    | Type_fun _ ->
      raise (SType_error_pure ("Comparing fun-type",t1))
    | _ -> (Expr_oper_prim (e1, op, e2), bool_type)
  else
    raise (Type_error_pure ("Different compare subtypes",t1,t2));;

let get_oper_or (e1, t1) op (e2, t2) =
  if t1 <> bool_type then
    raise (SType_error_pure ("Logical operation over non-boolean type",t1))
  else
  if t2 <> bool_type then
    raise (SType_error_pure ("Logical operation over non-boolean type",t2))
  else
    (Expr_oper_prim (e1, op, e2), bool_type);;
    
let get_oper_xor (e1, t1) op (e2, t2) =
  if t1 <> bool_type then
    raise (SType_error_pure ("Logical operation over non-boolean type",t1))
  else
  if t2 <> bool_type then
    raise (SType_error_pure ("Logical operation over non-boolean type",t2))
  else
    (Expr_oper_prim (e1, op, e2), bool_type);;

let get_oper_and (e1, t1) op (e2, t2) =
  if t1 <> bool_type then
    raise (SType_error_pure ("Logical operation over non-boolean type",t1))
  else
  if t2 <> bool_type then
    raise (SType_error_pure ("Logical operation over non-boolean type",t2))
  else
    (Expr_oper_prim (e1, op, e2), bool_type);;
    
let get_oper_add (e1, t1) op (e2, t2) =
  match (t1, t2) with
  | (Type_prim (Primitives.TP_int ), Type_prim (Primitives.TP_int )) ->
    (Expr_oper_prim (e1, op, e2), int_type)
  | (Type_prim (Primitives.TP_rat ), Type_prim (Primitives.TP_rat )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | (Type_prim (Primitives.TP_int ), Type_prim (Primitives.TP_rat )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | (Type_prim (Primitives.TP_rat ), Type_prim (Primitives.TP_int )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | _ -> raise (Type_error_pure ("Arithmetic operation over non-numeric type",t1,t2));;

let get_oper_sub (e1, t1) op (e2, t2) =
  match (t1, t2) with
  | (Type_prim (Primitives.TP_int ), Type_prim (Primitives.TP_int )) ->
    (Expr_oper_prim (e1, op, e2), int_type)
  | (Type_prim (Primitives.TP_rat ), Type_prim (Primitives.TP_rat )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | (Type_prim (Primitives.TP_int ), Type_prim (Primitives.TP_rat )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | (Type_prim (Primitives.TP_rat ), Type_prim (Primitives.TP_int )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | _ -> raise (Type_error_pure ("Arithmetic operation over non-numeric type",t1,t2));;    

let get_oper_mul (e1, t1) op (e2, t2) =
  match (t1, t2) with
  | (Type_prim (Primitives.TP_int ), Type_prim (Primitives.TP_int )) ->
    (Expr_oper_prim (e1, op, e2), int_type)
  | (Type_prim (Primitives.TP_rat ), Type_prim (Primitives.TP_rat )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | (Type_prim (Primitives.TP_int ), Type_prim (Primitives.TP_rat )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | (Type_prim (Primitives.TP_rat ), Type_prim (Primitives.TP_int )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | _ -> raise (Type_error_pure ("Arithmetic operation over non-numeric type",t1,t2));;    
  
let get_oper_div (e1, t1) op (e2, t2) =
  match (t1, t2) with
  | (Type_prim (Primitives.TP_int ), Type_prim (Primitives.TP_int )) ->
    (Expr_oper_prim (e1, op, e2), int_type)
  | (Type_prim (Primitives.TP_rat ), Type_prim (Primitives.TP_rat )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | (Type_prim (Primitives.TP_int ), Type_prim (Primitives.TP_rat )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | (Type_prim (Primitives.TP_rat ), Type_prim (Primitives.TP_int )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | _ -> raise (Type_error_pure ("Arithmetic operation over non-numeric type",t1,t2));;      

let get_oper_mod (e1, t1) op (e2, t2) =
  match (t1, t2) with
  | (Type_prim (Primitives.TP_int ), Type_prim (Primitives.TP_int )) ->
    (Expr_oper_prim (e1, op, e2), int_type)
  | (Type_prim (Primitives.TP_rat ), Type_prim (Primitives.TP_rat )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | (Type_prim (Primitives.TP_int ), Type_prim (Primitives.TP_rat )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | (Type_prim (Primitives.TP_rat ), Type_prim (Primitives.TP_int )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | _ -> raise (Type_error_pure ("Arithmetic operation over non-numeric type",t1,t2));;      

let get_oper_pow (e1, t1) op (e2, t2) =
  match (t1, t2) with
  | (Type_prim (Primitives.TP_rat ), Type_prim (Primitives.TP_rat )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | (Type_prim (Primitives.TP_int ), Type_prim (Primitives.TP_int )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | (Type_prim (Primitives.TP_int ), Type_prim (Primitives.TP_rat )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | (Type_prim (Primitives.TP_rat ), Type_prim (Primitives.TP_int )) ->
    (Expr_oper_prim (e1, op, e2), rat_type)
  | (Type_prim (Primitives.TP_str ), Type_prim (Primitives.TP_str )) ->
    (Expr_oper_prim (e1, op, e2), str_type)
  | _ -> raise (Type_error_pure ("Arithmetic operation over non-numeric type",t1,t2));;

(* ========================================================================== *)
(* ========================================================================== *)

(* trans prim *)
let rec get_expr_prim v val_ids typ_ids =
  let t = Primitives.get_type v in
  (Expr_val_prim v, Type_prim t)

(* trans oper *)
and get_expr_oper (e1, op, e2) val_ids typ_ids =
  let the_e1 = get_expr e1 val_ids typ_ids in
  let the_e2 = get_expr e2 val_ids typ_ids in
  match op with
  | Primitives.Op_equal ->
    get_oper_equal the_e1 op the_e2
  | Primitives.Op_scolon ->
    get_oper_scolon the_e1 op the_e2
  | Primitives.Op_ge ->
    get_oper_ge the_e1 op the_e2
  | Primitives.Op_le ->
    get_oper_le the_e1 op the_e2
  | Primitives.Op_geq ->
    get_oper_geq the_e1 op the_e2
  | Primitives.Op_leq ->
    get_oper_leq the_e1 op the_e2
  | Primitives.Op_neq ->
    get_oper_neq the_e1 op the_e2
  | Primitives.Op_or ->
    get_oper_or the_e1 op the_e2
  | Primitives.Op_and ->
    get_oper_and the_e1 op the_e2
  | Primitives.Op_add ->
    get_oper_add the_e1 op the_e2
  | Primitives.Op_xor ->
    get_oper_xor the_e1 op the_e2
  | Primitives.Op_sub ->
    get_oper_sub the_e1 op the_e2
  | Primitives.Op_mul ->
    get_oper_mul the_e1 op the_e2
  | Primitives.Op_div ->
    get_oper_div the_e1 op the_e2
  | Primitives.Op_mod ->
    get_oper_mod the_e1 op the_e2
  | Primitives.Op_pow ->
    get_oper_pow the_e1 op the_e2

(* trans if *)
and get_expr_if (e1, e2, e3) val_ids typ_ids =
  let (w1, t1) = get_expr e1 val_ids typ_ids in
    if t1 <> bool_type then
      raise (Type_error_pure ("Non-boolean condition",t1,bool_type))
    else
  let (w2, t2) = get_expr e2 val_ids typ_ids in
  let (w3, t3) = get_expr e3 val_ids typ_ids in
    if t2 <> t3 then
      raise (Type_error_pure ("Different if-then-else subtypes",t2,t3))
    else
      (Expr_if (w1, w2, w3), t2)

(* trans Cons arg -> expr *)
and  make_den_cons val_ids typ_ids poly_appl (name, den, expr) =
  assert (known_cons name);
  let (_, _, typ) as cons = get_cons name in
  match reduct_type typ poly_appl with
  | Type_fun (the_in, _) ->
    let the_den = trans_den den the_in in
    let new_idents = accumulate_idents val_ids the_den in
    let (the_expr, the_type) = get_expr expr new_idents typ_ids in
      ((cons, the_den, the_expr), the_type)
  | _ -> assert false

(* trans match *)
and get_expr_match (e, m_list) val_ids typ_ids =
  assert (m_list <> []);
  let (the_e, the_typ) = get_expr e val_ids typ_ids in
  let given_conss = List.map (fun (n,_,_) -> n) m_list in
    match the_typ with
    | Type_alg (_, appl, all_conss) -> 
      if List.sort compare all_conss = List.sort compare given_conss then
        let den_conss_types = List.map (make_den_cons val_ids typ_ids appl) m_list in
	let (out_conss, out_types) = List.split den_conss_types in
	if List.for_all (fun x -> x = List.hd out_types) out_types then
	  (Expr_match (the_e, out_conss), List.hd out_types)
        else
	  raise (SType_error_pure ("Different match-expresion subtypes",List.hd out_types))
      else
        raise (SType_error_pure ("Wrong list of constructors", the_typ))
    | _ -> raise (SType_error_pure ("This type should be alg-type", the_typ))

(* trans appl *)
and get_expr_appl (f, x) val_ids typ_ids =
  let (the_f, typ_f) = get_expr f val_ids typ_ids in
  let (the_x, typ_x) = get_expr x val_ids typ_ids in
  match typ_f with
  | Type_fun (a, b) ->
    if a = typ_x then
      (Expr_appl (the_f, the_x), b)
    else
      raise (Type_error_pure ("Wrong input type of applicated function",a,typ_x))
  | _ ->
    raise (SType_error_pure ("Application of non-functional expression",typ_f))


(* trans fun *)
and get_expr_fun (den, t_in, expr) val_ids typ_ids =
  let the_t_in = trans_type typ_ids t_in in
  let the_den = trans_den den the_t_in in
  let new_idents = accumulate_idents val_ids the_den in
  let (the_expr, t_out) = get_expr expr new_idents typ_ids in
    (Expr_fun (the_den, the_expr), Type_fun (the_t_in, t_out))


(* trans name *)
and get_expr_name (name, poly_appl) val_ids typ_ids =
  if known_ident name val_ids then
    if poly_appl = None then
      let (_, _, t) as i = get_ident name val_ids in
        (Expr_ident i, t)
    else
      raise (Ident_error_pure ("Poly-appl to ident", name))
  else
  if known_cons name then
    let (_, _, the_type) as c = get_cons name in
    let fact_types = List.map (trans_type typ_ids) (simply_option poly_appl) in
      (Expr_cons c, reduct_type the_type fact_types)
  else
  if known_value name then
    let (_, the_type, _) = get_value name in
    let fact_types = List.map (trans_type typ_ids) (simply_option poly_appl) in
      (Expr_ref name, reduct_type the_type fact_types)
  else
    raise (Ident_error_pure ("Unknown ident",name))


(* trans prod *)
and get_expr_prod e_list val_ids typ_ids =
  let (exprs, types) = List.split (List.map (fun e -> get_expr e val_ids typ_ids) e_list) in
    if List.length exprs = 1 then
      (List.hd exprs, List.hd types)
    else
      (Expr_prod exprs, Type_prod types)


(* trans letin *)
and get_expr_letin (den, eq_ex, in_ex) val_ids typ_ids =
  let (the_in_expr, t_eq) = get_expr eq_ex val_ids typ_ids in
  let the_den = trans_den den t_eq in
  let new_idents = accumulate_idents val_ids the_den in
  let (the_out_expr, t_out) = get_expr in_ex new_idents typ_ids in
    (Expr_letin (the_den, the_in_expr, the_out_expr), t_out)


(* trans expr *)
and get_expr expr val_ids typ_ids =
  match expr with
  | Syntax_tree.Expr_prim e            -> get_expr_prim e val_ids typ_ids
  | Syntax_tree.Expr_oper (e1, op, e2) -> get_expr_oper (e1, op, e2) val_ids typ_ids
  | Syntax_tree.Expr_if    (w,t,f)     -> get_expr_if    (w,t,f) val_ids typ_ids
  | Syntax_tree.Expr_match (e, m)      -> get_expr_match (e, m) val_ids typ_ids
  | Syntax_tree.Expr_appl  (f, x)      -> get_expr_appl  (f, x) val_ids typ_ids
  | Syntax_tree.Expr_fun   (a, t, e)   -> get_expr_fun (a, t, e) val_ids typ_ids
  | Syntax_tree.Expr_name  (n, a)      -> get_expr_name (n, a) val_ids typ_ids
  | Syntax_tree.Expr_prod  e           -> get_expr_prod  e val_ids typ_ids
  | Syntax_tree.Expr_letin (d,e1,e2)   -> get_expr_letin (d, e1, e2) val_ids typ_ids;;

(* ========================================================================== *)
(* ========================================================================== *)

(* wczytuje definicje konstruktora *)
let krok_cons_def poly_args res_type numer (name, cons_type) =
  let the_type_in = trans_type poly_args cons_type in
  let args_idents = List.map (fun n -> Type_ident n) poly_args in
  let the_type_out = reduct_type (poly_args, res_type) args_idents in
    add_cons_to_envi name numer (poly_args, Type_fun (the_type_in, the_type_out));
    numer + 1;;
    
(* czy konstruktor typu funkcyjnego *)
let is_cons_of_fun_type name =
  assert (known_cons name);
  let (_, _, (_, typ)) = get_cons name in
  match typ with
  | Type_fun (typ_in, _) -> is_fun_type typ_in
  | _ -> assert false;;
  
(* wczytuje definicje typu algebraicznego *)
let krok_alg_def name poly_args cons_list =
  let all_conss = List.map (fun (n, _) -> n) cons_list in
  let alg_type = Type_poly_alg (name, all_conss) in
    add_type_to_envi name (poly_args, alg_type);
    ignore (List.fold_left (krok_cons_def poly_args alg_type) 0 cons_list);
    if List.exists is_cons_of_fun_type all_conss then
      set_fun_alg_type name;;

(* wczytuje definicje wartosci *)
let krok_val_def (name, poly_args, typ, expr) =
  let (the_expr, the_type) = get_expr expr [] poly_args in
    add_value_to_envi name the_expr (poly_args, the_type);  
    match typ with
    | None -> ()
    | Some dec_typ ->
      let the_type_dec = trans_type poly_args dec_typ in
      if the_type_dec = the_type then
        ()
      else
        raise (Type_error_pure ("Type different then declared", the_type, the_type_dec));;

(* wczytuje definicje rekurencyjna *)
let krok_rec_def (n, a, t, e) =
  let new_expr = Syntax_tree.Expr_fun (Syntax_tree.Den_arg n, t, e) in
  let (the_expr, t) = get_expr new_expr [] a in
  match t with
  | Type_fun (t_in, t_out) ->
    if t_in = t_out then
      match t_in with
      | Type_fun _ ->
        add_value_to_envi n (Expr_rec the_expr) (a, t_in)
      | _ ->
        raise (SType_error_pure ("Rec definition of non-fun value", t_in))
    else
      raise (SType_error_pure ("Rec definition with wrong type of body", t))
  | _ -> raise (SType_error_pure ("This should be functional type in rec-def", t));;

(* sprawdza argumenty polimorficzne *)
let krok_args args =
  let czy_duplikat arg =
    if List.length (List.filter (fun a -> a = arg) args) > 1 then
      raise (Ident_error_pure ("Duplicate poly-arg on poly-args list", arg))
    else
      if known_type arg then
        raise (Ident_error_pure ("Poly-arg ident known as type name", arg))
      else
        ()
  in
    List.iter czy_duplikat args;;

(* wczytuje jedna definicje *)
let krok_przetw (name, args, def) =
  try
    krok_args args;
    match def with
    | Syntax_tree.Type_def t -> 
    ( match t with
      | Syntax_tree.Type_simp typ -> 
        add_type_to_envi name (args, trans_type args typ)
      | Syntax_tree.Type_alg  typ ->
        krok_alg_def name args typ
    )
    | Syntax_tree.Val_def (t, e) ->
      krok_val_def (name, args, t, e)
    | Syntax_tree.Rec_def (n, t, e) ->
      krok_rec_def (n, args, t, e)
  with
  | Ident_error_pure (t, i)    -> raise (Ident_error (t, i, name))
  | SType_error_pure (t, y)    -> raise (SType_error (t, y, name))
  |  Type_error_pure (t, y, p) -> raise ( Type_error (t, y, p, name));;


(* zwraca syntax pliku *)
let make_syntax chan_in =
  let lexbuf = Lexing.from_channel chan_in in
    try
      Parser.main Lexer.token lexbuf
    with
    | Parsing.Parse_error -> raise (Primitives.Error (Lexer.make_error lexbuf))

(* inicjuje envi *)
let init _ =
  initialize_envi;;

(* dorzuca tFile do the_envi *)
let add chan_in =
  let syntax = make_syntax chan_in in
  try
    List.iter krok_przetw syntax;
  with
  | Semantic_error s ->
    raise (Primitives.Error (str_of_sem_error s))   
  | Ident_error (t, i, g) ->
    raise (Primitives.Error (str_of_ident_error (t, i, g)))
  | SType_error (t, y, g) ->
    raise (Primitives.Error (str_of_stype_error (t, y, g)))
  |  Type_error (t, y, p, g) ->
    raise (Primitives.Error (str_of_type_error (t, y, p, g)));;

(* zwraca tFile *)
let stop _ =
  try
    if List.mem_assoc "main" the_envi.values then
      let (_, (a, mf_type), _) = List.assoc "main" the_envi.values in
        if a = [] then
          match mf_type with
          | Type_fun (t_in, _ ) ->
            if t_in = unit_type then
              the_envi
            else
	      raise (SType_error ("Wrong 'main' argument type", t_in, "main"))
          | _ -> 
            raise (SType_error ("'main' is not a function", mf_type, "main"))
        else
          raise (SType_error ("Polymorphic 'main' value", mf_type, "main"))
    else
      raise (Semantic_error ("No 'main' function"))
  with
  | Semantic_error s ->
    raise (Primitives.Error (str_of_sem_error s))   
  | Ident_error (t, i, g) ->
    raise (Primitives.Error (str_of_ident_error (t, i, g)))
  | SType_error (t, y, g) ->
    raise (Primitives.Error (str_of_stype_error (t, y, g)))
  |  Type_error (t, y, p, g) ->
    raise (Primitives.Error (str_of_type_error (t, y, p, g)));;