exception Error of string;;
exception Run_time_error of string;;

(* operator prymitywny *)
type tOper_prim =
  | Op_equal
  | Op_scolon
  | Op_ge
  | Op_le
  | Op_geq
  | Op_leq
  | Op_neq
  | Op_or
  | Op_and
  | Op_xor
  | Op_add
  | Op_sub
  | Op_mul
  | Op_div
  | Op_mod
  | Op_pow;;

(* typ prymitywny *)
type tType_prim =
  | TP_unit
  | TP_bool
  | TP_int
  | TP_rat
  | TP_char
  | TP_str;;

(* wartosc prymitywna *)
type tVal_prim =
  | VP_unit
  | VP_bool of bool
  | VP_int of Big_int.big_int
  | VP_rat of Num.num
  | VP_char of char
  | VP_str of string;;

(* funkcja prymitywna *)
type tFun_prim = tVal_prim list -> tVal_prim;;

(* ========================================================================== *)

let act_file_name = ref "";;

let show_full_val_info = ref false;;

let show_result_val = ref false;;

let show_time = ref false;;

(* ========================================================================== *)

let print_error tresc =
  prerr_string ("Error in " ^ !act_file_name ^ ": " ^ tresc ^ "\n");
  exit 0;;

let print_runtime tresc =
  prerr_string ("Raised run-time-error: \"" ^ tresc ^ "\".\n");
  exit 0;;

let str_of_type t =
  (
    match t with
    | TP_unit -> "unit"
    | TP_bool -> "bool"
    | TP_int -> "int"
    | TP_rat -> "rat"
    | TP_char -> "char"
    | TP_str -> "str"
  );;

let str_of_val v =
  match v with
  | VP_unit     -> "()"
  | VP_bool b   -> if b then "true" else "false"
  | VP_int i    -> Big_int.string_of_big_int i
  | VP_rat r    -> Num.string_of_num r
  | VP_char c   -> String.concat "" ["'"; String.make 1 c; "'"]
  | VP_str s -> String.concat "" ["\""; s; "\""];;

(* ========================================================================== *)

let compare v1 v2 =
  match (v1, v2) with
  | (VP_unit, VP_unit) ->
    0
  | (VP_bool b1, VP_bool b2) ->
    compare b1 b2
  | (VP_int i1, VP_int i2) ->
    Big_int.compare_big_int i1 i2
  | (VP_rat r1, VP_rat r2) ->
    Num.compare_num r1 r2
  | (VP_char c1, VP_char c2) ->
    compare c1 c2
  | (VP_str s1, VP_str s2) ->
    String.compare s1 s2
  | _ -> assert false;;


(* ========================================================================== *)

let unit = VP_unit;;
let zero = VP_int (Big_int.big_int_of_int 0);;

let int_of_str s = VP_int (Big_int.big_int_of_string s);;
let rat_of_str s = VP_rat (Num.num_of_string s);;
let char_of_char c = VP_char c;;
let str_of_str s = VP_str s;;

let get_type v =
  match v with
  | VP_unit   -> TP_unit
  | VP_bool _ -> TP_bool
  | VP_int  _ -> TP_int
  | VP_rat  _ -> TP_rat
  | VP_char _ -> TP_char
  | VP_str  _ -> TP_str;;
  
let get_op o =
  match o with
  | ">"  -> Op_ge
  | "<"  -> Op_le
  | ">=" -> Op_geq
  | "<=" -> Op_leq
  | "<>" -> Op_neq
  
  | "||" -> Op_or
  | "&&" -> Op_and
  | "$$" -> Op_xor
  
  | "+"  -> Op_add
  | "-"  -> Op_sub
  | "/"  -> Op_div
  | "%"  -> Op_mod
  | "^"  -> Op_pow
  | _ -> assert false;;
  

(* ========================================================================== *)
(* ========================================================================== *)

let max_rand_res =
  let n2  = Num.num_of_int 2 in
  let n30 = Num.num_of_int 30 in
    Num.pred_num (Num.power_num n2 n30);;

let f_rand _ =
  VP_rat (Num.div_num (Num.num_of_int (Random.int (Num.int_of_num max_rand_res))) max_rand_res);;

(* ========================================================================== *)

let f_floor l =
  match l with
  | [VP_rat r] ->
    VP_int (Num.big_int_of_num (Num.floor_num r))
  | _ -> assert false;;  
  
let f_roi l =
  match l with
  | [VP_int i] ->
    VP_rat (Num.num_of_big_int i)
  | _ -> assert false;;

(* ========================================================================== *)

let f_pbool l =
  match l with
  | [VP_bool b] ->
    let str =
      if b then
        "true"
      else
        "false"
    in
      print_string str;
      VP_unit
  | _ -> assert false;;

let f_pint l =
  match l with
  | [VP_int i] ->
    print_string (Big_int.string_of_big_int i);
    VP_unit
  | _ -> assert false;;

let f_prat l =
  match l with
  | [VP_rat r] ->
    print_string (Num.string_of_num r);
    VP_unit
  | _ -> assert false;;

let f_nl _ = print_newline (); VP_unit;;

let f_pch l =
  match l with
  | [VP_char c] ->
      print_char c;
      VP_unit
  | _ -> assert false;;

let f_pstr l =
  match l with
  | [VP_str s] ->
      print_string s;
      VP_unit
  | _ -> assert false;;

(* ========================================================================== *)

let f_rbool _ =
  let c = input_char stdin in
  match c with
  | 'y' -> VP_bool true
  | 'n' -> VP_bool false
  | _ -> raise (Run_time_error "Wrong input in read_bool");;

let f_rint _ =
try
  VP_int (Big_int.big_int_of_string (read_line ()))
with
| _ -> raise (Run_time_error "Wrong input in read_int");;

let f_rrat _ =
try
  VP_rat (Num.num_of_string (read_line ()))
with
| _ -> raise (Run_time_error "Wrong input in read_rat");;

let f_rchar _ =
try
  VP_char (input_char stdin)
with
| _ -> raise (Run_time_error "Wrong input in read_char");;

let f_rstr _ =
try
  VP_str (read_line ())
with
| _ -> raise (Run_time_error "Wrong input in read_str");;

(* ========================================================================== *)
(* ========================================================================== *)


let all_prim_funs =
[
  ("rand"         , (f_rand , [TP_unit; TP_rat]));  
  ("floor"        , (f_floor, [TP_rat ; TP_int]));    
  ("rat_of_int"   , (f_roi  , [TP_int ; TP_rat]));

  ("print_newline", (f_nl   , [TP_unit; TP_unit]));

  ("print_bool"   , (f_pbool, [TP_bool; TP_unit]));
  ("print_int"    , (f_pint , [TP_int ; TP_unit]));
  ("print_rat"    , (f_prat , [TP_rat ; TP_unit]));
  ("print_char"   , (f_pch  , [TP_char; TP_unit]));
  ("print_str"    , (f_pstr , [TP_str ; TP_unit]));

  ("read_bool"    , (f_rbool, [TP_unit; TP_bool]));
  ("read_char"    , (f_rchar, [TP_unit; TP_char]));
  ("read_int"     , (f_rint , [TP_unit; TP_int ]));
  ("read_rat"     , (f_rrat , [TP_unit; TP_rat ]));
  ("read_str"     , (f_rstr , [TP_unit; TP_str ]))
];;

let all_prim_vals =
[
  ("true" , (VP_bool true , TP_bool));
  ("false", (VP_bool false, TP_bool))
];;

let all_prim_typs =
[
  ("unit", TP_unit);
  ("bool", TP_bool);
  ("int" , TP_int );
  ("rat" , TP_rat );
  ("char", TP_char);
  ("str" , TP_str)
];;