// Compilation:
//  gcc -std=c17 -Wall -Wextra -O2 pancake_blocks.c -o pancake_blocks
// Usage:
//  ./pancake_blocks block_length

#include <inttypes.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>

// Pancake is represented as 8-bit signed number to speed up computations. This
// allow to consider stacks up to 126 pancakes with quard pancake 127. Longer
// blocks cannot be calculated in a reasonable time anyway. If someone would
// like to consider longer blocks, simply change the definitions below.
// See also the function block_string_length.
typedef int8_t pancake_t;
#define PRI_PANCAKE PRId8

// Calculate the number of chars for printing block of length n.
static int block_string_length(int n) {
  if (n < 9)
    return 2 * n + 2;
  else // This works for blocks shorter than 99.
    return 3 * n - 6;
}

// Blocks are always ended with guard pancake.

// Print pancake block including the guard pancake.
//  block - pointer to table containing n + 1 pancakes (including the guard pancake)
//  n     - length of block
static void print_block(pancake_t const *block, int n) {
  for (int i = 0; i <= n; ++i)
    printf("%" PRI_PANCAKE " ", block[i]);
}

// Check if two given pancakes form adjacency.
static bool adjacent(pancake_t x, pancake_t y) {
  return x + 1 == y || x == y + 1;
}

// Apply flip to block.
//  r    - pointer to table, where output block is stored
//  p    - pointer to table, where input block is stored
//  flip - the number of reversed pancakes
//  n    - length of block
static void make_flip(pancake_t *r, pancake_t const *p, int flip, int n) {
  int i;
  for (i = 0; i < flip; ++i)
    r[i] = p[flip - i - 1];
  for (i = flip; i <= n; ++i)
    r[i] = p[i];
}

// Check recursively if block is front-guarded.
//  block - pointer to table containing n + 1 pancakes (including the guard pancake)
//  n     - length of block
//  first - pancake which schould not return to the beginning
static bool is_guarded(pancake_t const *block, int n, pancake_t first) {
  // We consider all possible flips in a loop.
  for (int flip = 2; flip <= n; ++flip) {
    // The following condition checks if the flip is improve, i.e. it checks if
    // the number of adjacencies is increased.
    if (adjacent(block[0], block[flip]) - adjacent(block[flip - 1], block[flip]) == 1) {
      if (block[flip - 1] == first) {
        // If the flip moves the first pancake to the beginning,
        // then the block is not front-guarded.
        return false;
      }
      else {
        // We make the flip and we consider a next flip recursively.
        pancake_t next_block[n + 1];
        make_flip(next_block, block, flip, n);
        if (is_guarded(next_block, n, first) == false) {
          return false;
        }
      }
    }
  }
  return true;
}

// Check recursively if block is sortable without a waste.
//  block    - pointer to table containing n + 1 pancakes (including the guard pancake)
//  n        - length of block
//  improves - the number of improves made so far
static bool is_sortable_without_a_waste(pancake_t const *block, int n, int improves) {
  // If n improves is made and there was no waste, then block is sorted.
  if (improves == n)
    return true;

  // We consider all possible flips in a loop.
  for (int flip = 2; flip <= n; ++flip) {
    // The following condition checks if the flip is improve, i.e. it checks if
    // the number of adjacencies is increased.
    if (adjacent(block[0], block[flip]) - adjacent(block[flip - 1], block[flip]) == 1) {
      // If the flip is improve, we apply it and we consider a next flip recursively.
      pancake_t next_block[n + 1];
      make_flip(next_block, block, flip, n);
      if (is_sortable_without_a_waste(next_block, n, improves + 1))
        return true;
    }
  }
  return false;
}

// Compute reversed block.
//  rev_block - pointer to table, where output block is stored
//  block     - pointer to table, where input block is stored
//  n         - length of block
static void reverse(pancake_t *rev_block, pancake_t const *block, int n) {
  for (int i = 0; i <= n; ++i)
    rev_block[i] = block[n - i];
}

// Compute inversed block (understood as the inverse permutation).
//  inv_block - pointer to table, where output block is stored
//  block     - pointer to table, where input block is stored
//  n         - length of block
static void inverse(pancake_t *inv_block, pancake_t const *block, int n) {
  for (int i = 0; i <= n; ++i)
    inv_block[block[i] - 1] = (pancake_t)(i + 1);
}

// Compute dual block.
//  dual_block - pointer to table, where output block is stored
//  block      - pointer to table, where input block is stored
//  n          - length of block
static void dual(pancake_t *dual_block, pancake_t const *block, int n) {
  for (int i = 0; i <= n; ++i)
    dual_block[n - i] = (pancake_t)(n + 2) - block[i];
}

// Determine and print properties of the given block.
//  block - pointer to table, where block is stored
//  n     - length of block
static void analyze_block(pancake_t const *block, int n) {
  pancake_t inv_block[n + 1], dual_block[n + 1];

  inverse(inv_block, block, n);
  dual(dual_block, block, n);

  print_block(block, n);
  printf("| ");
  print_block(inv_block, n);
  printf("| ");
  print_block(dual_block, n);
  printf("|");

  if (is_sortable_without_a_waste(block, n, 0)) {
    printf(" sortable-without-a-waste");
  }
  else {
    if (is_guarded(block, n, block[0]))
      printf(" front-guarded");
    // To check if block is back-guarded we reverse it
    // and we check if reversed block is front-guarded.
    pancake_t rev_block[n + 1];
    reverse(rev_block, block, n);
    if (is_guarded(rev_block, n, rev_block[0]))
      printf(" back-guarded");
  }
  printf("\n");
}

// Chceck if block p does not contain pancake i in positions 1 to k - 1.
static bool is_different(pancake_t const *p, pancake_t i, int k) {
  for (int j = 1; j < k; ++j)
    if (i == p[j])
      return false;
  return true;
}

// Generate recursively all blocks of the given length.
//  p - pointer to table, where block is stored
//  k - index of considered pancake
//  n - length of block
static void generate_blocks(pancake_t *p, int k, int n) {
  // Initial k elements of the block are fixed.
  if (k == n) {
    // Block is completed.
    analyze_block(p, n);
  }
  else {
    // We consider all possible pancakes in position k such that they are
    // different from pancakes in positions 0 to k - 1 and they do not create
    // an adjacency to pancake in position k - 1. We take care that the last
    // pancake (in position n - 1) does not create an adjacency to the guard
    // pancake n + 1. Note that in C positions are numbered from 0.
    for (int i = 2; i < n || (i == n && k < n - 1); i++) {
      if (!adjacent((pancake_t)i, p[k - 1]) && is_different(p, (pancake_t)i, k)) {
        p[k] = (pancake_t)i;
        generate_blocks(p, k + 1, n);
      }
    }
  }
}

int main(int argc, char *args[]) {
  if (argc == 2) {
    int n = atoi(args[1]);
    if (n > 1) {
      pancake_t block[n + 1];

      // We set 1 as the first pancake of the block.
      block[0] = 1;
      for (int i = 1; i < n; ++i)
        block[i] = 0;
      // We set n + 1 as the guard pancake.
      block[n] = (pancake_t)(n + 1);

      int len = block_string_length(n);
      printf("%-*s| %-*s| %-*s| properties\n",
             len, "block", len, "its inverse", len, "dual block");
      generate_blocks(block, 1, n);
      return 0;
    }
  }
  fprintf(stderr, "Usage:\n%s block_length\n", args[0]);
  return 1;
}
