# -*- coding: utf-8 -*-
"""
Created on Sun May  4 16:42:26 2025

@author: glori
"""

import pandas as pd
import plotly.graph_objects as go

df = pd.read_csv("norway_covid_2020_2024.csv")
df['date'] = pd.to_datetime(df['date'])
df = df[df['date'] <= '2022-12-31']  


milestones = {
    "First Case": df[df["total_cases"] > 0].iloc[0]['date'],
    "First Lockdown": pd.Timestamp("2020-03-12"),
    "First Vaccine Administered": pd.Timestamp("2020-12-27"),
    "Peak of Cases": df[df["new_cases"] == df["new_cases"].max()].iloc[0]['date']
}

one_mil = df[df["people_vaccinated"] >= 1_000_000]
if not one_mil.empty:
    milestones["1 Million Vaccinated"] = one_mil.iloc[0]['date']

ninety_pct = df[df["people_fully_vaccinated"] >= 0.9 * df["population"]]
if not ninety_pct.empty:
    milestones["90% Coverage"] = ninety_pct.iloc[0]['date']

fig = go.Figure()

fig.add_trace(go.Scatter(
    x=df['date'],
    y=df['new_cases'],
    mode='lines',
    name='Daily New Cases',
    line=dict(color='lightgrey', width=2),
    opacity=0.5
))

color_map = {
    "First Case": "red",
    "First Lockdown": "orange",
    "First Vaccine Administered": "blue",
    "Peak of Cases": "purple",
    "1 Million Vaccinated": "green",
    "90% Coverage": "darkgreen"
}

positions = ['top center', 'bottom center'] * len(milestones)

for i, (label, date) in enumerate(milestones.items()):
    y_value = df[df["date"] == date]["new_cases"].values[0] if date in df["date"].values else 1
    fig.add_trace(go.Scatter(
        x=[date],
        y=[y_value],
        mode="markers+text",
        marker=dict(size=10, color=color_map.get(label, 'black')),
        text=[label],
        textposition=positions[i],
        name=label,
        hovertext=f"{label}: {date.strftime('%Y-%m-%d')}",
        textfont=dict(size=10)
    ))

fig.add_shape(
    type="line",
    x0=min(milestones.values()),
    x1=max(milestones.values()),
    y0=1,
    y1=1,
    line=dict(color="LightSkyBlue", width=4),
    xref='x',
    yref='y'
)

fig.update_layout(
    title="COVID-19 Key Milestones in Norway with Daily New Cases",
    xaxis_title="Date",
    yaxis_title="Daily New Cases",
    template="plotly_white",
    showlegend=False,
    height=400
)

fig.write_html("norway_milestones_with_cases.html")

