import pandas as pd
import matplotlib.pyplot as plt
import matplotlib.ticker as mticker
import seaborn as sns

def plot_cases_vs_deaths(df, save_path):
    # Apply Seaborn theme
    sns.set_theme(style="whitegrid")

    # Ensure 'date' column is datetime
    df['date'] = pd.to_datetime(df['date'])

    # Add a 2024 data cut-off: filter data to include only dates before 2024
    df = df[df['date'] < '2024-01-01']

    # Filter non-zero values
    nonzero_cases = df[df['new_cases'] > 0]
    nonzero_deaths = df[df['new_deaths'] > 0]

    # Create figure and axes with adjusted figsize for a taller plot
    # Original figsize was (12, 6). Increased height by 33% (from 6 to 8)
    fig, ax1 = plt.subplots(figsize=(12, 8)) # Adjusted aspect ratio

    # Define base font size and calculate the 40% increased size
    base_font_size = plt.rcParams['font.size'] # Get default font size
    increased_font_size = base_font_size * 1.8

    # Custom y-axis formatter
    def format_y_axis_labels(y, position):
        if y >= 1_000_000:
            return f'{y / 1_000_000:.1f}M'
        elif y >= 1_000:
            return f'{y / 1_000:.0f}k'
        else:
            return f'{y:.0f}'

    # Plot new_cases on ax1
    line1 = ax1.plot(nonzero_cases['date'], nonzero_cases['new_cases'],
                     color='blue', linewidth=2, label='Cases')
    ax1.set_ylabel('Cases', color='blue', fontsize=increased_font_size) # Increased font size
    ax1.tick_params(axis='y', labelcolor='blue', labelsize=increased_font_size * 0.9) # Slightly smaller for ticks, but still increased
    ax1.yaxis.set_major_formatter(mticker.FuncFormatter(format_y_axis_labels))

    # Blue horizontal grid lines for left axis
    ax1.grid(True, which='major', axis='y', color='blue', alpha=0.5, linestyle='-', linewidth=0.5)
    ax1.grid(True, which='minor', axis='y', color='blue', alpha=0.5, linestyle='-', linewidth=0.3)

    # Create second y-axis for new_deaths
    ax2 = ax1.twinx()
    line2 = ax2.plot(nonzero_deaths['date'], nonzero_deaths['new_deaths'],
                     color='red', linewidth=1.5, label='Deaths')
    ax2.set_ylabel('Deaths', color='red', fontsize=increased_font_size) # Increased font size
    ax2.tick_params(axis='y', labelcolor='red', labelsize=increased_font_size * 0.9) # Slightly smaller for ticks, but still increased
    ax2.yaxis.set_major_formatter(mticker.FuncFormatter(format_y_axis_labels))

    # Red horizontal grid lines for right axis
    ax2.grid(True, which='major', axis='y', color='red', alpha=0.5, linestyle='-', linewidth=0.5)
    ax2.grid(True, which='minor', axis='y', color='red', alpha=0.5, linestyle='-', linewidth=0.3)

    # Add grey vertical grid lines for x-axis
    ax1.grid(True, which='major', axis='x', color='gray', alpha=0.5, linestyle='-', linewidth=0.5)
    ax1.grid(True, which='minor', axis='x', color='gray', alpha=0.5, linestyle='-', linewidth=0.3)

    # Title and combined legend
    plt.title('COVID-19 Cases and Deaths Over Time in Germany', fontsize=14 * 1.4) # Increased font size
    lines, labels = ax1.get_legend_handles_labels()
    lines2, labels2 = ax2.get_legend_handles_labels()
    ax2.legend(lines + lines2, labels + labels2, loc='upper left', fontsize=increased_font_size * 0.9) # Increased font size

    # Adjust x-axis tick label size
    ax1.tick_params(axis='x', labelsize=increased_font_size * 0.9)

    fig.tight_layout()
    plt.savefig(save_path)
    plt.show()
