import pandas as pd

def fill_total_and_new_cases(df, total_col='total_cases', new_col='new_cases', date_col='date'):
    """
    Fill missing values in `total_cases` using interpolation and compute `new_cases` as the difference.
    
    Parameters:
        df (pd.DataFrame): The input DataFrame (must include date_col, total_col, new_col)
        total_col (str): Name of the total cumulative column (e.g., 'total_cases')
        new_col (str): Name of the daily change column (e.g., 'new_cases')
        date_col (str): Name of the date column to sort by
    
    Returns:
        pd.DataFrame: DataFrame with two new columns:
            - total_col + '_filled'
            - new_col + '_filled'
    """
    df = df.copy()
    df = df.sort_values(date_col).reset_index(drop=True)
    
    # Copy total column
    total = df[total_col].copy()
    
    # Linear interpolation (fills gaps >1)
    total_filled = total.interpolate(method='linear', limit_direction='both')

    # Fill single NaNs using average of neighbors
    for i in range(1, len(total) - 1):
        if pd.isna(total.iloc[i]) and not pd.isna(total.iloc[i - 1]) and not pd.isna(total.iloc[i + 1]):
            total_filled.iloc[i] = (total.iloc[i - 1] + total.iloc[i + 1]) / 2

    # Add filled total column
    df[f'{total_col}'] = total_filled

    # Compute new cases as the daily difference
    df[f'{new_col}'] = df[f'{total_col}'].diff().fillna(0)
    df[f'{new_col}'] = df[f'{new_col}'].round().astype('Int64')  # keeps NA-compatible type

    print(f"Shape of dataframe after filling NaN's: {df.shape}")


    return df
