import pandas as pd
import plotly.graph_objects as go
import os

# --- 1. CARREGAR E PREPARAR OS DADOS DE DESEMPREGO ---

# Carregar o arquivo de desemprego
df_unemployment_raw = pd.read_csv('../data/unemployment.csv')

# Transformar os dados do formato largo para o formato longo
years = [str(year) for year in range(2010, 2025)]
df_melt = df_unemployment_raw.melt(
    id_vars=['Country Name', 'Country Code', 'Indicator Name', 'Indicator Code'],
    value_vars=years,
    var_name='Year',
    value_name='Value'
)

# Filtrar para os países e o indicador de desemprego
countries = ['Rwanda', 'South Africa', 'Poland']
df_filtered = df_melt[df_melt['Country Name'].isin(countries)]
df_filtered = df_filtered[df_filtered['Indicator Name'].str.contains("Unemployment")]

# Limpar os dados: remover valores nulos e converter o ano para inteiro
df_unemployment = df_filtered.dropna(subset=['Value']).copy()
df_unemployment['Year'] = df_unemployment['Year'].astype(int)

print("Dados de desemprego carregados e preparados com sucesso.")

# --- 2. CRIAR O GRÁFICO ESTÁTICO ---

# Cores para os países
color_map = {
    'Rwanda': 'blue',
    'Poland': 'red',
    'South Africa': 'green'
}

# Criar a figura
fig = go.Figure()

# Adicionar uma linha (trace) para cada país
for country in countries:
    country_data = df_unemployment[df_unemployment['Country Name'] == country]
    fig.add_trace(go.Scatter(
        x=country_data['Year'],
        y=country_data['Value'],
        mode='lines+markers',
        name=country,
        line=dict(color=color_map[country], width=2.5),
        marker=dict(size=5),
        hovertemplate=(
            "<b>" + country + "</b><br>" +
            "Year: %{x}<br>" +
            "Unemployment: %{y:.2f}%" +
            "<extra></extra>"
        )
    ))

# --- 3. CONFIGURAR O LAYOUT E ADICIONAR ANOTAÇÃO ---

# Aplicar o layout
fig.update_layout(
    title=dict(
        text="<b>Unemployment Rate Comparison Over Time</b>",
        x=0.5,
        font=dict(size=26, family="Arial", color="black")
    ),
    xaxis=dict(
        title=dict(text="<b>Year</b>", font=dict(size=20, family="Arial", color="black")),
        tickfont=dict(size=14, family="Arial", color="black"),
        range=[2015, df_unemployment['Year'].max() + 1],
        showgrid=True,
        gridcolor="lightgrey",
        showline=True,
        linecolor='black',
        linewidth=1
    ),
    yaxis=dict(
        title=dict(text="<b>Unemployment Rate (% of total labor force)</b>", font=dict(size=20, family="Arial", color="black")),
        tickfont=dict(size=14, family="Arial", color="black"),
        # --- ALTERAÇÃO AQUI: Aumentando o limite do eixo Y para criar espaço ---
        range=[0, 50], # Aumentado de 1.05 para 1.20
        showgrid=True,
        gridcolor="lightgrey",
        showline=True,
        linecolor='black',
        linewidth=1
    ),
    plot_bgcolor="white",
    margin=dict(l=120, r=50, t=90, b=80),
    # --- ALTERAÇÃO AQUI: Movendo a legenda para o canto superior direito ---
    legend=dict(
        font=dict(size=18, family="Arial", color="black"),
        yanchor="top",
        y=0.98, # Posição perto do topo
        xanchor="right",
        x=0.98  # Posição perto da direita
    )
)

# Adicionar a linha vertical para a pandemia
fig.add_vline(
    x=2020, 
    line_width=2, 
    line_dash="dash", 
    line_color="black",
    annotation_text="Start of COVID-19",
    annotation_position="top left",
    annotation_font_size=12
)


# --- 4. EXIBIR E SALVAR O GRÁFICO ---

if __name__ == "__main__":
    fig.show()
    fig.write_html("../Plots/Economy/unemployment_comparison.html", include_plotlyjs='cdn')
    fig.write_image("../Plots/Economy/unemployment_comparison.png", width=1200, height=800)