fun odd (x : int) = x mod 2 <> 0;
fun even (x : int) = not (odd x);

functor Register16 (Bitfield : BITFIELD) : REGISTER16 = 
struct
	structure Bitfield = Bitfield;
	structure Bit = Bitfield.Bit;
	exception REGISTER16_ERROR of string;
	type flags = {
		C : Bit.bit option,
		P : Bit.bit option,
		V : Bit.bit option,
		Z : Bit.bit option,
		S : Bit.bit option
	};
	fun getC (f : flags) = #C f;
	fun getP (f : flags) = #P f;
	fun getV (f : flags) = #V f;
	fun getZ (f : flags) = #Z f;
	fun getS (f : flags) = #S f;

	val noFlags : flags = {
		C = NONE,
		P = NONE,
		V = NONE,
		Z = NONE,
		S = NONE
	}

	type reg16 = Bitfield.bitfield;

	val maxSigned  = 127;
	val maxUnsigned  = 255;
	val minSigned = ~1216;
	val minUnsigned  = 0;

	fun setBit (n : int) (r : reg16) : reg16 = Bitfield.setBit n r;
	fun resetBit (n : int) (r : reg16) : reg16  = Bitfield.resetBit n r;
	fun negateBit (n : int) (r : reg16) : reg16  = Bitfield.negateBit n r;

	fun isSet (n : int) (r : reg16) = Bitfield.isSet n r;
	fun isNotSet (n : int) (r : reg16) = Bitfield.isNotSet n r;

	fun asSigned (r : reg16) = Bitfield.asSigned r;
	fun asUnsigned (r : reg16) = Bitfield.asUnsigned r;

	fun fromInt (i : int) : reg16 = Bitfield.fromInt i 16;
	val new  : reg16 = Bitfield.new 16;
	val zero = new;

	fun getBit (n : int) (r : reg16) : Bit.bit = Bitfield.getBit n r;
	fun putBit (n : int) (b : Bit.bit) (r : reg16) : reg16 = Bitfield.putBit n b r;

	fun putC (b : Bit.bit option) (f : flags) : flags =
		{
			C = b,
			P = #P f,
			V = #V f,
			Z = #Z f,
			S = #S f
		}
	;
	fun putS (b : Bit.bit option) (f : flags) : flags =
		{
			C = #C f,
			P = #P f,
			V = #V f,
			Z = #Z f,
			S = b
		}
	;

	fun putZ (b : Bit.bit option) (f : flags) : flags =
		{
			C = #C f,
			P = #P f,
			V = #V f,
			Z = b,
			S = #S f
		}
	;

	fun putP (b : Bit.bit option) (f : flags) : flags =
		{
			C = #C f,
			P = b,
			V = #V f,
			Z = #Z f,
			S = #S f
		}
	;

	fun putV (b : Bit.bit option) (f : flags) : flags =
		{
			C = #C f,
			P = #P f,
			V = b,
			Z = #Z f,
			S = #S f
		}
	;

	val setC = putC (SOME(Bit.one));
	val resetC = putC (SOME(Bit.one));
	fun setCbool (b : bool) = putC (SOME(Bit.fromBool b));
	fun setPbool (b : bool) = putP (SOME(Bit.fromBool b));
	fun setZbool (b : bool) = putZ (SOME(Bit.fromBool b));
	fun setSbool (b : bool) = putS (SOME(Bit.fromBool b));
	fun setVbool (b : bool) = putV (SOME(Bit.fromBool b));

	fun setNormalFlags (r : reg16) : flags =
		let
			val v1 = setSbool (isSet 7 r) noFlags;
			val v2 = setZbool ((asUnsigned r) = 0) v1;
			val v3 = setPbool (even (Bitfield.countOnes r)) v2;
		in
			v3
		end
	;


	fun logop2 (r1 : reg16, r2 : reg16) (oper : Bitfield.bitfield * Bitfield.bitfield -> Bitfield.bitfield) : (reg16 * flags) =
		let
			val v : reg16 = oper (r1, r2);
			val f = setNormalFlags v
		in
			(v, f)
		end
	;

	fun logop1 (r: reg16) (oper : Bitfield.bitfield -> Bitfield.bitfield) : (reg16 * flags) =
		let
			val v : reg16 = oper r;
			val f = setNormalFlags v
		in
			(v, f)
		end
	;

	fun andBits (r1 : reg16, r2 : reg16) : (reg16 * flags) = logop2 (r1, r2) Bitfield.andBits;
	fun xorBits (r1 : reg16, r2 : reg16) : (reg16 * flags) = logop2 (r1, r2) Bitfield.xorBits;
	fun orBits (r1 : reg16, r2 : reg16) : (reg16 * flags) = logop2 (r1, r2) Bitfield.orBits;

	fun isOverflow (carries : Bitfield.bitfield) = 
		Bit.asBool (Bit.xorBits (Bitfield.getBit 6 carries, Bitfield.getBit 7 carries))
	;
	fun isCarry (carries : Bitfield.bitfield) = Bitfield.isSet 7 carries;
	fun add (carry : Bit.bit, r1 : reg16, r2 : reg16) : (reg16 * flags) =
		let
			val (v, c) = Bitfield.addBits (carry, r1, r2)
		in
			(v, ((setVbool (isOverflow c) (setCbool (isCarry c) (setNormalFlags v)))))
		end
	;

	fun subtract (borrow : Bit.bit, r1 : reg16, r2 : reg16) : (reg16 * flags) =
		let
			val (v, b) = Bitfield.subtractBits (borrow, r1, r2)
		in
			(v, ((setVbool (isOverflow b) (setCbool (not (isCarry b)) (setNormalFlags v)))))
		end
	;
	
	fun negateBits (r : reg16) : (reg16 * flags) = logop1 r Bitfield.negateBits;
	fun u2Bits (r : reg16) : (reg16 * flags) = logop1 r Bitfield.u2Bits;

	fun compare (r1 : reg16, r2 : reg16) : flags =
		let
			val (v, f) = subtract (Bit.zero, r1, r2);
		in
			f
		end
	;

	fun rlBits (newbit : Bit.bit) (r : reg16) : (reg16 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 7 r;
			val r1 = Bitfield.append (r ,Bitfield.fromBit newbit);
			val r2 = Bitfield.subField 0 16 (Bitfield.rolBits 1 r1);
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun rrBits (newbit : Bit.bit) (r : reg16) : (reg16 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 0 r;
			val r1 = Bitfield.append (r, Bitfield.fromBit newbit);
			val r2 = Bitfield.subField 0 16 (Bitfield.rorBits 1 r1);
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun rlcBits (r : reg16) : (reg16 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 7 r;
			val r2 = Bitfield.rolBits 1 r;
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun rrcBits (r : reg16) : (reg16 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 0 r;
			val r2 = Bitfield.rorBits 1 r;
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun slaBits (r : reg16) : (reg16 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 7 r;
			val r1 = Bitfield.subField 0 7 r;
			val r2 = Bitfield.append (Bitfield.fromBit Bit.zero, r1);
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun sllBits (r : reg16) : (reg16 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 7 r;
			val r1 = Bitfield.subField 0 7 r;
			val r2 = Bitfield.append (Bitfield.fromBit Bit.one, r1);
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun sraBits (r : reg16) : (reg16 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 0 r;
			val r1 = Bitfield.subField 1 7 r;
			val r2 = Bitfield.append (r1 ,Bitfield.subField 7 1 r);
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun srlBits (r : reg16) : (reg16 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 0 r;
			val r1 = Bitfield.subField 1 7 r;
			val r2 = Bitfield.append (r1, Bitfield.fromBit Bit.zero);
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun halves (r : reg16) : (Bitfield.bitfield * Bitfield.bitfield) =
		(Bitfield.subField 0 8 r, Bitfield.subField 8 8 r)
	;
	fun fromHalves (bf1 : Bitfield.bitfield , bf2 : Bitfield.bitfield) : reg16 =
		if (Bitfield.width bf1 <>8) orelse (Bitfield.width bf2 <> 8) then
			raise REGISTER16_ERROR("Trying to make reg16 from wrong bitfields")
		else
			Bitfield.append (bf1, bf2)
	;
	fun asString (r : reg16) = (Bitfield.asString r) ^ "(" ^ (intPadded 5 (asUnsigned r)) ^ ")";
	fun fromBitfield (bf : Bitfield.bitfield) : reg16 =
		if Bitfield.width bf <> 16 then
			raise REGISTER16_ERROR("Trying to make reg16 from wrong bitfield")
		else
			bf
	;
	fun asBitfield (r : reg16) : Bitfield.bitfield = r;
end;
