signature BUS = 
sig
	type bus_state;
	type bus_change;
	val initial_state : bus_state;
	val no_bus_change : bus_change;
	val checkSignalConflict : bus_change list -> (bus_state * (bus_change option));
(*	val get : bus_state -> int;
	val set : int ->bus_state;*)
end;

signature DEVICE =
sig
	structure bus : BUS;
	type device_state;
	val initial_state : device_state;
(*	val set_device_id : int -> device_state -> device_state;
*)
	val tick : (bus.bus_state * device_state) -> (bus.bus_state * device_state);
end;

functor MemoryModule (bus1 : BUS_Z80, bit1 : BIT) : DEVICE = 
struct
	structure Bus = bus1;
	structure Bit = bit1;
	type memory = int array;
	
	datatype device_state = DS of (memory * (Bus.bus_state * memory -> device_state * Bus.bus_change));
	fun memory_reading (bs : bus.bus_state, mem : memory) : (device_state * bus.bus_change) =
		let
			val adr = Bitfield.asInt (#Address bs);
			val data = Array.sub (mem, adr);
			val dataf = Bitfield.fromInt data 8;
		in
			if (Bit.isNotSet (#MREQ bs)) andalso (Bit.isNotSet (#RD bs)) then
				Bus.setData dataf Bus.no_bus_change
			else
				memory_waiting (bs, mem)
		end
	and
	fun memory_writing (bs : bus.bus_state, mem : memory) : (device_state * bus.bus_change) =
		let
			val adr = Bitfield.asInt (#Address bs);
			val data = Bitfield.asInt (#Data bs);
		in
			if (Bit.isNotSet (#MREQ bs)) andalso (Bit.isNotSet (#WR bs)) then
				let
					val v = Array.update (mem, adr, data);
				in
					((mem, memory_writing), Bus.no_bus_change)
				end
			else
				memory_waiting (bs, mem)
		end
	and
	fun memory_waiting (bs : bus.bus_state, mem : memory) : (device_state * bus.bus_change) =
		if Bit.isNotSet ( #MREQ bs) then
			if Bit.isNotSet ( #RD bs) then
				memory_reading (bs, mem) 
			else
				if Bit.isNotSet (#WR bs) then
					memory_writing (bs, mem)
				else
					((mem, memory_waiting), Bus.no_bus_change)
					
		else
			((mem, memory_waiting), Bus.no_bus_change)
	;
	val initial_state : device_state = (Array.array (65536, 0), memory_waiting);
end;

signature MOTHERBOARD = 
sig
	type mb_state;
	val initial_state : mb_state;
	val tick : mb_state -> mb_state;
end;


functor Mb3 (
	structure dev1 : DEVICE; 
	structure dev2 : DEVICE; 
	structure dev3 : DEVICE; 
	structure bus : BUS;
	sharing type dev1.bus.bus_state = dev2.bus.bus_state;
	sharing type dev2.bus.bus_state = dev3.bus.bus_state;
	sharing type dev3.bus.bus_state = bus.bus_state;
	
	) : MOTHERBOARD =
struct
	type mb_state = (
		bus.bus_state *
		dev1.device_state * 
		dev2.device_state * 
		dev3.device_state
	);
	val initial_state : mb_state =  (
		bus.initial_state,
		dev1.initial_state,
		dev2.initial_state, 
		dev3.initial_state
	);
	fun tick ((d1, d2, d3, bstate) : mb_state) : mb_state = 
		let
			val (b1, d1')  = dev1.tick (bstate, d1);
			val (b2, d2')  = dev1.tick (bstate, d1);
			val (b3, d3')  = dev1.tick (bstate, d1);
			val l = [b1, b2, b3];
			val (newbus, err) = bus.checkSignalConflict l;
		in
			if isSome(err) then
				raise BUS_CONFLICT(l)
			else
				(newbus, d1', d2', d3')
		end
	;
(*	fun tick ((d1, d2, d3, bstate) : mb_state) = 
		let
			val (b1, d1')  = dev1.tick (bstate, d1);
		in
			let
				val (b2, d2')  = dev2.tick (b1, d2);
			in
				let
					val (b3, d3')  = dev3.tick (b2, d3);
				in
					(d1', d2', d3', b3)
				end
			end
		end
	;
*)
end;

functor CpuZ81 (bus1 : BUS) : DEVICE =
struct
	structure bus = bus1;
	type reg8 = int;
	type reg16 = int;
	type cpu_state = {
		A  : reg8,
		A' : reg8,
		F  : reg8,
		F' : reg8,
		B  : reg8,
		B' : reg8,
		C  : reg8,
		C' : reg8,
		D  : reg8,
		D' : reg8,
		E  : reg8,
		E' : reg8,
		H  : reg8,
		H' : reg8,
		L  : reg8,
		L' : reg8,
		I  : reg8,
		R  : reg8,
		IX : reg16,
		IY : reg16,
		SP : reg16,
		PC : reg16,
		IM : int,	(* Interrupt mode *)
		IE : bool,	(* Interrupts enabled *)
		HLT : bool	(* Halt state *)
	};
        datatype device_state = DS of (
		bus.bus_state -> (bus.bus_state * device_state)
	);
	fun getDS (DS(f : (bus.bus_state -> (bus.bus_state * device_state)))) = f;
	fun makeDS (f : (bus.bus_state -> (bus.bus_state * device_state))) = DS(f);
	fun wait_n_cycles 
		(cnt : int)
		(next_state : device_state)
		(bs : bus.bus_state) : (bus.bus_state * device_state) =
		if cnt <= 1 then
			(getDS next_state) bs
		else
			(bs, DS(wait_n_cycles (cnt-1) next_state))
	;

	fun normal_cpu_state (arg : bus.bus_state) : (bus.bus_state * device_state) =
		if (bus.get arg <1) then
			(bus.set ((bus.get arg)+1), DS(normal_cpu_state))
		else (
			(bus.set ((bus.get arg)+1)) ,
			DS( (wait_n_cycles 3) (DS(normal_cpu_state)))
		)
	;

        val initial_state : device_state = DS(normal_cpu_state);

	fun set_device_id (i:int) (d:device_state) = d;

        fun tick (bs : bus.bus_state, DS(ds)) =
		ds bs
	;


end;

functor BusZ80(B : BITFIELD) : BUS_Z80 = 
struct
	structure Bitfield = B;
	type bus_state = {
		M1     : Bitfield.Bit.bit,
		MREQ   : Bitfield.Bit.bit,
		IOREQ  : Bitfield.Bit.bit,
		RD     : Bitfield.Bit.bit,
		WR     : Bitfield.Bit.bit,
		RFSH   : Bitfield.Bit.bit,
		HALT   : Bitfield.Bit.bit,
		WAIT   : Bitfield.Bit.bit,
		RESET  : Bitfield.Bit.bit,
		INT    : Bitfield.Bit.bit,
		NMI    : Bitfield.Bit.bit,
		BUSRQ  : Bitfield.Bit.bit,
		BUSAK  : Bitfield.Bit.bit,
		Address : Bitfield.bitfield,
		Data   : Bitfield.bitfield
	};
	type bus_change = {
		M1     : Bitfield.Bit.bit option,
		MREQ   : Bitfield.Bit.bit option,
		IOREQ  : Bitfield.Bit.bit option,
		RD     : Bitfield.Bit.bit option,
		WR     : Bitfield.Bit.bit option,
		RFSH   : Bitfield.Bit.bit option,
		HALT   : Bitfield.Bit.bit option,
		WAIT   : Bitfield.Bit.bit option,
		RESET  : Bitfield.Bit.bit option,
		INT    : Bitfield.Bit.bit option,
		NMI    : Bitfield.Bit.bit option,
		BUSRQ  : Bitfield.Bit.bit option,
		BUSAK  : Bitfield.Bit.bit option,
		Address : Bitfield.bitfield option,
		Data   : Bitfield.bitfield option
	};
	val initial_state : bus_state = {
		M1     = Bitfield.Bit.one,
		MREQ   = Bitfield.Bit.one,
		IOREQ  = Bitfield.Bit.one,
		RD     = Bitfield.Bit.one,
		WR     = Bitfield.Bit.one,
		RFSH   = Bitfield.Bit.one,
		HALT   = Bitfield.Bit.one,
		WAIT   = Bitfield.Bit.one,
		RESET  = Bitfield.Bit.one,
		INT    = Bitfield.Bit.one,
		NMI    = Bitfield.Bit.one,
		BUSRQ  = Bitfield.Bit.one,
		BUSAK  = Bitfield.Bit.one,
		Address = Bitfield.fromInt 0 16,
		Data    = Bitfield.fromInt 0 8
	};
	val no_bus_change : bus_change = {
		M1     = NONE,
		MREQ   = NONE,
		IOREQ  = NONE,
		RD     = NONE,
		WR     = NONE,
		RFSH   = NONE,
		HALT   = NONE,
		WAIT   = NONE,
		RESET  = NONE,
		INT    = NONE,
		NMI    = NONE,
		BUSRQ  = NONE,
		BUSAK  = NONE,
		Address = NONE,
		Data    = NONE
	};
end;

structure c1 = CpuZ81(Bus);
structure c2 = CpuZ81(Bus);
structure c3 = CpuZ81(Bus);

fun go (n : int) (d : Bus.bus_state, c : c1.device_state) = 
let 
	val (d1,c1) =  c1.tick (d, c);
in
	if (n = 0) then
		[]
	else
		(Bus.get d1) :: go (n-1) (d1,c1)
end;

(*structure komp = Mb3(c1,c2,c3,Bus);*)


