functor BusZ80(
	structure Bitfield : BITFIELD;
	structure Register8 : REGISTER8;
	structure Register16 : REGISTER16;
	sharing Register8.Bitfield = Register16.Bitfield;
	sharing Register8.Bitfield = Bitfield;
) : BUS_Z80 = 
struct
	structure Bitfield = Bitfield;
	structure Bit = Bitfield.Bit;
	structure Register8 = Register8;
	structure Register16 = Register16;
	datatype cycle_state = ASCENDING | HIGH | DESCENDING | LOW;
	type bus_state = {
		Cycle  : cycle_state,
		M1     : Bit.bit,
		MREQ   : Bit.bit,
		IORQ  : Bit.bit,
		RD     : Bit.bit,
		WR     : Bit.bit,
		RFSH   : Bit.bit,
		HALT   : Bit.bit,
		WAIT   : Bit.bit,
		RESET  : Bit.bit,
		INT    : Bit.bit,
		NMI    : Bit.bit,
		BUSRQ  : Bit.bit,
		BUSAK  : Bit.bit,
		Address : Register16.reg16,
		Data   : Register8.reg8
	};
	type bus_change = {
		M1     : Bit.bit option,
		MREQ   : Bit.bit option,
		IORQ  : Bit.bit option,
		RD     : Bit.bit option,
		WR     : Bit.bit option,
		RFSH   : Bit.bit option,
		HALT   : Bit.bit option,
		WAIT   : Bit.bit option,
		RESET  : Bit.bit option,
		INT    : Bit.bit option,
		NMI    : Bit.bit option,
		BUSRQ  : Bit.bit option,
		BUSAK  : Bit.bit option,
		Address : Register16.reg16 option,
		Data   : Register8.reg8 option
	};


	fun next_cycle_state ASCENDING = HIGH
	  | next_cycle_state HIGH = DESCENDING
	  | next_cycle_state DESCENDING = LOW
	  | next_cycle_state LOW = ASCENDING
	;

	val initial_state : bus_state = {
		Cycle  = ASCENDING,
		M1     = Bit.one,
		MREQ   = Bit.one,
		IORQ  = Bit.one,
		RD     = Bit.one,
		WR     = Bit.one,
		RFSH   = Bit.one,
		HALT   = Bit.one,
		WAIT   = Bit.zero,
		RESET  = Bit.one,
		INT    = Bit.one,
		NMI    = Bit.one,
		BUSRQ  = Bit.one,
		BUSAK  = Bit.one,
		Address = Register16.fromInt 65535,
		Data    = Register8.fromInt 255
	};
	val no_signals_state = initial_state; (* FIXME *)
	val no_bus_change : bus_change = {
		M1     = NONE,
		MREQ   = NONE,
		IORQ  = NONE,
		RD     = NONE,
		WR     = NONE,
		RFSH   = NONE,
		HALT   = NONE,
		WAIT   = NONE,
		RESET  = NONE,
		INT    = NONE,
		NMI    = NONE,
		BUSRQ  = NONE,
		BUSAK  = NONE,
		Address = NONE,
		Data    = NONE
	};
	fun putM1 (b : Bit.bit) (bc : bus_change) : bus_change = {
		M1     = SOME b,
		MREQ   = #MREQ bc,
		IORQ  = #IORQ bc,
		RD     = #RD bc,
		WR     = #WR bc,
		RFSH   = #RFSH bc,
		HALT   = #HALT bc,
		WAIT   = #WAIT bc,
		RESET  = #RESET bc,
		INT    = #INT bc,
		NMI    = #NMI bc,
		BUSRQ  = #BUSRQ bc,
		BUSAK  = #BUSAK bc,
		Address = #Address bc,
		Data    = #Data bc
	};
	fun setM1bool b = putM1 (Bit.fromBool b);
	val setM1 = putM1 Bit.one;
	val resetM1 = putM1 Bit.zero;

	fun putMREQ (b : Bit.bit) (bc : bus_change) : bus_change = {
		M1     = #M1 bc,
		MREQ   = SOME b,
		IORQ  = #IORQ bc,
		RD     = #RD bc,
		WR     = #WR bc,
		RFSH   = #RFSH bc,
		HALT   = #HALT bc,
		WAIT   = #WAIT bc,
		RESET  = #RESET bc,
		INT    = #INT bc,
		NMI    = #NMI bc,
		BUSRQ  = #BUSRQ bc,
		BUSAK  = #BUSAK bc,
		Address = #Address bc,
		Data    = #Data bc
	};
	fun setMREQbool b = putMREQ (Bit.fromBool b);
	val setMREQ = putMREQ Bit.one;
	val resetMREQ = putMREQ Bit.zero;

	fun putIORQ (b : Bit.bit) (bc : bus_change) : bus_change = {
		M1     = #M1 bc,
		MREQ   = #MREQ bc,
		IORQ  = SOME b,
		RD     = #RD bc,
		WR     = #WR bc,
		RFSH   = #RFSH bc,
		HALT   = #HALT bc,
		WAIT   = #WAIT bc,
		RESET  = #RESET bc,
		INT    = #INT bc,
		NMI    = #NMI bc,
		BUSRQ  = #BUSRQ bc,
		BUSAK  = #BUSAK bc,
		Address = #Address bc,
		Data    = #Data bc
	};
	fun setIORQbool b = putIORQ (Bit.fromBool b);
	val setIORQ = putIORQ Bit.one;
	val resetIORQ = putIORQ Bit.zero;

	fun putRD (b : Bit.bit) (bc : bus_change) : bus_change = {
		M1     = #M1 bc,
		MREQ   = #MREQ bc,
		IORQ  = #IORQ bc,
		RD     = SOME b,
		WR     = #WR bc,
		RFSH   = #RFSH bc,
		HALT   = #HALT bc,
		WAIT   = #WAIT bc,
		RESET  = #RESET bc,
		INT    = #INT bc,
		NMI    = #NMI bc,
		BUSRQ  = #BUSRQ bc,
		BUSAK  = #BUSAK bc,
		Address = #Address bc,
		Data    = #Data bc
	};
	fun setRDbool b = putRD (Bit.fromBool b);
	val setRD = putRD Bit.one;
	val resetRD = putRD Bit.zero;

	fun putWR (b : Bit.bit) (bc : bus_change) : bus_change = {
		M1     = #M1 bc,
		MREQ   = #MREQ bc,
		IORQ  = #IORQ bc,
		RD     = #RD bc,
		WR     = SOME b,
		RFSH   = #RFSH bc,
		HALT   = #HALT bc,
		WAIT   = #WAIT bc,
		RESET  = #RESET bc,
		INT    = #INT bc,
		NMI    = #NMI bc,
		BUSRQ  = #BUSRQ bc,
		BUSAK  = #BUSAK bc,
		Address = #Address bc,
		Data    = #Data bc
	};
	fun setWRbool b = putWR (Bit.fromBool b);
	val setWR = putWR Bit.one;
	val resetWR = putWR Bit.zero;

	fun putRFSH (b : Bit.bit) (bc : bus_change) : bus_change = {
		M1     = #M1 bc,
		MREQ   = #MREQ bc,
		IORQ  = #IORQ bc,
		RD     = #RD bc,
		WR     = #WR bc,
		RFSH   = SOME b,
		HALT   = #HALT bc,
		WAIT   = #WAIT bc,
		RESET  = #RESET bc,
		INT    = #INT bc,
		NMI    = #NMI bc,
		BUSRQ  = #BUSRQ bc,
		BUSAK  = #BUSAK bc,
		Address = #Address bc,
		Data    = #Data bc
	};
	fun setRFSHbool b = putRFSH (Bit.fromBool b);
	val setRFSH = putRFSH Bit.one;
	val resetRFSH = putRFSH Bit.zero;

	fun putHALT (b : Bit.bit) (bc : bus_change) : bus_change = {
		M1     = #M1 bc,
		MREQ   = #MREQ bc,
		IORQ  = #IORQ bc,
		RD     = #RD bc,
		WR     = #WR bc,
		RFSH   = #RFSH bc,
		HALT   = SOME b,
		WAIT   = #WAIT bc,
		RESET  = #RESET bc,
		INT    = #INT bc,
		NMI    = #NMI bc,
		BUSRQ  = #BUSRQ bc,
		BUSAK  = #BUSAK bc,
		Address = #Address bc,
		Data    = #Data bc
	};
	fun setHALTbool b = putHALT (Bit.fromBool b);
	val setHALT = putHALT Bit.one;
	val resetHALT = putHALT Bit.zero;

	fun putWAIT (b : Bit.bit) (bc : bus_change) : bus_change = {
		M1     = #M1 bc,
		MREQ   = #MREQ bc,
		IORQ  = #IORQ bc,
		RD     = #RD bc,
		WR     = #WR bc,
		RFSH   = #RFSH bc,
		HALT   = #HALT bc,
		WAIT   = SOME b,
		RESET  = #RESET bc,
		INT    = #INT bc,
		NMI    = #NMI bc,
		BUSRQ  = #BUSRQ bc,
		BUSAK  = #BUSAK bc,
		Address = #Address bc,
		Data    = #Data bc
	};
	fun setWAITbool b = putWAIT (Bit.fromBool b);
	val setWAIT = putWAIT Bit.one;
	val resetWAIT = putWAIT Bit.zero;

	fun putRESET (b : Bit.bit) (bc : bus_change) : bus_change = {
		M1     = #M1 bc,
		MREQ   = #MREQ bc,
		IORQ  = #IORQ bc,
		RD     = #RD bc,
		WR     = #WR bc,
		RFSH   = #RFSH bc,
		HALT   = #HALT bc,
		WAIT   = #WAIT bc,
		RESET  = SOME b,
		INT    = #INT bc,
		NMI    = #NMI bc,
		BUSRQ  = #BUSRQ bc,
		BUSAK  = #BUSAK bc,
		Address = #Address bc,
		Data    = #Data bc
	};
	fun setRESETbool b = putRESET (Bit.fromBool b);
	val setRESET = putRESET Bit.one;
	val resetRESET = putRESET Bit.zero;

	fun putINT (b : Bit.bit) (bc : bus_change) : bus_change = {
		M1     = #M1 bc,
		MREQ   = #MREQ bc,
		IORQ  = #IORQ bc,
		RD     = #RD bc,
		WR     = #WR bc,
		RFSH   = #RFSH bc,
		HALT   = #HALT bc,
		WAIT   = #WAIT bc,
		RESET  = #RESET bc,
		INT    = SOME b,
		NMI    = #NMI bc,
		BUSRQ  = #BUSRQ bc,
		BUSAK  = #BUSAK bc,
		Address = #Address bc,
		Data    = #Data bc
	};
	fun setINTbool b = putINT (Bit.fromBool b);
	val setINT = putINT Bit.one;
	val resetINT = putINT Bit.zero;

	fun putNMI (b : Bit.bit) (bc : bus_change) : bus_change = {
		M1     = #M1 bc,
		MREQ   = #MREQ bc,
		IORQ  = #IORQ bc,
		RD     = #RD bc,
		WR     = #WR bc,
		RFSH   = #RFSH bc,
		HALT   = #HALT bc,
		WAIT   = #WAIT bc,
		RESET  = #RESET bc,
		INT    = #INT bc,
		NMI    = SOME b,
		BUSRQ  = #BUSRQ bc,
		BUSAK  = #BUSAK bc,
		Address = #Address bc,
		Data    = #Data bc
	};
	fun setNMIbool b = putNMI (Bit.fromBool b);
	val setNMI = putNMI Bit.one;
	val resetNMI = putNMI Bit.zero;

	fun putBUSRQ (b : Bit.bit) (bc : bus_change) : bus_change = {
		M1     = #M1 bc,
		MREQ   = #MREQ bc,
		IORQ  = #IORQ bc,
		RD     = #RD bc,
		WR     = #WR bc,
		RFSH   = #RFSH bc,
		HALT   = #HALT bc,
		WAIT   = #WAIT bc,
		RESET  = #RESET bc,
		INT    = #INT bc,
		NMI    = #NMI bc,
		BUSRQ  = SOME b,
		BUSAK  = #BUSAK bc,
		Address = #Address bc,
		Data    = #Data bc
	};
	fun setBUSRQbool b = putBUSRQ (Bit.fromBool b);
	val setBUSRQ = putBUSRQ Bit.one;
	val resetBUSRQ = putBUSRQ Bit.zero;

	fun putBUSAK (b : Bit.bit) (bc : bus_change) : bus_change = {
		M1     = #M1 bc,
		MREQ   = #MREQ bc,
		IORQ  = #IORQ bc,
		RD     = #RD bc,
		WR     = #WR bc,
		RFSH   = #RFSH bc,
		HALT   = #HALT bc,
		WAIT   = #WAIT bc,
		RESET  = #RESET bc,
		INT    = #INT bc,
		NMI    = #NMI bc,
		BUSRQ  = #BUSRQ bc,
		BUSAK  = SOME b,
		Address = #Address bc,
		Data    = #Data bc
	};
	fun setBUSAKbool b = putBUSAK (Bit.fromBool b);
	val setBUSAK = putBUSAK Bit.one;
	val resetBUSAK = putBUSAK Bit.zero;

	fun putAddress (bf : Register16.reg16) (bc : bus_change) : bus_change = {
		M1     = #M1 bc,
		MREQ   = #MREQ bc,
		IORQ  = #IORQ bc,
		RD     = #RD bc,
		WR     = #WR bc,
		RFSH   = #RFSH bc,
		HALT   = #HALT bc,
		WAIT   = #WAIT bc,
		RESET  = #RESET bc,
		INT    = #INT bc,
		NMI    = #NMI bc,
		BUSRQ  = #BUSRQ bc,
		BUSAK  = #BUSAK bc,
		Address = SOME bf,
		Data    = #Data bc
	};
	
	fun putData (bf : Register8.reg8) (bc : bus_change) : bus_change = {
		M1     = #M1 bc,
		MREQ   = #MREQ bc,
		IORQ  = #IORQ bc,
		RD     = #RD bc,
		WR     = #WR bc,
		RFSH   = #RFSH bc,
		HALT   = #HALT bc,
		WAIT   = #WAIT bc,
		RESET  = #RESET bc,
		INT    = #INT bc,
		NMI    = #NMI bc,
		BUSRQ  = #BUSRQ bc,
		BUSAK  = #BUSAK bc,
		Address = #Address bc,
		Data    = SOME bf
	};

	fun checkSignalConflict (bcl : bus_change list, bs : bus_state) : (bus_state * (bus_change option)) =
	(*
		Checks if there are signals clashes and generates from bus
		changes new bus state (including cycle state)
	*)
		let
			fun checkErr (current : ''a option, next : ''a option, errors : Bit.bit option) =
				if isSome(next) andalso isSome(current) then
					if valOf (next) = valOf (current) then
						if isSome (errors) andalso valOf(errors) = Bit.one then 
							SOME Bit.one
						else
							SOME Bit.zero
					else
						SOME Bit.one
				else 
					NONE
			;
			fun eff (current : 'a option, next : 'a option, errors : 'a option) =
				if isSome(next) then next else current
			;
			fun checkErr2 (current : Register16.reg16 option, next : Register16.reg16 option, errors : Register16.reg16 option) =
				if isSome(next) andalso isSome(current) then
					next
				else 
					NONE
			;
			fun checkErr3 (current : Register8.reg8 option, next : Register8.reg8 option, errors : Register8.reg8 option) =
				if isSome(next) andalso isSome(current) then
					next
				else 
					NONE
			;
			fun add (current : bus_change, next : bus_change, errors : bus_change) : (bus_change * bus_change) =
			(
			 {
				M1 = eff (#M1 current, #M1 next, #M1 errors),
				MREQ = eff (#MREQ current, #MREQ next, #MREQ errors),
				IORQ = eff (#IORQ current, #IORQ next, #IORQ errors),
				RD = eff (#RD current, #RD next, #RD errors),
				WR = eff (#WR current, #WR next, #WR errors),
				RFSH = eff (#RFSH current, #RFSH next, #RFSH errors),
				HALT = eff (#HALT current, #HALT next, #HALT errors),
				WAIT = eff (#WAIT current, #WAIT next, #WAIT errors),
				RESET = eff (#RESET current, #RESET next, #RESET errors),
				INT = eff (#INT current, #INT next, #INT errors),
				NMI = eff (#NMI current, #NMI next, #NMI errors),
				BUSRQ = eff (#BUSRQ current, #BUSRQ next, #BUSRQ errors),
				BUSAK = eff (#BUSAK current, #BUSAK next, #BUSAK errors),
				Address = eff (#Address current, #Address next, #Address errors),
				Data = eff (#Data current, #Data next, #Data errors)
			 }	
			,
			 {
				M1 = checkErr (#M1 current, #M1 next, #M1 errors),
				MREQ = checkErr (#MREQ current, #MREQ next, #MREQ errors),
				IORQ = checkErr (#IORQ current, #IORQ next, #IORQ errors),
				RD = checkErr (#RD current, #RD next, #RD errors),
				WR = checkErr (#WR current, #WR next, #WR errors),
				RFSH = checkErr (#RFSH current, #RFSH next, #RFSH errors),
				HALT = checkErr (#HALT current, #HALT next, #HALT errors),
				WAIT = checkErr (#WAIT current, #WAIT next, #WAIT errors),
				RESET = checkErr (#RESET current, #RESET next, #RESET errors),
				INT = checkErr (#INT current, #INT next, #INT errors),
				NMI = checkErr (#NMI current, #NMI next, #NMI errors),
				BUSRQ = checkErr (#BUSRQ current, #BUSRQ next, #BUSRQ errors),
				BUSAK = checkErr (#BUSAK current, #BUSAK next, #BUSAK errors),
				Address = checkErr2 (#Address current, #Address next, #Address errors),
				Data = checkErr3 (#Data current, #Data next, #Data errors)
			 }	
			)	
			;
			fun f [] (state, errors, conflict) = (state, errors, conflict)
			  | f (a::tl) (state, errors, conflict) = 
				let
					val (s, e) = add (state, a, errors)
				in
					f tl (s, e, if e <> errors then true else conflict)
				end
			;
			fun convert (a, b) = if isSome(a) then valOf(a) else b;
			fun conv (bc : bus_change, bs : bus_state) : bus_state = 
			{
				Cycle = next_cycle_state (#Cycle bs),
				M1 = convert (#M1 bc, #M1 no_signals_state),
				MREQ = convert (#MREQ bc, #MREQ no_signals_state),
				IORQ = convert (#IORQ bc, #IORQ no_signals_state),
				RD = convert (#RD bc, #RD no_signals_state),
				WR = convert (#WR bc, #WR no_signals_state),
				RFSH = convert (#RFSH bc, #RFSH no_signals_state),
				HALT = convert (#HALT bc, #HALT no_signals_state),
				WAIT = convert (#WAIT bc, #WAIT no_signals_state),
				RESET = convert (#RESET bc, #RESET no_signals_state),
				INT = convert (#INT bc, #INT no_signals_state),
				NMI = convert (#NMI bc, #NMI no_signals_state),
				BUSRQ = convert (#BUSRQ bc, #BUSRQ no_signals_state),
				BUSAK = convert (#BUSAK bc, #BUSAK no_signals_state),
				Address = convert (#Address bc, #Address no_signals_state),
				Data = convert (#Data bc, #Data no_signals_state)
			}
			;
			val no_errors = no_bus_change;
			val (s, e, c) = f bcl (no_bus_change, no_errors, false);
		in
			(conv (s, bs), if c then SOME e else NONE)
		end
	;
	fun cycleAsString ASCENDING 	= "ASCENDING "
	  | cycleAsString HIGH 		= "HIGH      "
	  | cycleAsString DESCENDING 	= "DESCENDING"
	  | cycleAsString LOW 		= "LOW       "
	;
	fun asString (bs : bus_state)  =
		"{" ^
		" Cycle=" ^ cycleAsString (#Cycle bs) ^
		" M1=" ^ Bit.asString (#M1 bs) ^
		" MREQ=" ^ Bit.asString (#MREQ bs) ^
		" IORQ=" ^ Bit.asString (#IORQ bs) ^
		" RD=" ^ Bit.asString (#RD bs) ^
		" WR=" ^ Bit.asString (#WR bs) ^
		" RFSH=" ^ Bit.asString (#RFSH bs) ^
		" HALT=" ^ Bit.asString (#HALT bs) ^
		" WAIT=" ^ Bit.asString (#WAIT bs) ^
		" RESET=" ^ Bit.asString (#RESET bs) ^
		" INT=" ^ Bit.asString (#INT bs) ^
		" NMI=" ^ Bit.asString (#NMI bs) ^
		" BUSRQ=" ^ Bit.asString (#BUSRQ bs) ^
		" BUSAK=" ^ Bit.asString (#BUSAK bs) ^
		" Address=" ^ Register16.asString (#Address bs) ^
		" Data=" ^ Register8.asString (#Data bs) ^
		" }"
	;

	fun getM1 (bs : bus_state) : Bit.bit = #M1 bs;
	fun isSetM1 (bs : bus_state) = Bit.isSet (getM1 bs);
	fun isNotSetM1 (bs : bus_state) = Bit.isNotSet (getM1 bs);

	fun getMREQ (bs : bus_state) : Bit.bit = #MREQ bs;
	fun isSetMREQ (bs : bus_state) = Bit.isSet (getMREQ bs);
	fun isNotSetMREQ (bs : bus_state) = Bit.isNotSet (getMREQ bs);

	fun getIORQ (bs : bus_state) : Bit.bit = #IORQ bs;
	fun isSetIORQ (bs : bus_state) = Bit.isSet (getIORQ bs);
	fun isNotSetIORQ (bs : bus_state) = Bit.isNotSet (getIORQ bs);

	fun getRD (bs : bus_state) : Bit.bit = #RD bs;
	fun isSetRD (bs : bus_state) = Bit.isSet (getRD bs);
	fun isNotSetRD (bs : bus_state) = Bit.isNotSet (getRD bs);

	fun getWR (bs : bus_state) : Bit.bit = #WR bs;
	fun isSetWR (bs : bus_state) = Bit.isSet (getWR bs);
	fun isNotSetWR (bs : bus_state) = Bit.isNotSet (getWR bs);

	fun getHALT (bs : bus_state) : Bit.bit = #HALT bs;
	fun isSetHALT (bs : bus_state) = Bit.isSet (getHALT bs);
	fun isNotSetHALT (bs : bus_state) = Bit.isNotSet (getHALT bs);

	fun getWAIT (bs : bus_state) : Bit.bit = #WAIT bs;
	fun isSetWAIT (bs : bus_state) = Bit.isSet (getWAIT bs);
	fun isNotSetWAIT (bs : bus_state) = Bit.isNotSet (getWAIT bs);

	fun getINT (bs : bus_state) : Bit.bit = #INT bs;
	fun isSetINT (bs : bus_state) = Bit.isSet (getINT bs);
	fun isNotSetINT (bs : bus_state) = Bit.isNotSet (getINT bs);

	fun getNMI (bs : bus_state) : Bit.bit = #NMI bs;
	fun isSetNMI (bs : bus_state) = Bit.isSet (getNMI bs);
	fun isNotSetNMI (bs : bus_state) = Bit.isNotSet (getNMI bs);

	fun getBUSRQ (bs : bus_state) : Bit.bit = #BUSRQ bs;
	fun isSetBUSRQ (bs : bus_state) = Bit.isSet (getBUSRQ bs);
	fun isNotSetBUSRQ (bs : bus_state) = Bit.isNotSet (getBUSRQ bs);

	fun getBUSAK (bs : bus_state) : Bit.bit = #BUSAK bs;
	fun isSetBUSAK (bs : bus_state) = Bit.isSet (getBUSAK bs);
	fun isNotSetBUSAK (bs : bus_state) = Bit.isNotSet (getBUSAK bs);

	fun getRFSH (bs : bus_state) : Bit.bit = #RFSH bs;
	fun isSetRFSH (bs : bus_state) = Bit.isSet (getRFSH bs);
	fun isNotSetRFSH (bs : bus_state) = Bit.isNotSet (getRFSH bs);

	fun getRESET (bs : bus_state) : Bit.bit = #RESET bs;
	fun isSetRESET (bs : bus_state) = Bit.isSet (getRESET bs);
	fun isNotSetRESET (bs : bus_state) = Bit.isNotSet (getRESET bs);

	fun getData (bs : bus_state)  = #Data bs;
	fun getAddress (bs : bus_state)  = #Address bs;
	fun getCycle (bs : bus_state)  = #Cycle bs;




end;
