/* dazukofs: access control stackable filesystem

   Copyright (C) 1997-2004 Erez Zadok
   Copyright (C) 2001-2004 Stony Brook University
   Copyright (C) 2004-2007 International Business Machines Corp.
   Copyright (C) 2008 John Ogness
     Author: John Ogness <dazukocode@ogness.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <linux/module.h>
#include <linux/fs.h>
#include <linux/namei.h>
#include <linux/mount.h>
#include <linux/uaccess.h>
#include <linux/fs_stack.h>

#include "dazukofs_fs.h"

static struct inode_operations dazukofs_symlink_iops;
static struct inode_operations dazukofs_dir_iops;
static struct inode_operations dazukofs_main_iops;

static int dazukofs_inode_test(struct inode *inode,
			       void *candidate_lower_inode)
{
	if (GET_LOWER_INODE(inode) ==
	    (struct inode *)candidate_lower_inode) {
		return 1;
	}

	return 0;
}

static void dazukofs_init_inode(struct inode *inode, struct inode *lower_inode)
{
        SET_LOWER_INODE(inode, lower_inode);
        inode->i_ino = lower_inode->i_ino;
        inode->i_version++;

        if (S_ISDIR(lower_inode->i_mode)) {
                inode->i_op = &dazukofs_dir_iops;
                inode->i_fop = &dazukofs_dir_fops;
        } else {
                inode->i_op = &dazukofs_main_iops;
                inode->i_fop = &dazukofs_main_fops;

                if (S_ISLNK(lower_inode->i_mode))
                        inode->i_op = &dazukofs_symlink_iops;
                else  if (special_file(lower_inode->i_mode)) {
                        init_special_inode(inode, lower_inode->i_mode,
                                           lower_inode->i_rdev);
                }
        }
        inode->i_mapping->a_ops = &dazukofs_aops;
}

static int dazukofs_inode_set(struct inode *inode, void *lower_inode)
{
	dazukofs_init_inode(inode, (struct inode *)lower_inode);
	return 0;
}


int dazukofs_interpose(struct dentry *lower_dentry, struct dentry *dentry,
		     struct super_block *sb, int hash)
{
	struct inode *inode;
	struct inode *lower_inode;

	lower_inode = igrab(lower_dentry->d_inode);

	if (!lower_inode) 
		return -ESTALE;

	inode = iget5_locked(sb, (unsigned long)lower_inode,
			     dazukofs_inode_test, dazukofs_inode_set,
			     lower_inode);
	if (!inode) {
		iput(lower_inode);
		return -EACCES;
	}

	if (inode->i_state & I_NEW) 
		unlock_new_inode(inode);
		/*
		 * This is a new node so we leave the lower_node "in use"
		 * and do not call iput().
		 */
	else 
		/*
		 * This is not a new node so we decrement the usage count.
		 */
		iput(lower_inode);


	if (hash)
		d_add(dentry, inode);
	else
		d_instantiate(dentry, inode);

	fsstack_copy_attr_all(inode, lower_inode, NULL);
	fsstack_copy_inode_size(inode, lower_inode);

	return 0;
}

/* creates a new lower dentry */
static struct dentry *dazukofs_new_lower_dentry(struct qstr *name,
						struct dentry *lower_base, 
						struct nameidata *nd)
{
	struct dentry *new_dentry;
	struct dentry *tmp;
	struct inode *lower_inode;

	lower_inode = lower_base->d_inode;
	/* XXX: is this check still necessary? (see __lookup_hash())
	   This is (hopely) protected by locked upper inode */
	if (IS_DEADDIR(lower_inode)) {
		new_dentry = ERR_PTR(-ENOENT);
		goto out;
	}
	tmp = d_alloc(lower_base, name);
	if (!tmp) {
		new_dentry = ERR_PTR(-ENOMEM);
		goto out;
	}
	new_dentry = lower_inode->i_op->lookup(lower_inode, tmp, nd);
	/* lookup() seemingly is allowed to return its own dentry (which 
	 * may indeed be a dentry or only an error). If so
	 * we use it and discard ours. Otherwise we use ours */ 
	if (!new_dentry) 
		new_dentry = tmp;
	else 
		dput(tmp);
		
out:		
	return new_dentry;
}


/* get lower dentry for given name */
static struct dentry *dazukofs_lookup_one_lower(struct qstr *name, 
						struct dentry *lower_base,
						struct vfsmount *lower_mnt)
{
	struct dentry *result;
	struct nameidata nd;
	int err;

	err = vfs_path_lookup(lower_base, lower_mnt, name->name, 0, &nd);

	if (!err) {
		/* inode already exists on disk */
		result = nd.path.dentry;
		/* we dont need the mount */
		mntput(nd.path.mnt);
		goto out;
	}
	if (err != -ENOENT) { /* this is a REAL error */	
		result = ERR_PTR(err);
		goto out;
	}	
	/* create a new (lower) dentry */
	result = dazukofs_new_lower_dentry(name, lower_base, &nd);
out:
	return result;

}


/**
 * Description: Called when the VFS needs to look up an inode in a parent
 * directory. The name to look for is found in the dentry. This method
 * must call d_add() to insert the found inode into the dentry. The
 * "i_count" field in the inode structure should be incremented. If the
 * named inode does not exist a NULL inode should be inserted into the
 * dentry (this is called a negative dentry). Returning an error code
 * from this routine must only be done on a real error, otherwise
 * creating inodes with system calls like create(2), mknod(2), mkdir(2)
 * and so on will fail. If you wish to overload the dentry methods then
 * you should initialise the "d_dop" field in the dentry; this is a
 * pointer to a struct "dentry_operations". This method is called with
 * the directory inode semaphore held.
 */
static struct dentry *dazukofs_lookup(struct inode *dir, struct dentry *dentry,
				      struct nameidata *nd)
{
	struct dentry *lower_dentry;
	struct dentry *lower_dentry_parent;
	struct vfsmount *lower_mnt;
	int err;

	lower_dentry_parent = GET_LOWER_DENTRY(dentry->d_parent);
	/* should be released in d_release() */
	lower_mnt = mntget(GET_LOWER_MNT(dentry->d_parent));

	lower_dentry = dazukofs_lookup_one_lower(&dentry->d_name, 
						 lower_dentry_parent, 
						 lower_mnt); 
	if (IS_ERR(lower_dentry)) {
		mntput(lower_mnt);
		return lower_dentry;
	}
	BUG_ON(!atomic_read(&lower_dentry->d_count));

	dentry->d_op = &dazukofs_dops;
	SET_DENTRY_INFO(dentry, kmem_cache_zalloc(dazukofs_dentry_info_cachep,
			GFP_KERNEL));

	if (!GET_DENTRY_INFO(dentry)) {
		mntput(lower_mnt);
		dput(lower_dentry);
		return ERR_PTR(-ENOMEM);
	}
	SET_LOWER_DENTRY(dentry, lower_dentry, lower_mnt);
	/* from now lower dentry and lower mount are released in 
	  d_release of upper dentry */
	fsstack_copy_attr_atime(dir, lower_dentry_parent->d_inode);
	if (!lower_dentry->d_inode) {
		/*
		 * We want to add because we could not find in lower.
		 */
		d_add(dentry, NULL);
		goto ok;
	}

	err = dazukofs_interpose(lower_dentry, dentry, dir->i_sb, 1);
	if (err) 
		return ERR_PTR(err);

ok:
	return NULL; /* tell caller to use its own, passed dentry, not ours */
}

/**
 * Description: Called by the mknod(2) system call to create a device
 * (char, block) inode or a named pipe (FIFO) or socket. Only required if
 * you want to support creating these types of inodes. You will probably
 * need to call d_instantiate() just as you would in the create() method.
 */
static int dazukofs_mknod(struct inode *dir, struct dentry *dentry, int mode,
			  dev_t dev)
{
	struct dentry *lower_dentry = GET_LOWER_DENTRY(dentry);
	struct dentry *lower_dentry_parent = dget(lower_dentry->d_parent);
	struct inode *lower_dentry_parent_inode = lower_dentry_parent->d_inode;
	int err = -ENOENT;

	mutex_lock_nested(&(lower_dentry_parent_inode->i_mutex),
			  I_MUTEX_PARENT);

	err = vfs_mknod(lower_dentry_parent_inode, lower_dentry, mode, dev);
	if (err)
		goto out;

	err = dazukofs_interpose(lower_dentry, dentry, dir->i_sb, 0);
	if (err)
		goto out;

	fsstack_copy_attr_times(dir, lower_dentry_parent_inode);
	fsstack_copy_inode_size(dir, lower_dentry_parent_inode);
out:
	mutex_unlock(&(lower_dentry_parent_inode->i_mutex));
	dput(lower_dentry_parent);

	return err;
}

/**
 * Description: Called by the mkdir(2) system call. Only required if you
 * want to support creating subdirectories. You will probably need to call
 * d_instantiate() just as you would in the create() method.
 */
static int dazukofs_mkdir(struct inode *dir, struct dentry *dentry, int mode)
{
	struct dentry *lower_dentry = GET_LOWER_DENTRY(dentry);
	struct dentry *lower_dentry_parent = dget(lower_dentry->d_parent);
	struct inode *lower_dentry_parent_inode = lower_dentry_parent->d_inode;
	int err = -ENOENT;

	mutex_lock_nested(&(lower_dentry_parent_inode->i_mutex),
			  I_MUTEX_PARENT);

	err = vfs_mkdir(lower_dentry_parent_inode, lower_dentry, mode);
	if (err)
		goto out;

	err = dazukofs_interpose(lower_dentry, dentry, dir->i_sb, 0);
	if (err)
		goto out;

	fsstack_copy_attr_times(dir, lower_dentry_parent_inode);
	fsstack_copy_inode_size(dir, lower_dentry_parent_inode);
	dir->i_nlink = lower_dentry_parent_inode->i_nlink;
out:
	mutex_unlock(&(lower_dentry_parent_inode->i_mutex));
	dput(lower_dentry_parent);

	return err;
}

/**
 * Description: Called by the open(2) and creat(2) system calls. Only
 * required if you want to support regular files. The dentry you get
 * should not have an inode (i.e. it should be a negative dentry). Here
 * you will probably call d_instantiate() with the dentry and the newly
 * created inode.
 */
static int dazukofs_create(struct inode *dir, struct dentry *dentry, int mode,
			   struct nameidata *nd)
{
	struct vfsmount *lower_mnt = GET_LOWER_MNT(dentry);
	struct dentry *lower_dentry = GET_LOWER_DENTRY(dentry);
	struct dentry *lower_dentry_parent = dget(lower_dentry->d_parent);
	struct inode *lower_dentry_parent_inode = lower_dentry_parent->d_inode;
	struct vfsmount *vfsmount_save;
	struct dentry *dentry_save;
	int err = -ENOENT;

	mutex_lock_nested(&(lower_dentry_parent_inode->i_mutex),
			  I_MUTEX_PARENT);

	vfsmount_save = nd->path.mnt;
	dentry_save = nd->path.dentry;

	nd->path.mnt = mntget(lower_mnt);
	nd->path.dentry = dget(lower_dentry);

	err = vfs_create(lower_dentry_parent_inode, lower_dentry, mode, nd);

	mntput(lower_mnt);
	dput(lower_dentry);

	nd->path.mnt = vfsmount_save;
	nd->path.dentry = dentry_save;

	if (err)
		goto out;

	err = dazukofs_interpose(lower_dentry, dentry, dir->i_sb, 0);
	if (err)
		goto out;

	fsstack_copy_attr_times(dir, lower_dentry_parent_inode);
	fsstack_copy_inode_size(dir, lower_dentry_parent_inode);
out:
	mutex_unlock(&(lower_dentry_parent_inode->i_mutex));
	dput(lower_dentry_parent);

	return err;
}

/**
 * Description: Called by the symlink(2) system call. Only required if you
 * want to support symlinks. You will probably need to call d_instantiate()
 * just as you would in the create() method.
 */
static int dazukofs_symlink(struct inode *dir, struct dentry *dentry,
			    const char *symname)
{
	struct dentry *lower_dentry = GET_LOWER_DENTRY(dentry);
	struct dentry *lower_dentry_parent = dget(lower_dentry->d_parent);
	struct inode *lower_dentry_parent_inode = lower_dentry_parent->d_inode;
	int err = -ENOENT;

	mutex_lock_nested(&(lower_dentry_parent_inode->i_mutex),
			  I_MUTEX_PARENT);

	err = vfs_symlink(lower_dentry_parent_inode, lower_dentry, symname,
			  S_IALLUGO);
	if (err)
		goto out;

	err = dazukofs_interpose(lower_dentry, dentry, dir->i_sb, 0);
	if (err)
		goto out;

	fsstack_copy_attr_times(dir, lower_dentry_parent_inode);
	fsstack_copy_inode_size(dir, lower_dentry_parent_inode);
out:
	mutex_unlock(&(lower_dentry_parent_inode->i_mutex));
	dput(lower_dentry_parent);

	return err;
}

/**
 * Description: Called by the readlink(2) system call. Only required if
 * you want to support reading symbolic links.
 */
static int dazukofs_readlink(struct dentry *dentry, char __user *buf,
			     int bufsiz)
{
	struct dentry *lower_dentry = GET_LOWER_DENTRY(dentry);
	struct inode *lower_dentry_inode = lower_dentry->d_inode;
	int err = 0;

	if (!lower_dentry_inode) {
		err = -ENOENT;
		d_drop(dentry);
		goto out;
	}

	if (!lower_dentry_inode->i_op ||
	    !lower_dentry_inode->i_op->readlink) {
		err = -EINVAL;
		goto out;
	}

	err = lower_dentry_inode->i_op->readlink(lower_dentry, buf, bufsiz);
	if (err)
		goto out;

	fsstack_copy_attr_times(dentry->d_inode, lower_dentry_inode);
out:
	return err;
}

/**
 * Description: Called by the VFS to follow a symbolic link to the inode
 * it points to. Only required if you want to support symbolic links. This
 * method returns a void pointer cookie that is passed to put_link().
 */
static void *dazukofs_follow_link(struct dentry *dentry, struct nameidata *nd)
{
	mm_segment_t fs_save;
	int rc;
	char *buf;
	int len = PAGE_SIZE;
	int err = 0;

	/*
	 * Released in dazukofs_put_link(). Only release here on error.
	 */
	buf = kmalloc(len, GFP_KERNEL);
	if (!buf) {
		err = -ENOMEM;
		goto out;
	}

	fs_save = get_fs();
	set_fs(get_ds());
	rc = dazukofs_readlink(dentry, (char __user *)buf, len);
	set_fs(fs_save);

	if (rc < 0) {
		err = rc;
		goto out_free;
	}
	buf[rc] = 0;

	nd_set_link(nd, buf);
	goto out;

out_free:
	kfree(buf);
out:
	return ERR_PTR(err);
}

/**
 * Description: Called by the VFS to release resources allocated by
 * follow_link(). The cookie returned by follow_link() is passed to this
 * method as the last parameter. It is used by filesystems such as NFS
 * where page cache is not stable (i.e. page that was installed when the
 * symbolic link walk started might not be in the page cache at the end
 * of the walk).
 */
static void dazukofs_put_link(struct dentry *dentry, struct nameidata *nd,
			      void *ptr)
{
	/*
	 * Release the char* from dazukofs_follow_link().
	 */
	kfree(nd_get_link(nd));
}

/**
 * Description: Called by the VFS to check for access rights on a
 * POSIX-like filesystem.
 */
static int dazukofs_permission(struct inode *inode, int mask,
			       struct nameidata *nd)
{
	struct vfsmount *lower_mnt = NULL;
	struct dentry *lower_dentry = NULL;
	struct vfsmount *vfsmnt_save = NULL;
	struct dentry *dentry_save = NULL;
	int err;

	if (nd) {
		lower_mnt = GET_LOWER_MNT(nd->path.dentry);
		lower_dentry = GET_LOWER_DENTRY(nd->path.dentry);

		vfsmnt_save = nd->path.mnt;
		dentry_save = nd->path.dentry;

		nd->path.mnt = mntget(lower_mnt);
		nd->path.dentry = dget(lower_dentry);
	}

	err = permission(GET_LOWER_INODE(inode), mask, nd);

	if (nd) {
		mntput(lower_mnt);
		dput(lower_dentry);

		nd->path.mnt = vfsmnt_save;
		nd->path.dentry = dentry_save;
	}

        return err;
}

/**
 * Description: Called by the VFS to set attributes for a file. This method
 * is called by chmod(2) and related system calls.
 */
static int dazukofs_setattr(struct dentry *dentry, struct iattr *ia)
{
	struct dentry *lower_dentry = GET_LOWER_DENTRY(dentry);
	struct inode *inode = dentry->d_inode;
	struct inode *lower_inode = GET_LOWER_INODE(inode);
	int err;


	if (!lower_inode->i_op || !lower_inode->i_op->setattr) {
		/* XXX: at this point it is too late to handle inodes properly that do NOT
		specify setattr(). VFS would normally call inode_setattr() in 
		this case. So to force this we jump back to notify_change which will
		handle this for us. But this is dangerous:
		The attributes might be modified by notify_change() in a way
		that results in invalid attributes. This seems to be the case
		if the inode for which this function is called has the setuid
		or setgid bit set */ 
		if (ia->ia_valid & (ATTR_KILL_SUID | ATTR_KILL_SGID)) 
			ia->ia_valid &= ~ATTR_MODE;
		err = notify_change(lower_dentry, ia);
	} else 
		err = lower_inode->i_op->setattr(lower_dentry, ia);
	
	if (!err) {
		fsstack_copy_attr_all(inode, lower_inode, NULL);
		fsstack_copy_inode_size(inode, lower_inode);
	} 

	return err;
}
/**
 * Description: Called by the VFS to set an extended attribute for a file.
 * Extended attribute is a name:value pair associated with an inode. This
 * method is called by setxattr(2) system call.
 */
static int dazukofs_setxattr(struct dentry *dentry, const char *name,
			     const void *value, size_t size, int flags)
{
	struct dentry *lower_dentry = GET_LOWER_DENTRY(dentry);
	struct inode *lower_dentry_inode = lower_dentry->d_inode;
	int err = 0;

	if (!lower_dentry_inode) {
		err = -ENOENT;
		d_drop(dentry);
		goto out;
	}

	if (!lower_dentry_inode->i_op ||
	    !lower_dentry_inode->i_op->setxattr) {
		err = -EOPNOTSUPP;
		goto out;
	}

	err = lower_dentry_inode->i_op->setxattr(lower_dentry, name, value,
						 size, flags);

	fsstack_copy_attr_all(dentry->d_inode, lower_dentry_inode, NULL);
	fsstack_copy_inode_size(dentry->d_inode, lower_dentry_inode);
out:
	return err;
}

/**
 * Description: Called by the VFS to retrieve the value of an extended
 * attribute name. This method is called by getxattr(2) function call.
 */
static ssize_t dazukofs_getxattr(struct dentry *dentry, const char *name,
				 void *value, size_t size)
{
	struct dentry *lower_dentry = GET_LOWER_DENTRY(dentry);
	struct inode *lower_dentry_inode = lower_dentry->d_inode;
	ssize_t err = 0;

	if (!lower_dentry_inode) {
		err = -ENOENT;
		d_drop(dentry);
		goto out;
	}

	if (!lower_dentry_inode->i_op ||
	    !lower_dentry_inode->i_op->getxattr) {
		err = -EOPNOTSUPP;
		goto out;
	}

	err = lower_dentry_inode->i_op->getxattr(lower_dentry, name,
						 value, size);
out:
	return err;
}

/**
 * Description: Called by the VFS to list all extended attributes for a
 * given file. This method is called by listxattr(2) system call.
 */
static ssize_t dazukofs_listxattr(struct dentry *dentry, char *list,
				  size_t size)
{
	struct dentry *lower_dentry = GET_LOWER_DENTRY(dentry);
	struct inode *lower_dentry_inode = lower_dentry->d_inode;
	int err = 0;

	if (!lower_dentry_inode) {
		err = -ENOENT;
		d_drop(dentry);
		goto out;
	}

	if (!lower_dentry_inode->i_op ||
	    !lower_dentry_inode->i_op->listxattr) {
		err = -EOPNOTSUPP;
		goto out;
	}

	err = lower_dentry_inode->i_op->listxattr(lower_dentry, list, size);
out:
	return err;
}

/**
 * Description: Called by the VFS to remove an extended attribute from a
 * file. This method is called by removexattr(2) system call.
 */
static int dazukofs_removexattr(struct dentry *dentry, const char *name)
{
	struct dentry *lower_dentry = GET_LOWER_DENTRY(dentry);
	struct inode *lower_dentry_inode = lower_dentry->d_inode;
	int err = 0;

	if (!lower_dentry_inode) {
		err = -ENOENT;
		d_drop(dentry);
		goto out;
	}

	if (!lower_dentry_inode->i_op ||
	    !lower_dentry_inode->i_op->removexattr) {
		err = -EOPNOTSUPP;
		goto out;
	}

	err = lower_dentry_inode->i_op->removexattr(lower_dentry, name);
out:
	return err;
}

/**
 * Description: Called by the link(2) system call. Only required if you want
 * to support hard links. You will probably need to call d_instantiate()
 * just as you would in the create() method.
 */
static int dazukofs_link(struct dentry *old_dentry, struct inode *dir,
			 struct dentry *new_dentry)
{
	struct dentry *lower_old_dentry = GET_LOWER_DENTRY(old_dentry);
	struct dentry *lower_new_dentry = GET_LOWER_DENTRY(new_dentry);
	struct dentry *lower_dentry_parent = dget(lower_new_dentry->d_parent);
	struct inode *lower_dentry_parent_inode = lower_dentry_parent->d_inode;
	int err = -ENOENT;

	mutex_lock_nested(&(lower_dentry_parent_inode->i_mutex),
			  I_MUTEX_PARENT);

	err = vfs_link(lower_old_dentry, lower_dentry_parent_inode,
		       lower_new_dentry);
	if (err)
		goto out;

	err = dazukofs_interpose(lower_new_dentry, new_dentry, dir->i_sb, 0);
	if (err)
		goto out;

	fsstack_copy_attr_times(dir, lower_dentry_parent_inode);
	fsstack_copy_inode_size(dir, lower_dentry_parent_inode);
out:
	mutex_unlock(&(lower_dentry_parent_inode->i_mutex));
	dput(lower_dentry_parent);

	return err;
}

/**
 * Description: Called by the unlink(2) system call. Only required if you
 * want to support deleting inodes.
 */
static int dazukofs_unlink(struct inode *dir, struct dentry *dentry)
{
	struct dentry *lower_dentry = GET_LOWER_DENTRY(dentry);
	struct dentry *lower_dentry_parent = dget(lower_dentry->d_parent);
	struct inode *lower_dentry_parent_inode = lower_dentry_parent->d_inode;
	int err;

	mutex_lock_nested(&(lower_dentry_parent_inode->i_mutex),
			  I_MUTEX_PARENT);

	err = vfs_unlink(lower_dentry_parent_inode, lower_dentry);
	if (err)
		goto out;

	fsstack_copy_attr_times(dir, lower_dentry_parent_inode);
	dentry->d_inode->i_nlink =
		GET_LOWER_INODE(dentry->d_inode)->i_nlink;
	fsstack_copy_attr_times(dentry->d_inode, dir);
out:
	mutex_unlock(&(lower_dentry_parent_inode->i_mutex));
	dput(lower_dentry_parent);

	return err;
}

/**
 * Description: Called by the rmdir(2) system call. Only required if you
 * want to support deleting subdirectories.
 */
static int dazukofs_rmdir(struct inode *dir, struct dentry *dentry)
{
	struct dentry *lower_dentry = GET_LOWER_DENTRY(dentry);
	struct dentry *lower_dentry_parent = dget(lower_dentry->d_parent);
	struct inode *lower_dentry_parent_inode = lower_dentry_parent->d_inode;
	int err;

	mutex_lock_nested(&(lower_dentry_parent_inode->i_mutex),
			  I_MUTEX_PARENT);

	err = vfs_rmdir(lower_dentry_parent_inode, lower_dentry);
	if (err)
		goto out;

	fsstack_copy_attr_times(dir, lower_dentry_parent_inode);
	dir->i_nlink = lower_dentry_parent_inode->i_nlink;
out:
	mutex_unlock(&(lower_dentry_parent_inode->i_mutex));
	dput(lower_dentry_parent);

	if (!err)
		d_drop(dentry);

	return err;
}

/**
 * Description: Called by the rename(2) system call to rename the object to
 * have the parent and name given by the second inode and dentry.
 */
static int dazukofs_rename(struct inode *old_dir, struct dentry *old_dentry,
			   struct inode *new_dir, struct dentry *new_dentry)
{
	struct dentry *lower_old_dentry = GET_LOWER_DENTRY(old_dentry);
	struct dentry *lower_new_dentry = GET_LOWER_DENTRY(new_dentry);
	struct dentry *lower_old_dentry_parent =
		dget(lower_old_dentry->d_parent);
	struct dentry *lower_new_dentry_parent =
		dget(lower_new_dentry->d_parent);
	struct inode *lower_old_dentry_parent_inode =
		lower_old_dentry_parent->d_inode;
	struct inode *lower_new_dentry_parent_inode =
		lower_new_dentry_parent->d_inode;
	int err = -ENOENT;

	if (!lower_old_dentry_parent_inode) {
		d_drop(old_dentry);
		goto out;
	}

	if (!lower_new_dentry_parent_inode) {
		d_drop(new_dentry);
		goto out;
	}

	lock_rename(lower_old_dentry_parent, lower_new_dentry_parent);
	err = vfs_rename(lower_old_dentry_parent_inode, lower_old_dentry,
			 lower_new_dentry_parent_inode, lower_new_dentry);
	unlock_rename(lower_old_dentry_parent, lower_new_dentry_parent);

	if (err)
		goto out;

	fsstack_copy_attr_all(new_dir, lower_new_dentry_parent_inode, NULL);
	if (new_dir != old_dir)
		fsstack_copy_attr_all(old_dir, lower_old_dentry_parent_inode,
				      NULL);
out:
	dput(lower_old_dentry_parent);
	dput(lower_new_dentry_parent);

	return err;
}

/**
 * Unused operations:
 *   - create
 *   - lookup
 *   - link
 *   - unlink
 *   - symlink
 *   - mkdir
 *   - rmdir
 *   - mknod
 *   - rename
 *   - truncate
 *   - getattr
 *   - truncate_range
 *   - fallocate
 */
static struct inode_operations dazukofs_symlink_iops = {
	.readlink	= dazukofs_readlink,
	.follow_link	= dazukofs_follow_link,
	.put_link	= dazukofs_put_link,
	.permission	= dazukofs_permission,
	.setattr	= dazukofs_setattr,
	.setxattr	= dazukofs_setxattr,
	.getxattr	= dazukofs_getxattr,
	.listxattr	= dazukofs_listxattr,
	.removexattr	= dazukofs_removexattr,
};

/**
 * Unused operations:
 *   - readlink
 *   - follow_link
 *   - put_link
 *   - truncate
 *   - getattr
 *   - truncate_range
 *   - fallocate
 */
static struct inode_operations dazukofs_dir_iops = {
	.create		= dazukofs_create,
	.lookup		= dazukofs_lookup,
	.link		= dazukofs_link,
	.unlink		= dazukofs_unlink,
	.symlink	= dazukofs_symlink,
	.mkdir		= dazukofs_mkdir,
	.rmdir		= dazukofs_rmdir,
	.mknod		= dazukofs_mknod,
	.rename		= dazukofs_rename,
	.permission	= dazukofs_permission,
	.setattr	= dazukofs_setattr,
	.setxattr	= dazukofs_setxattr,
	.getxattr	= dazukofs_getxattr,
	.listxattr	= dazukofs_listxattr,
	.removexattr	= dazukofs_removexattr,
};

/**
 * Unused operations:
 *   - create
 *   - lookup
 *   - link
 *   - unlink
 *   - symlink
 *   - mkdir
 *   - rmdir
 *   - mknod
 *   - rename
 *   - readlink
 *   - follow_link
 *   - put_link
 *   - truncate
 *   - getattr
 *   - truncate_range
 *   - fallocate
 */
static struct inode_operations dazukofs_main_iops = {
	.permission	= dazukofs_permission,
	.setattr	= dazukofs_setattr,
	.setxattr	= dazukofs_setxattr,
	.getxattr	= dazukofs_getxattr,
	.listxattr	= dazukofs_listxattr,
	.removexattr	= dazukofs_removexattr,
};
