/* dazukofs: access control stackable filesystem

   Copyright (C) 2008 John Ogness
     Author: John Ogness <dazukocode@ogness.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <linux/list.h>
#include <linux/sched.h>
#include <linux/file.h>
#include <linux/fs.h>
#include <linux/mount.h>

#include "dev.h"
#include "dazukofs_fs.h"
//#include "dazukofs.h"

struct dazukofs_proc {
	struct list_head list;
	struct task_struct *curr;
};

struct dazukofs_event {
	int type;
	unsigned long event_id;
	struct dentry *dentry;
	struct vfsmount *mnt;
	pid_t pid;
	wait_queue_head_t queue;

	/* protects: deny, deprecated, assigned */
	struct mutex assigned_mutex;

	int deny;
	int deprecated;
	int assigned;
};

struct dazukofs_event_container {
	struct list_head list;
	struct dazukofs_event *event;
	struct file *file;
	struct file *grp_file;
	int fd;
};

struct dazukofs_group {
	struct list_head list;
	char *name;
	size_t name_length;
	unsigned long group_id;
	struct dazukofs_event_container todo_list;
	wait_queue_head_t queue;
	struct dazukofs_event_container working_list;
	atomic_t use_count;
	int tracking;
	int track_count;
	int deprecated;
};

static struct dazukofs_group group_list;

static wait_queue_head_t __group_count_queue;
static int __group_count;
static int __group_count_busy; /* if set, __group_count is protected */

/* protects: group_list, grp->members, last_event_id,
 *	     todo_list, working_list */
static struct mutex work_mutex;

/* protects: __group_count_busy */
static struct mutex group_count_mutex;

static struct mutex proc_mutex;
static struct dazukofs_proc proc_list;

static struct kmem_cache *dazukofs_group_cachep;
static struct kmem_cache *dazukofs_event_container_cachep;
static struct kmem_cache *dazukofs_event_cachep;
static struct kmem_cache *dazukofs_proc_cachep;

static int last_event_id;

/**
 * dazukofs_init_events - initialize event handling infrastructure
 *
 * Description: This is called once to initialize all the structures
 * needed to manage event handling.
 *
 * Returns 0 on success.
 */
int dazukofs_init_events(void)
{
	mutex_init(&proc_mutex);
	mutex_init(&work_mutex);
	mutex_init(&group_count_mutex);

	INIT_LIST_HEAD(&proc_list.list);
	INIT_LIST_HEAD(&group_list.list);

	init_waitqueue_head(&__group_count_queue);

	dazukofs_group_cachep =
		kmem_cache_create("dazukofs_group_cache",
				  sizeof(struct dazukofs_group), 0,
				  SLAB_HWCACHE_ALIGN, NULL, NULL);
	if (!dazukofs_group_cachep)
		goto error_out;

	dazukofs_event_container_cachep =
		kmem_cache_create("dazukofs_event_container_cache",
				  sizeof(struct dazukofs_event_container), 0,
				  SLAB_HWCACHE_ALIGN, NULL, NULL);
	if (!dazukofs_event_container_cachep)
		goto error_out;

	dazukofs_event_cachep =
		kmem_cache_create("dazukofs_event_cache",
				  sizeof(struct dazukofs_event), 0,
				  SLAB_HWCACHE_ALIGN, NULL, NULL);
	if (!dazukofs_event_cachep)
		goto error_out;

	dazukofs_proc_cachep = 
		kmem_cache_create("dazukofs_proc_cache",
				  sizeof(struct dazukofs_proc), 0,
				  SLAB_HWCACHE_ALIGN, NULL, NULL);
	if (!dazukofs_proc_cachep)
		goto error_out;

	return 0;

error_out:
	if (dazukofs_group_cachep)
		kmem_cache_destroy(dazukofs_group_cachep);
	if (dazukofs_event_container_cachep)
		kmem_cache_destroy(dazukofs_event_container_cachep);
	if (dazukofs_event_cachep)
		kmem_cache_destroy(dazukofs_event_cachep);
	if (dazukofs_proc_cachep)
		kmem_cache_destroy(dazukofs_proc_cachep);
	return -ENOMEM;
}

static int capture_group_count(int *cache)
{
	int err = 0;

	if (*cache)
		return 0;

	mutex_lock(&group_count_mutex);
	if (__group_count_busy)
		err = -1;
	else
		__group_count_busy = 1;
	mutex_unlock(&group_count_mutex);

	if (err == 0)
		*cache = 1;
	return err;
}

static int get_group_count_interruptible(void)
{
	int cache = 0;
	int ret = wait_event_interruptible(__group_count_queue,
				           capture_group_count(&cache) == 0);
	if (ret == 0)
		ret = __group_count;
	return ret;
}

static int get_group_count(void)
{
	int cache = 0;
	wait_event(__group_count_queue, capture_group_count(&cache) == 0);
	return __group_count;
}

static void put_group_count(void)
{
	mutex_lock(&group_count_mutex);
	__group_count_busy = 0;
	mutex_unlock(&group_count_mutex);
	wake_up(&__group_count_queue);
}

/**
 * release_event - release (and possible free) an event
 * @evt: the event to release
 * @decrement_assigned: flag to signal if the assigned count should be
 *                      decremented (only for registered processes)
 * @deny: flag if file access event should be denied
 *
 * Description: This function will decrement the assigned count for the
 * event. The "decrement_assigned" flag is used to distinguish between
 * the anonymous process accessing a file and the registered process. The
 * assigned count is only incremented for registered process (although the
 * anonymous process will also have a handle to the event).
 *
 * For the anonymous process (decrement_assigned = false):
 * If the assigned count is not zero, there are registered processes that
 * have a handle to this event. The event is marked deprecated. Otherwise
 * we free the event.
 *
 * For a registered process (decrement_assigned = true):
 * The assigned count is decremented. If it is now zero and the event is
 * not deprecated, then the anonymous process still has a handle. In this
 * case we wake the anonymous process. Otherwise we free the event.
 *
 * Aside from releasing the event, the deny status of the event is also
 * updated. The "normal" release process involves the registered processes
 * first releasing (and providing their deny values) and finally the
 * anonymous process will release (and free) the event after reading the
 * deny value.
 */
static void release_event(struct dazukofs_event *evt, int decrement_assigned,
			  int deny)
{
	int free_event = 0;

	mutex_lock(&evt->assigned_mutex);
	if (deny)
		evt->deny |= 1;

	if (decrement_assigned) {
		evt->assigned--;
		if (evt->assigned == 0) {
			if (!evt->deprecated)
				wake_up(&evt->queue);
			else
				free_event = 1;
		}
	} else {
		if (evt->assigned == 0)
			free_event = 1;
		else
			evt->deprecated = 1;
	}
	mutex_unlock(&evt->assigned_mutex);

	if (free_event) {
		dput(evt->dentry);
		mntput(evt->mnt);
		kmem_cache_free(dazukofs_event_cachep, evt);
	}
}

/**
 * __clear_group_event_list - cleanup/release event list
 * @event_list - the list to clear
 *
 * Description: All events (and their containers) will be released/freed
 * for the given event list. The event list will be an empty (yet still
 * valid) list after this function is finished.
 *
 * IMPORTANT: This function requires work_mutex to be held!
 */
static void __clear_group_event_list(struct list_head *event_list)
{
	struct dazukofs_event_container *ec;
	struct list_head *pos;
	struct list_head *q;

	list_for_each_safe(pos, q, event_list) {
		ec = list_entry(pos, struct dazukofs_event_container, list);
		list_del(pos);

		release_event(ec->event, 1, 0);

		kmem_cache_free(dazukofs_event_container_cachep, ec);
	}
}

/**
 * __remove_group - clear all activity associated with the group
 * @grp: the group to clear
 *
 * Description: All pending and in-progress events are released/freed.
 * Any processes waiting on the queue are woken.
 *
 * IMPORTANT: This function requires work_mutex to be held!
 */
static void __remove_group(struct dazukofs_group *grp)
{
	grp->deprecated = 1;
	__group_count--;

	__clear_group_event_list(&grp->working_list.list);
	__clear_group_event_list(&grp->todo_list.list);

	/* notify all registered process waiting for an event */
	wake_up_all(&grp->queue);
}


static void cleanup_groups(void)
{
	struct list_head *q;
	struct list_head *pos;
	struct dazukofs_group *grp;

	list_for_each_safe(pos, q, &group_list.list) {
		grp = list_entry(pos, struct dazukofs_group, list);
		list_del(pos);

		__remove_group(grp);

		/* free group name */
		kfree(grp->name);

		/* free group */
		kmem_cache_free(dazukofs_group_cachep, grp);
	}
}

static void cleanup_ignored_procs(void)
{	struct list_head *q;
	struct list_head *pos;
	struct dazukofs_proc *proc;

	/* free the groups */
	list_for_each_safe(pos, q, &proc_list.list) {
		proc = list_entry(pos, struct dazukofs_proc, list);
		list_del(pos);
		/* free proc */
		kmem_cache_free(dazukofs_proc_cachep, proc);
	}
}


/**
 * dazukofs_destroy_events - cleanup/shutdown event handling infrastructure
 *
 * Description: Release all pending events, free all allocated structures.
 */
void dazukofs_destroy_events(void)
{
	/*
	 * We are not using any locks here because we assume
	 * everything else has been already cleaned up by
	 * the device layer.
	 */

	/* free the groups */
	cleanup_groups();
	/* free ignored processes */
	cleanup_ignored_procs();
	/* free everything else */
	kmem_cache_destroy(dazukofs_group_cachep);
	kmem_cache_destroy(dazukofs_event_container_cachep);
	kmem_cache_destroy(dazukofs_event_cachep);
	kmem_cache_destroy(dazukofs_proc_cachep);
}

/* requires work_mutex to be held */
static int __check_for_group(const char *name, int id, int track,
			     int *already_exists)
{
	struct dazukofs_group *grp;
	struct list_head *pos;
	struct list_head *q;
	int id_available = 1;

	*already_exists = 0;

	list_for_each_safe(pos, q, &group_list.list) {
		grp = list_entry(pos, struct dazukofs_group, list);
		if (grp->deprecated) {
			/* cleanup deprecated groups */
			if (atomic_read(&grp->use_count) == 0) {
				list_del(pos);
				kfree(grp->name);
				kmem_cache_free(dazukofs_group_cachep, grp);
			}
		} else {
			if (strcmp(name, grp->name) == 0) {
				*already_exists = 1;
				if (track)
					grp->tracking = 1;
				break;
			} else if (grp->group_id == id) {
				id_available = 0;
				break;
			}
		}
	}

	if (*already_exists)
		return 0;

	if (id_available) {
		/* we have found a free id */
		return 0;
	}

	return -1;
}

static struct dazukofs_group *create_group(const char *name, int id, int track)
{
	struct dazukofs_group *grp;

	grp = kmem_cache_zalloc(dazukofs_group_cachep, GFP_KERNEL);
	if (!grp)
		return NULL;

	atomic_set(&grp->use_count, 0);
	grp->group_id = id;
	grp->name = kstrdup(name, GFP_KERNEL);
	if (!grp->name) {
		kmem_cache_free(dazukofs_group_cachep, grp);
		return NULL;
	}
	grp->name_length = strlen(name);
	init_waitqueue_head(&grp->queue);
	INIT_LIST_HEAD(&grp->todo_list.list);
	INIT_LIST_HEAD(&grp->working_list.list);
	if (track)
		grp->tracking = 1;
	return grp;
}

int dazukofs_add_group(const char *name, int track)
{
	int ret = 0;
	int already_exists;
	int available_id = 0;
	struct dazukofs_group *grp;
	int grp_count = get_group_count_interruptible();

	if (grp_count < 0) {
		ret = grp_count;
		goto out1;
	}

	mutex_lock(&work_mutex);
	while (__check_for_group(name, available_id, track,
				 &already_exists) != 0) {
		/* try again with the next id */
		available_id++;
	}
	mutex_unlock(&work_mutex);

	if (already_exists)
		goto out2;

	/* if we are here, the group doesn't already exist */

	/* do we have room for a new group? */
	if (grp_count == GROUP_COUNT) {
		ret = -EPERM;
		goto out2;
	}

	grp = create_group(name, available_id, track);
	if (!grp) {
		ret = -ENOMEM;
		goto out2;
	}

	mutex_lock(&work_mutex);
	list_add_tail(&grp->list, &group_list.list);
	mutex_unlock(&work_mutex);

	__group_count++;
out2:
	put_group_count();
out1:
	return ret;
}

int dazukofs_remove_group(const char *name, int unused)
{
	int ret = 0;
	struct dazukofs_group *grp;
	struct list_head *pos;
	int grp_count = get_group_count_interruptible();

	if (grp_count < 0) {
		ret = grp_count;
		goto out1;
	}

	if (grp_count == 0)
		goto out2;

	mutex_lock(&work_mutex);
	/* set group deprecated */
	list_for_each(pos, &group_list.list) {
		grp = list_entry(pos, struct dazukofs_group, list);
		if (!grp->deprecated && strcmp(name, grp->name) == 0) {
			__remove_group(grp);
			break;
		}
	}
	mutex_unlock(&work_mutex);
out2:
	put_group_count();
out1:
	return ret;
}

int dazukofs_get_groups(char **buf)
{
	struct dazukofs_group *grp;
	char *tmp;
	struct list_head *pos;
	size_t buflen;
	size_t allocsize = 256;

tryagain:
	*buf = kzalloc(allocsize, GFP_KERNEL);
	if (!*buf)
		return -ENOMEM;
	tmp = *buf;
	buflen = 1;

	mutex_lock(&work_mutex);
	list_for_each(pos, &group_list.list) {
		grp = list_entry(pos, struct dazukofs_group, list);
		if (!grp->deprecated)
			buflen += grp->name_length + 3;
	}
	if (buflen < allocsize) {
		list_for_each(pos, &group_list.list) {
			grp = list_entry(pos, struct dazukofs_group, list);
			if (!grp->deprecated) {
				snprintf(tmp, (allocsize - 1) - (tmp - *buf),
					 "%lu:%s\n", grp->group_id,
					 grp->name);
				tmp += grp->name_length + 3;
			}
		}
		mutex_unlock(&work_mutex);
	} else {
		mutex_unlock(&work_mutex);
		allocsize *= 2;
		kfree(*buf);
		goto tryagain;
	}

	return 0;
}

/**
 * check_recursion - check if current process is recursing
 *
 * Description: A list of anonymous processes is managed to prevent
 * access event recursion. This function checks if the current process is
 * a part of that list.
 *
 * If the current process is found in the process list, it is removed.
 *
 * NOTE: The proc structure is not freed. It is only removed from the
 *       list. Since it is a recursive call, the caller can free the
 *       structure after the call chain is finished.
 *
 * Returns 0 if this is a recursive process call.
 */
static int check_recursion(void)
{
	struct dazukofs_proc *proc;
	struct list_head *pos;
	int found = 0;

	mutex_lock(&proc_mutex);
	list_for_each(pos, &proc_list.list) {
		proc = list_entry(pos, struct dazukofs_proc, list);
		if (proc->curr == current) {
			found = 1;
			list_del(pos);
			kmem_cache_free(dazukofs_proc_cachep, proc);
			break;
		}
	}
	mutex_unlock(&proc_mutex);

	/* process event if not found */
	return !found;
}

/**
 * event_assigned - check if event is (still) assigned
 * @event: event to check
 *
 * Description: This function checks if an event is still assigned. An
 * assigned event means that it is sitting on the todo or working list
 * of a group.
 *
 * Returns the number assigned count.
 */
static int event_assigned(struct dazukofs_event *event)
{
	int val;
	mutex_lock(&event->assigned_mutex);
	val = event->assigned;
	mutex_unlock(&event->assigned_mutex);
	return val;
}

/**
 * check_access_precheck - check if an access event should be generated
 *
 * Description: Check if the current process should cause an access event
 * to be generated.
 *
 * Returns 0 if an access event should be generated.
 */
static int check_access_precheck(int grp_count)
{
	/* do we have any groups? */
	if (grp_count == 0)
		return -1;

	/* am I a recursion process? */
	if (!check_recursion())
		return -1;
	/* am I an ignored process? */
	if (!dazukofs_check_ignore_process())
		return -1;

	return 0;
}

/**
 * assign_event_to_groups - post an event to be processed
 * @evt: the event to be posted
 * @ec_array: the containers for the event
 *
 * Description: This function will assign a unique id to the event.
 * The event will be associated with each container and the container is
 * placed on each group's todo list. Each group will also be woken to
 * handle the new event.
 */
static void
assign_event_to_groups(struct dazukofs_event *evt,
		       struct dazukofs_event_container *ec_array[])
{
	struct dazukofs_group *grp;
	struct list_head *pos;
	int i;

	mutex_lock(&work_mutex);
	mutex_lock(&evt->assigned_mutex);

	/* assign the event a "unique" id */

	last_event_id++;
	evt->event_id = last_event_id;

	/* assign the event to each group */
	i = 0;
	list_for_each(pos, &group_list.list) {
		grp = list_entry(pos, struct dazukofs_group, list);
		if (!grp->deprecated) {
			ec_array[i]->event = evt;

			evt->assigned++;
			list_add_tail(&ec_array[i]->list,
				      &grp->todo_list.list);

			/* notify someone to handle the event */
			wake_up(&grp->queue);

			i++;
		}
	}

	mutex_unlock(&evt->assigned_mutex);
	mutex_unlock(&work_mutex);
}

/**
 * allocate_event_and_containers - allocate an event and event containers
 * evt: event pointer to be assigned a new event
 * ec: event container array to be filled with new array of containers
 * @grp_count: the number of groups (size of the array)
 *
 * Description: New event and event container structures are allocated
 * and initialized.
 *
 * Returns 0 on success.
 */
static int
allocate_event_and_containers(struct dazukofs_event **evt,
			      struct dazukofs_event_container *ec_array[],
			      int grp_count)
{
	int i;

	*evt = kmem_cache_zalloc(dazukofs_event_cachep, GFP_KERNEL);
	if (!*evt)
		return -1;
	init_waitqueue_head(&(*evt)->queue);
	mutex_init(&(*evt)->assigned_mutex);

	/* allocate containers now while we don't have a lock */
	for (i = 0; i < grp_count; i++) {
		ec_array[i] = kmem_cache_zalloc(
				dazukofs_event_container_cachep, GFP_KERNEL);
		if (!ec_array[i])
			goto error_out;
	}

	return 0;

error_out:
	for (i--; i >= 0; i--) {
		kmem_cache_free(dazukofs_event_container_cachep, ec_array[i]);
		ec_array[i] = NULL;
	}
	kmem_cache_free(dazukofs_event_cachep, *evt);
	*evt = NULL;
	return -1;
}

/**
 * dazukofs_check_access - check for allowed file access
 * @dentry: the dentry associated with the file access
 * @mnt: the vfsmount associated with the file access
 *
 * Description: This is the only function used by the stackable filesystem
 * layer to check if a file may be accessed.
 *
 * Returns 0 if the file access is allowed.
 */
int dazukofs_check_access(struct dentry *dentry, struct vfsmount *mnt)
{
	struct dazukofs_event_container *ec_array[GROUP_COUNT];
	struct dazukofs_event *evt;
	int grp_count;
	int err = 0;


	grp_count = get_group_count();

	if (check_access_precheck(grp_count)) {
		put_group_count();
		return 0;
	}
	/* at this point, the access should be handled */

	if (allocate_event_and_containers(&evt, ec_array, grp_count)) {
		put_group_count();
		return -ENOMEM;
	}
	evt->dentry = dget(dentry);
	evt->mnt = mntget(mnt);
	evt->pid = current->pid;
	assign_event_to_groups(evt, ec_array);

	put_group_count();

	/* wait until event completely processed */
	wait_event(evt->queue, event_assigned(evt) == 0);

	if (evt->deny)
		err = -EPERM;

	release_event(evt, 0, 0);
	
	return err;
}


int dazukofs_group_open_tracking(unsigned long group_id)
{
	struct dazukofs_group *grp;
	struct list_head *pos;
	int tracking = 0;

	mutex_lock(&work_mutex);
	list_for_each(pos, &group_list.list) {
		grp = list_entry(pos, struct dazukofs_group, list);
		if (!grp->deprecated && grp->group_id == group_id) {
			if (grp->tracking) {
				atomic_inc(&grp->use_count);
				grp->track_count++;
				tracking = 1;
			}
			break;
		}
	}
	mutex_unlock(&work_mutex);
	return tracking;
}

/* called with work_mutex held */
static void reset_pending_events(struct file *grp_file, 
			        struct dazukofs_group *grp)
{
	struct dazukofs_event_container *ec = NULL;
	struct list_head *pos;
	struct list_head *tmp;
	int wakeup = 0;

	list_for_each_safe(pos, tmp, &grp->working_list.list) {
		ec = list_entry(pos, struct dazukofs_event_container, list);
		if (ec->grp_file == grp_file) {
			list_del(&ec->list);
			list_add(&ec->list, &grp->todo_list.list);
			ec->grp_file = NULL;
			wakeup = 1;
		}
	}
	if (wakeup)
		wake_up(&grp->queue);
}

void dazukofs_group_release_tracking(struct file *grp_file, unsigned long group_id)
{
	struct dazukofs_group *grp;
	struct list_head *pos;

	mutex_lock(&work_mutex);
	list_for_each(pos, &group_list.list) {
		grp = list_entry(pos, struct dazukofs_group, list);

		/* set unprocessed events of this daemon back to todo list */
		reset_pending_events(grp_file, grp);

		if (!grp->deprecated && grp->group_id == group_id) {
			if (grp->tracking) {
				atomic_dec(&grp->use_count);
				grp->track_count--;
				if (grp->track_count == 0)
					__remove_group(grp);
			}
			break;
		}
	}
	mutex_unlock(&work_mutex);
}

/**
 * dazukofs_return_event - return checked file access results
 * @event_id: the id of the event
 * @deny: a flag indicating if file access should be denied
 *
 * Description: This function is called by the device layer when returning
 * results from a checked file access event. If the event_id was valid, the
 * event container will be freed and the event released.
 *
 * Returns 0 on success.
 */
int dazukofs_return_event(unsigned long group_id, unsigned long event_id,
			  int deny)
{
	struct dazukofs_group *grp;
	struct dazukofs_event_container *ec;
	struct dazukofs_event *evt = NULL;
	struct list_head *pos;
	int found = 0;
	int ret = 0;

	mutex_lock(&work_mutex);
	list_for_each(pos, &group_list.list) {
		grp = list_entry(pos, struct dazukofs_group, list);
		if (!grp->deprecated && grp->group_id == group_id) {
			found = 1;
			break;
		}
	}
	if (!found) {
		ret = -EFAULT;
		mutex_unlock(&work_mutex);
		goto out;
	}

	found = 0;
	list_for_each(pos, &grp->working_list.list) {
		ec = list_entry(pos, struct dazukofs_event_container, list);
		evt = ec->event;
		if (evt->event_id == event_id) {
			found = 1;
			list_del(pos);
			kmem_cache_free(dazukofs_event_container_cachep, ec);
			break;
		}
	}
	mutex_unlock(&work_mutex);

	if (found)
		release_event(evt, 1, deny);
	else
		ret = -EFAULT;
out:
	return ret;
}

/**
 * unclaim_event - return an event to the todo list
 * @grp: group to which the event is assigned
 * @ec: event container of the event to be returned
 *
 * Description: This function puts the given event container on the todo
 * list and wake the group.
 */
static void unclaim_event(struct dazukofs_group *grp,
			  struct dazukofs_event_container *ec)
{
	/* put the event on the todo list */
	mutex_lock(&work_mutex);
	list_add(&ec->list, &grp->todo_list.list);
	mutex_unlock(&work_mutex);

	/* wake up someone else to handle the event */
	wake_up(&grp->queue);
}

/**
 * claim_event - grab an event from the todo list
 * @grp: the group
 *
 * Description: Take the first event from the todo list and move it to the
 * working list. The event is then returned to its called for processing.
 *
 * Returns the claimed event.
 */
static struct dazukofs_event_container *claim_event(struct dazukofs_group *grp)
{
	struct dazukofs_event_container *ec = NULL;

	/* move first todo-item to working list */
	mutex_lock(&work_mutex);
	if (!list_empty(&grp->todo_list.list)) {
		ec = list_first_entry(&grp->todo_list.list,
				      struct dazukofs_event_container, list);
		list_del(&ec->list);
		list_add(&ec->list, &grp->working_list.list);
	}
	mutex_unlock(&work_mutex);

	return ec;
}

/**
 * mask_proc - mask the current process
 * @proc: process structure to use for the list
 *
 * Description: Assign the current process to the provided proc structure
 * and add the structure to the list. The list is used to prevent
 * generating recursive file access events. The process is removed from
 * the list with the check_recursion() function.
 */
static int mask_proc(void)
{
	struct dazukofs_proc *proc;

	proc = kmem_cache_zalloc(dazukofs_proc_cachep, GFP_KERNEL);
	if (!proc)
		return -ENOMEM;
	proc->curr = current;
	mutex_lock(&proc_mutex);
	list_add(&proc->list, &proc_list.list);
	mutex_unlock(&proc_mutex);
	return 0;
}

/**
 * open_file - open a file for the current process (avoiding recursion)
 * @ec: event container to store opened file descriptor
 *
 * Description: This function will open a file using the information within
 * the provided event container. The calling process will be temporarily
 * masked so that the file open does not generate a file access event.
 *
 * Returns 0 on success.
 */
static int open_file(struct dazukofs_event_container *ec)
{
	struct dazukofs_event *evt = ec->event;
	int ret = 0;

	/* open the file read-only */

	ec->fd = get_unused_fd();
	if (ec->fd < 0) {
		ret = ec->fd;
		goto error_out1;
	}

	/* add myself to be ignored on file open (to avoid recursion) */
	ret = mask_proc();
	if (ret) 
		goto error_out2;

	ec->file = dentry_open(dget(evt->dentry), mntget(evt->mnt), 
			       O_RDONLY | O_LARGEFILE);
	if (IS_ERR(ec->file)) {
		check_recursion();
		ret = PTR_ERR(ec->file);
		goto error_out2;
	}

	fd_install(ec->fd, ec->file);

	return 0;

error_out2:
	put_unused_fd(ec->fd);
error_out1:
	return ret;
}

/**
 * is_event_available - check if an event is available for processing
 * @grp: the group
 *
 * Description: This function simply checks if there are any events posted
 * in the group's todo list.
 *
 * Returns 0 if there are no events in the todo list.
 */
static int is_event_available(struct dazukofs_group *grp)
{
	int ret = 0;

	mutex_lock(&work_mutex);
	if (!list_empty(&grp->todo_list.list))
		ret = 1;
	mutex_unlock(&work_mutex);

	return ret;
}

/**
 * dazukofs_get_event - get an event to process
 * @event_id: to be filled in with the new event id
 * @fd: to be filled in with the opened file descriptor
 * @pid: to be filled in with the pid of the process generating the event
 *
 * Description: This function is called by the device layer to get a new
 * file access event to process. It waits until an event has been
 * posted in the todo list (and is successfully claimed by this process).
 *
 * Returns 0 on success.
 */
int dazukofs_get_event(struct file *grp_file, unsigned long group_id, 
                       unsigned long *event_id, int *fd, pid_t *pid)
{
	struct dazukofs_group *grp;
	struct dazukofs_event_container *ec = NULL;
	struct list_head *pos;
	int found = 0;
	int ret = 0;

	mutex_lock(&work_mutex);
	list_for_each(pos, &group_list.list) {
		grp = list_entry(pos, struct dazukofs_group, list);
		if (!grp->deprecated && grp->group_id == group_id) {
			atomic_inc(&grp->use_count);
			found = 1;
			break;
		}
	}
	mutex_unlock(&work_mutex);

	if (!found) 
		return -EINVAL;

	while (!ec) {
		ret = wait_event_interruptible(grp->queue, 
				is_event_available(grp) || grp->deprecated);
		if (ret)
			goto out;

		if (grp->deprecated) {
			ret = -EINVAL;
			goto out;
		}
		ec = claim_event(grp);
	}
	ret = open_file(ec);

	if (ret && ret != EPERM) {
		unclaim_event(grp, ec);
		goto out;
	}
	*event_id = ec->event->event_id;
	*fd = ec->fd;
	*pid = ec->event->pid;
	ec->grp_file = grp_file;
out:
	atomic_dec(&grp->use_count);
	return ret;
}
