/* dazukofs: access control stackable filesystem

   Copyright (C) 1997-2004 Erez Zadok
   Copyright (C) 2001-2004 Stony Brook University
   Copyright (C) 2004-2007 International Business Machines Corp.
   Copyright (C) 2008 John Ogness
     Author: John Ogness <dazukocode@ogness.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#ifndef __DAZUKOFS_FS_H
#define __DAZUKOFS_FS_H

extern struct kmem_cache *dazukofs_dentry_info_cachep;
extern struct kmem_cache *dazukofs_file_info_cachep;
extern struct file_operations dazukofs_main_fops;
extern const struct file_operations dazukofs_dir_fops;
extern struct dentry_operations dazukofs_dops;
extern struct address_space_operations dazukofs_aops;

extern int dazukofs_interpose(struct dentry *lower_dentry,
			      struct dentry *dentry, struct super_block *sb,
			      int flag);

struct dazukofs_sb_info {
	struct super_block *lower_sb;
};

struct dazukofs_inode_info {
	struct inode *lower_inode;

	/*
	 * the inode (embedded)
	 */
	struct inode vfs_inode;
};

struct dazukofs_dentry_info {
	struct dentry *lower_dentry;
	struct vfsmount *lower_mnt;
};

struct dazukofs_file_info {
	struct file *lower_file;
};

void fsstack_copy_inode_size(struct inode *dst, const struct inode *src);

static inline
void fsstack_copy_attr_atime(struct inode *dst, const struct inode *src)
{
	dst->i_atime = src->i_atime;
}

static inline
void fsstack_copy_attr_times(struct inode *dst, const struct inode *src)
{
	dst->i_atime = src->i_atime;
	dst->i_mtime = src->i_mtime;
	dst->i_ctime = src->i_ctime;
}

static inline 
void fsstack_copy_attr_all(struct inode *dst, const struct inode *src)
{
	dst->i_mode = src->i_mode;
	dst->i_uid = src->i_uid;
	dst->i_gid = src->i_gid;
	dst->i_rdev = src->i_rdev;
	dst->i_atime = src->i_atime;
	dst->i_mtime = src->i_mtime;
	dst->i_ctime = src->i_ctime;
	dst->i_blkbits = src->i_blkbits;
	dst->i_flags = src->i_flags;
}

static inline
struct dazukofs_sb_info *GET_SB_INFO(struct super_block *upper_sb)
{
	return upper_sb->s_fs_info;
}

static inline void SET_SB_INFO(struct super_block *upper_sb,
			       struct dazukofs_sb_info *sbi)
{
	upper_sb->s_fs_info = sbi;
}

static inline struct super_block *GET_LOWER_SB(struct super_block *upper_sb)
{
	return ((struct dazukofs_sb_info *)upper_sb->s_fs_info)->lower_sb;
}

static inline void SET_LOWER_SB(struct super_block *upper_sb,
				struct super_block *lower_sb)
{
       ((struct dazukofs_sb_info *)upper_sb->s_fs_info)->lower_sb = lower_sb;
}

static inline
struct dazukofs_inode_info *GET_INODE_INFO(struct inode *upper_inode)
{
	return container_of(upper_inode, struct dazukofs_inode_info,
			    vfs_inode);
}

static inline struct inode *GET_LOWER_INODE(struct inode *upper_inode)
{
	return ((struct dazukofs_inode_info *)
		container_of(upper_inode, struct dazukofs_inode_info,
			     vfs_inode))->lower_inode;
}

static inline void SET_LOWER_INODE(struct inode *upper_inode,
				   struct inode *lower_inode)
{
	((struct dazukofs_inode_info *)
	 container_of(upper_inode, struct dazukofs_inode_info,
		      vfs_inode))->lower_inode = lower_inode;
}

static inline
struct dazukofs_dentry_info *GET_DENTRY_INFO(struct dentry *upper_dentry)
{
	return upper_dentry->d_fsdata;
}

static inline void SET_DENTRY_INFO(struct dentry *upper_dentry,
				   struct dazukofs_dentry_info *dentryi)
{
	upper_dentry->d_fsdata = dentryi;
}

static inline struct dentry *GET_LOWER_DENTRY(struct dentry *upper_dentry)
{
	return ((struct dazukofs_dentry_info *)
		upper_dentry->d_fsdata)->lower_dentry;
}

static inline struct vfsmount *GET_LOWER_MNT(struct dentry *upper_dentry)
{
	return ((struct dazukofs_dentry_info *)
		upper_dentry->d_fsdata)->lower_mnt;
}

static inline void SET_LOWER_DENTRY(struct dentry *upper_dentry,
				    struct dentry *lower_dentry,
				    struct vfsmount *lower_mnt)
{
	((struct dazukofs_dentry_info *)upper_dentry->d_fsdata)->lower_dentry =
		lower_dentry;
	((struct dazukofs_dentry_info *)upper_dentry->d_fsdata)->lower_mnt =
		lower_mnt;
}

static inline struct dazukofs_file_info *GET_FILE_INFO(struct file *upper_file)
{
	return upper_file->private_data;
}

static inline void SET_FILE_INFO(struct file *upper_file,
				 struct dazukofs_file_info *filei)
{
	upper_file->private_data = filei;
}

static inline struct file *GET_LOWER_FILE(struct file *upper_file)
{
	return ((struct dazukofs_file_info *)
		upper_file->private_data)->lower_file;
}

static inline void SET_LOWER_FILE(struct file *upper_file,
				  struct file *lower_file)
{
	((struct dazukofs_file_info *)upper_file->private_data)->lower_file =
		lower_file;
}

#endif  /* __DAZUKOFS_FS_H */
