/* dazukofs: access control stackable filesystem

   Copyright (C) 1997-2004 Erez Zadok
   Copyright (C) 2001-2004 Stony Brook University
   Copyright (C) 2004-2007 International Business Machines Corp.
   Copyright (C) 2008 John Ogness
     Author: John Ogness <dazukocode@ogness.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <linux/module.h>
#include <linux/fs.h>
#include <linux/mount.h>
#include <linux/file.h>


#include "dazukofs_fs.h"
#include "event.h"
//#include "dazukofs.h"

/**
 * Description: Called when the VFS needs to move the file position index.
 */
static loff_t dazukofs_llseek(struct file *file, loff_t offset, int origin)
{
	loff_t retval = -EINVAL;
	struct file *lower_file = GET_LOWER_FILE(file);

	lower_file->f_pos = file->f_pos;

	memcpy(&(lower_file->f_ra), &(file->f_ra),
	       sizeof(struct file_ra_state));

	if (lower_file->f_op && lower_file->f_op->llseek)
		retval = lower_file->f_op->llseek(lower_file, offset, origin);
	else
		retval = generic_file_llseek(lower_file, offset, origin);

	if (retval >= 0) {
		file->f_pos = lower_file->f_pos;
		file->f_version = lower_file->f_version;
	}

	return retval;
}

/**
 * Description: Called by read(2) and related system calls.
 */
static ssize_t dazukofs_read(struct file *file, char *buf, size_t count,
			   loff_t *ppos)
{
	int err = -EINVAL;
	struct file *lower_file = GET_LOWER_FILE(file);
	loff_t pos_copy = *ppos;

	if (!lower_file->f_op || !lower_file->f_op->read)
		goto out;

	err = lower_file->f_op->read(lower_file, buf, count, &pos_copy);

	lower_file->f_pos = pos_copy;
	*ppos = pos_copy;

	if (err >= 0) {
		fsstack_copy_attr_atime(file->f_dentry->d_inode,
					lower_file->f_dentry->d_inode);
	}

	memcpy(&(file->f_ra), &(lower_file->f_ra),
	       sizeof(struct file_ra_state));
out:
	return err;
}

/**
 * Description: Called by write(2) and related system calls.
 */
static ssize_t dazukofs_write(struct file *file, const char *buf,
			      size_t count, loff_t *ppos)
{
	int err = -EINVAL;
	struct file *lower_file = GET_LOWER_FILE(file);
	struct inode *inode = file->f_dentry->d_inode;
	struct inode *lower_inode = GET_LOWER_INODE(inode);
	loff_t pos_copy = *ppos;

	if (!lower_file->f_op || !lower_file->f_op->write)
		goto out;

	err = lower_file->f_op->write(lower_file, buf, count, &pos_copy);

	lower_file->f_pos = pos_copy;
	*ppos = pos_copy;

	if (err >= 0)
		fsstack_copy_attr_atime(inode, lower_inode);

	memcpy(&(file->f_ra), &(lower_file->f_ra),
	       sizeof(struct file_ra_state));

	mutex_lock(&inode->i_mutex);
	i_size_write(inode, i_size_read(lower_inode));
	mutex_unlock(&inode->i_mutex);
out:
	return err;
}

/**
 * Description: Called when the VFS needs to read the directory contents.
 */
static int dazukofs_readdir(struct file *file, void *dirent, filldir_t filldir)
{
	int err;
	struct file *lower_file = GET_LOWER_FILE(file);
	struct inode *inode = file->f_dentry->d_inode;

	lower_file->f_pos = file->f_pos;

	err = vfs_readdir(lower_file, filldir, dirent);

	file->f_pos = lower_file->f_pos;

	if (err >= 0)
		fsstack_copy_attr_atime(inode, lower_file->f_dentry->d_inode);

	return err;
}

/**
 * Description: Called by the ioctl(2) system call.
 */
static int dazukofs_ioctl(struct inode *inode, struct file *file,
			  unsigned int cmd, unsigned long arg)
{
	struct file *lower_file = GET_LOWER_FILE(file);
	struct inode *lower_inode = GET_LOWER_INODE(inode);
	int err = -ENOTTY;

	if (!lower_file->f_op || !lower_file->f_op->ioctl) 
		goto out;

	err = lower_file->f_op->ioctl(lower_inode, lower_file, cmd, arg);
	fsstack_copy_attr_all(inode, lower_inode);
out:
	return err;
}

static long dazukofs_unlocked_ioctl (struct file *file, unsigned int cmd, 
                                     unsigned long arg)
{
	struct inode *inode = file->f_dentry->d_inode;
	struct file *lower_file = GET_LOWER_FILE(file);
	struct inode *lower_inode = lower_file->f_dentry->d_inode;
	int err = -ENOTTY;

	if (!lower_file->f_op || !lower_file->f_op->unlocked_ioctl) 
		goto out;

	err = lower_file->f_op->unlocked_ioctl(lower_file, cmd, arg);
	fsstack_copy_attr_all(inode, lower_inode);
out:
	return err;
}
static long dazukofs_compat_ioctl (struct file *file, unsigned int cmd, 
                                   unsigned long arg)
{
	struct inode *inode = file->f_dentry->d_inode;
	struct file *lower_file = GET_LOWER_FILE(file);
	struct inode *lower_inode = lower_file->f_dentry->d_inode;
	int err = -ENOTTY;

	if (lower_file->f_op || !lower_file->f_op->compat_ioctl) 
		goto out;

	err = lower_file->f_op->compat_ioctl(lower_file, cmd, arg);
	fsstack_copy_attr_all(inode, lower_inode);
out:
	return err;
}

/**
 * Description: Called by the VFS when an inode should be opened. When the
 * VFS opens a file, it creates a new "struct file". It then calls the open
 * method for the newly allocated file structure. You might think that the
 * open method really belongs in "struct inode_operations", and you may be
 * right. I think it's done the way it is because it makes filesystems
 * simpler to implement. The open() method is a good place to initialize
 * the "private_data" member in the file structure if you want to point to
 * a device structure.
 */
static int dazukofs_open(struct inode *inode, struct file *file)
{
	struct dentry *dentry = file->f_dentry;
	struct dentry *lower_dentry = dget(GET_LOWER_DENTRY(dentry));
	struct vfsmount *lower_mnt = mntget(GET_LOWER_MNT(dentry));
	struct dazukofs_file_info *fi;
	struct file *lower_file;

	if (S_ISREG(inode->i_mode)) {
		int err;

		err = dazukofs_check_access(file->f_dentry, file->f_vfsmnt);

		if (err) {
			dput(lower_dentry);
			mntput(lower_mnt);
			return err;
		}
	}
	fi = kmem_cache_alloc(dazukofs_file_info_cachep, GFP_KERNEL);
	if (!fi) {
		dput(lower_dentry);
		mntput(lower_mnt);
		return -ENOMEM;
	}
	memset(fi, 0, sizeof(*fi));
	SET_FILE_INFO(file, fi);

	lower_file = dentry_open(lower_dentry, lower_mnt, file->f_flags);
	if (IS_ERR(lower_file)) {
		/* dentry_open() already did dput() and mntput() */
		kmem_cache_free(dazukofs_file_info_cachep, GET_FILE_INFO(file));
		return PTR_ERR(lower_file);
	}

	SET_LOWER_FILE(file, lower_file);

	return 0;
}

/**
 * Description: Called by the close(2) system call to flush a file.
 */
static int dazukofs_flush(struct file *file, fl_owner_t td)
{
	struct file *lower_file = GET_LOWER_FILE(file);
	int err = 0;

	if (!lower_file || !lower_file->f_op || !lower_file->f_op->flush)
		goto out;

	err = lower_file->f_op->flush(lower_file, td);
out:
	return err;
}

/**
 * Description: Called when the last reference to an open file is closed.
 */
static int dazukofs_release(struct inode *inode, struct file *file)
{
	struct inode *lower_inode;

	lower_inode = GET_LOWER_INODE(inode);

	/* implicitly calls lower files release() */
	fput(GET_LOWER_FILE(file));
	inode->i_blocks = lower_inode->i_blocks;

	kmem_cache_free(dazukofs_file_info_cachep, GET_FILE_INFO(file));

	return 0;
}

/**
 * Description: Called by the fsync(2) system call.
 */
static int dazukofs_fsync(struct file *file, struct dentry *dentry,
			  int datasync)
{
	struct file *lower_file = GET_LOWER_FILE(file);
	struct dentry *lower_dentry = GET_LOWER_DENTRY(dentry);
	int err = -EINVAL;

	if (!lower_file || !lower_file->f_op || !lower_file->f_op->fsync)
		goto out;

	err = lower_file->f_op->fsync(lower_file, lower_dentry, datasync);
out:
	return err;
}

/**
 * Description: .called by the fcntl(2) system call when asynchronous
 * (non-blocking) mode is enabled for a file.
 */
static int dazukofs_fasync(int fd, struct file *file, int flag)
{
	struct file *lower_file = GET_LOWER_FILE(file);
	int err = 0;

	if (!lower_file || !lower_file->f_op || !lower_file->f_op->fasync)
		goto out;

	err = lower_file->f_op->fasync(fd, lower_file, flag);
out:
	return err;
}

static int dazukofs_mmap (struct file *file, struct vm_area_struct *vm)
{

	struct file *lower_file = GET_LOWER_FILE(file);

	/* if lower fs does not support mmap(), we dont call generic_file_readonly_mmap(), 
	 * since this would result in calling lower readpage(), which might not be 
	 * defined by lower fs if mmap() is not supported*/
	if (!lower_file->f_op || !lower_file->f_op->mmap) 
		return -ENODEV;
	

	return generic_file_readonly_mmap(file, vm);
}


/**
 * Unused operations:
 *   - owner
 *   - aio_read (generic)
 *   - aio_write (generic)
 *   - poll
 *   - unlocked_ioctl
 *   - compat_ioctl
 *   - mmap (generic)
 *   - aio_fsync
 *   - lock
 *   - sendpage
 *   - get_unmapped_area
 *   - check_flags
 *   - dir_notify
 *   - flock
 *   - splice_write
 *   - splice_read (generic)
 *   - setlease
 */
struct file_operations dazukofs_main_fops = {
	.llseek		= dazukofs_llseek,
	.read		= dazukofs_read,
	.write		= dazukofs_write,
	.readdir	= dazukofs_readdir,
	.ioctl		= dazukofs_ioctl,
	.unlocked_ioctl = dazukofs_unlocked_ioctl,
	.compat_ioctl   = dazukofs_compat_ioctl,
	.mmap		= dazukofs_mmap,	
	.open		= dazukofs_open,
	.flush		= dazukofs_flush,
	.release	= dazukofs_release,
	.fsync		= dazukofs_fsync,
	.fasync		= dazukofs_fasync,
};

/**
 * Unused operations:
 *   - owner
 *   - llseek
 *   - read
 *   - write
 *   - aio_read
 *   - aio_write
 *   - poll
 *   - unlocked_ioctl
 *   - compat_ioctl
 *   - mmap (generic)
 *   - aio_fsync
 *   - lock
 *   - sendpage
 *   - get_unmapped_area
 *   - check_flags
 *   - dir_notify
 *   - flock
 *   - splice_write
 *   - splice_read (generic)
 *   - setlease
 */
struct file_operations dazukofs_dir_fops = {
	.readdir	= dazukofs_readdir,
	.ioctl		= dazukofs_ioctl,
	.unlocked_ioctl = dazukofs_unlocked_ioctl,
	.compat_ioctl   = dazukofs_compat_ioctl,
	.mmap		= dazukofs_mmap,	
	.open		= dazukofs_open,
	.flush		= dazukofs_flush,
	.release	= dazukofs_release,
	.fsync		= dazukofs_fsync,
	.fasync		= dazukofs_fasync,
};
