/* dazukofs: access control stackable filesystem

   Copyright (C) 2008 John Ogness
     Author: John Ogness <dazukocode@ogness.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/platform_device.h>
#include <asm/uaccess.h>

#include "dazukofs_fs.h"
#include "event.h"
#include "dev.h"




struct group_device {
	struct cdev cdev;
	struct device dev;
	struct class_device cls_dev;
	int id;
	int track;
};

static struct group_device *grp_devs;
static int minor_start;  /* initial minor number */

static int dazukofs_group_open(struct inode *inode,
			       struct file *file)
{
	int id;
	struct group_device *gdev;

	id = MINOR(inode->i_rdev) - minor_start;
	gdev = &grp_devs[id];

	gdev->track = dazukofs_group_open_tracking(id);
	gdev->id = id;
	file->private_data = gdev;
	return 0;
}

static int dazukofs_group_release(struct inode *inode,
				  struct file *file)
{
	struct group_device *gdev;

	gdev = (struct group_device *) file->private_data;

	if (gdev->track)
		dazukofs_group_release_tracking(file, gdev->id);
	return 0;
}

static ssize_t dazukofs_group_read(struct file *file,
				   char __user *buffer, size_t length,
				   loff_t *pos)
{
#define DAZUKOFS_MIN_READ_BUFFER 43
	char tmp[DAZUKOFS_MIN_READ_BUFFER];
	ssize_t tmp_used;
	pid_t pid;
	int fd;
	int err;
	unsigned long event_id;
	struct group_device *gdev;
  
	gdev = (struct group_device *) file->private_data;

	if (*pos > 0)
		return 0;

	if (length < DAZUKOFS_MIN_READ_BUFFER)
		return -EINVAL;

	err = dazukofs_get_event(file, gdev->id, &event_id, &fd, &pid);
	if (err) {
		if (err == -ERESTARTSYS)
			return -EINTR;
		return err;
	}

	tmp_used = snprintf(tmp, sizeof(tmp)-1, "id=%lu\nfd=%d\npid=%d\n",
			    event_id, fd, pid);
	if (tmp_used >= sizeof(tmp))
		return -EINVAL;

	if (copy_to_user(buffer, tmp, tmp_used))
		return -EFAULT;

	*pos = tmp_used;

	return tmp_used;
}

static ssize_t dazukofs_group_write(struct file *file,
				    const char __user *buffer, size_t length,
				    loff_t *pos)
{
#define DAZUKOFS_MAX_WRITE_BUFFER 19
	char tmp[DAZUKOFS_MAX_WRITE_BUFFER];
	int response;
	unsigned long event_id;
	char *p;
	char *p2;
	int ret;
	struct group_device *gdev;

	gdev = (struct group_device *) file->private_data;

	if (length >= DAZUKOFS_MAX_WRITE_BUFFER)
		length = DAZUKOFS_MAX_WRITE_BUFFER - 1;

	if (copy_from_user(tmp, buffer, length))
		return -EFAULT;
	tmp[length] = 0;

	p = strstr(tmp, "id=");
	if (!p)
		return -EINVAL;
	event_id = simple_strtoul(p + 3, &p2, 10);

	/*
	 * checkpatch.pl recommends using strict_strtoul() instead of
	 * simple_strtoul(). However, we _want_ a function that stops
	 * on non-number characters rather than errors out.
	 */

	p = strstr(p2, "r=");
	if (!p)
		return -EINVAL;
	response = (*(p + 2)) - '0';

	ret = dazukofs_return_event(gdev->id, event_id, response);
	if (ret == 0) {
		*pos += length;
		ret = length;
	} else if (ret == -ERESTARTSYS) {
		ret = -EINTR;
	}

	return ret;
}

struct file_operations grp_fops = {
	.owner	   = THIS_MODULE,
	.open 	   = dazukofs_group_open,
	.release   = dazukofs_group_release,
	.read	   = dazukofs_group_read,
	.write 	   = dazukofs_group_write,
};


static void dazukofs_group_class_dev_release(struct class_device *cls_dev)
{
	// nothing
}

static void dazukofs_group_dev_release(struct device *dev)
{
	// Nothing
}

static int __dazukofs_group_dev_init(dev_t devt, struct class *cls, 
				     struct group_device *gdev, char *name)
{
	int err;
	
	cdev_init(&gdev->cdev, &grp_fops);
	gdev->cdev.owner = THIS_MODULE;

	err = cdev_add(&gdev->cdev, devt, GROUP_COUNT);

	if (err)
		return -1; 

	gdev->dev.parent = &platform_bus;
	gdev->dev.release = dazukofs_group_dev_release;
	strncpy(gdev->dev.bus_id, name, BUS_ID_SIZE);

	err = device_register(&gdev->dev);

	if (err) 
		goto err_out;

	gdev->cls_dev.release = dazukofs_group_class_dev_release;
	gdev->cls_dev.class = cls; 
	gdev->cls_dev.dev = &gdev->dev;
	strncpy(gdev->cls_dev.class_id, name, BUS_ID_SIZE);
	gdev->cls_dev.devt = devt;	
	
	err = class_device_register(&gdev->cls_dev); 

	if (err) 
		goto err_out2;

	return 0;

err_out2:
	device_unregister(&gdev->dev);
err_out:
	cdev_del(&gdev->cdev);

	return err;
}

int dazukofs_group_dev_init(int dev_major, int dev_minor_start,
			    struct class *dazukofs_class)
{
	int err;
	int i;
	dev_t devt;
	struct group_device *gdev;
	char name[BUS_ID_SIZE];

	grp_devs = kzalloc(sizeof(struct group_device) * GROUP_COUNT,
			   GFP_KERNEL);

	if (!grp_devs)
		return -ENOMEM;

	/* create group devices */
	for (i = 0; i < GROUP_COUNT; i++) {
		gdev = &grp_devs[i];
		devt = MKDEV(dev_major, dev_minor_start + i);
		snprintf(name, sizeof(name), "%s.%i", DEVICE_NAME, i);		
		err = __dazukofs_group_dev_init(devt, dazukofs_class, 
                                                gdev, name); 		
		if (err)
			goto undo;
	}

	minor_start = dev_minor_start;
	return i;

undo:
	while(i) {		
		i--;
		gdev = &grp_devs[i];
		class_device_unregister(&gdev->cls_dev);
		device_unregister(&gdev->dev);
		cdev_del(&gdev->cdev);
	}
	kfree(grp_devs);
	return err;
}

void dazukofs_group_dev_destroy(int dev_major, int dev_minor_start,
				int dev_minor_end,
				struct class *dazukofs_class)
{
	int i;

	for (i = 0; i < GROUP_COUNT; i++) {
		struct group_device *gdev;

		gdev = &grp_devs[i];

		class_device_unregister(&gdev->cls_dev);
		device_unregister(&gdev->dev);
		cdev_del(&gdev->cdev);
	}
	kfree(grp_devs);
}
