/*
  This file is a part of the nesccov extensions to the nesc compiler.

  Copyright(c) 2011-2014 University of Warsaw. All rights reserved.
  All rights reserved.

  Redistribution and use in source and binary forms, with or without 
  modification, are permitted provided that the following conditions 
  are met:

    * Redistributions of source code must retain the above copyright 
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright 
      notice, this list of conditions and the following disclaimer in 
      the documentation and/or other materials provided with the 
      distribution.
    * Neither the name of University of Warsaw nor the names of its 
      contributors may be used to endorse or promote products derived 
      from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include <nesccov-runtime.h>
#include <NescCovCondInfoSerialDump.h>
#include <nesccov-tos.h>



/**
 * An implementation of a dumper of condition coverage
 * information for nesccov that works over the serial
 * active message interface.
 *
 * @author Karol Strzelecki <ks249241@students.mimuw.edu.pl>
 * @author Konrad Iwanicki <iwanicki@mimuw.edu.pl>
 */
module NescCovCondInfoSerialDumpP @safe()
{
    provides
    {
        interface NescCovInfoDump as Dump;
    }
    uses
    {
        interface AMSend;
        interface Packet;
    }
}
implementation
{

    message_t   packet;
    bool        dumpStarted = FALSE;
    bool        dumpDoneNotified;
    bool        locked;

    uint32_t    bytes_sent_q;

    task void dumpStepTask();
    void fillMsg(NescCovCondInfoDumpSerialMsg * msg);



    command error_t Dump.startDump()
    {
        if (dumpStarted)
        {
            return EALREADY;
        }
        if (NESCCOV_MAX_COND_DUMP_DATA_SIZE < sizeof(uint32_t) || TM_COND_ARRAY_SIZE <= 0)
        {
            return ESIZE;
        }

        bytes_sent_q = 0;
        dumpDoneNotified = FALSE;
        locked = FALSE;
        dumpStarted = TRUE;

        post dumpStepTask();

        return SUCCESS;
    }



    task void dumpStepTask()
    {
        NescCovCondInfoDumpSerialMsg * msg =
                (NescCovCondInfoDumpSerialMsg *)call Packet.getPayload(
                        &packet,
                        sizeof(NescCovCondInfoDumpSerialMsg)
                );
    
        if (locked)
        {
            return;
        }
        if (msg == NULL)
        {
            signal Dump.dumpDone(ESIZE);
            return;
        }

        call Packet.clear(&packet);
        
        fillMsg(msg);

        if (call AMSend.send(AM_BROADCAST_ADDR, &packet, sizeof(NescCovCondInfoDumpSerialMsg)) == SUCCESS)
        {
            locked = TRUE;
	    }
        else
        {
            dumpStarted = FALSE;
            signal Dump.dumpDone(FAIL);
        }
    }



    inline uint8_t howMany()
    {
        uint32_t tmp = TM_COND_ARRAY_SIZE + sizeof(uint32_t);
        if (bytes_sent_q >= tmp)
        {
            return 0;
        }
        tmp -= bytes_sent_q;
        return (uint32_t)NESCCOV_MAX_COND_DUMP_DATA_SIZE > tmp ?
                (uint8_t)tmp : (uint8_t)NESCCOV_MAX_COND_DUMP_DATA_SIZE;
    }



    void fillMsg(NescCovCondInfoDumpSerialMsg * msg)
    {
        uint32_t   i;
        uint8_t    j;
        uint8_t    howMany_c = howMany();

        memset(msg, 0, sizeof(NescCovCondInfoDumpSerialMsg));
        j = 0;
        i = bytes_sent_q;
        if (bytes_sent_q == 0)
        {
            uint32_t tmpSz = TM_COND_ARRAY_SIZE;
            msg->data[0] = (uint8_t)(tmpSz >> 24);
            msg->data[1] = (uint8_t)(tmpSz >> 16);
            msg->data[2] = (uint8_t)(tmpSz >> 8);
            msg->data[3] = (uint8_t)(tmpSz);
            j += sizeof(uint32_t);
            i += sizeof(uint32_t);
        }
        for (i -= sizeof(uint32_t); j < howMany_c; ++i, ++j)
        {
            msg->data[j] = TM_COND_ARRAY_NAME[i];
        }
        msg->len = howMany_c;
        bytes_sent_q += msg->len;
    }



    event void AMSend.sendDone(message_t * bufPtr, error_t error) 
    {
        if (&packet == bufPtr)
        {
            locked = FALSE;
            if (error == SUCCESS)
            {
                if (bytes_sent_q < sizeof(uint32_t) + TM_COND_ARRAY_SIZE)
                {
                    post dumpStepTask();
                }
                else if (dumpDoneNotified)
                {
                    dumpStarted = FALSE;
            	    signal Dump.dumpDone(SUCCESS);
                }
                else
                {
                    dumpDoneNotified = TRUE;
                    post dumpStepTask();
                }
            }
            else
            {
                dumpStarted = FALSE;
                signal Dump.dumpDone(FAIL);
            }
        }
    }

}

