/*
  This file is a part of the nesccov extensions to the nesc compiler.

  Copyright(c) 2011-2014 University of Warsaw. All rights reserved.
  All rights reserved.

  Redistribution and use in source and binary forms, with or without 
  modification, are permitted provided that the following conditions 
  are met:

    * Redistributions of source code must retain the above copyright 
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright 
      notice, this list of conditions and the following disclaimer in 
      the documentation and/or other materials provided with the 
      distribution.
    * Neither the name of University of Warsaw nor the names of its 
      contributors may be used to endorse or promote products derived 
      from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
package net.tinyos.tools;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;

import net.tinyos.message.Message;
import net.tinyos.message.MessageListener;
import net.tinyos.message.MoteIF;
import net.tinyos.packet.BuildSource;
import net.tinyos.packet.PhoenixSource;
import net.tinyos.util.PrintStreamMessenger;



/**
 * A Java client for the nesccov dumpers working
 * over the serial active message stack.
 * 
 * @author Konrad Iwanicki <iwanicki@mimuw.edu.pl>
 */
public class NescCovSerialClient {

	private static MoteIF moteIf;
	
	private static void printInfo(PrintStream str) {
		str.println("nesccov: Code coverage assessment tool for nesc");
		str.println("         Java client for serial active message dumpers");
		str.println("Copyright (C) 2011-2014, University of Warsaw");
		str.println();
	}
	
	private static void printUsage(PrintStream str) {
		str.println("Usage:");
		str.println("  " + NescCovSerialClient.class.getName() + " [-comm <packet_source>]");
		str.println("Where <packet_source> can be one of the following:");
		str.println(BuildSource.sourceHelp());
		str.println();
	}
	
	private static void printWarning(PrintStream str, String warn) {
		str.println("Warning: " + warn + ".");
	}
	
	private static void printError(PrintStream str, String err) {
		str.println("ERROR: " + str + "!");
		str.println();
	}
	
	private static void backupFileIfNecessary(String fileName) {
		File fm = new File(fileName);
		if (fm.exists()) {
			File fb = new File(fileName + ".bak");
			if (fb.exists()) {
				fb.delete();
			}
			fm.renameTo(fb);
		}
	}
	
	private static void dumpMsgToStream(short[] data, int len, ByteArrayOutputStream str) {
		for (int i = 0; i < len; ++i) {
			str.write(data[i]);
		}
	}
	
	private static void dumpStreamToFile(ByteArrayOutputStream str, String fileName) {
		byte[] data = str.toByteArray();
		str.reset();
		System.out.println("Dumping a " + data.length + "-byte report to file \"" + fileName + "\"");
		try {
			FileOutputStream fstr = new FileOutputStream(fileName);
			fstr.write(data);
			fstr.close();
		} catch (FileNotFoundException e) {
			printWarning(System.err, "unable to create file \"" + fileName + "\"");
		} catch (IOException e) {
			printWarning(System.err, "I/O error writing file \"" + fileName + "\"");
		}
	}

	/**
	 * The main function.
	 * @param args The command-line arguments.
	 */
	public static void main(String[] args) {
		
		printInfo(System.out);
		
		String sourceName = null;
		if (args.length == 2) {
			if (!args[0].equals("-comm")) {
				printError(System.err, "invalid option \"" + args[0] + "\"");
				printUsage(System.err);
				System.exit(1);
			}
			sourceName = args[1];
		} else if (args.length != 0) {
			printError(System.err, "invalid arguments");
			printUsage(System.err);
			System.exit(1);
		}
		
		PhoenixSource phoenix;
		if (sourceName == null) {
		    phoenix = BuildSource.makePhoenix(PrintStreamMessenger.err);
		}
		else {
		    phoenix = BuildSource.makePhoenix(sourceName, PrintStreamMessenger.err);
		}
		if (phoenix == null)
		{
			printError(System.err, "invalid packet source");
			printUsage(System.err);
			System.exit(1);
		}
		moteIf = new MoteIF(phoenix);
		new LineCoverageClient(moteIf);
		new CondCoverageClient(moteIf);
	}

	/**
	 * A listener of line coverage messages.
	 */
	private static class LineCoverageClient implements MessageListener {

		private static final String FILE_NAME = "nesccov_line.exec";

		private MoteIF                  moteIf;
		private ByteArrayOutputStream   byteArrStr;
		
		public LineCoverageClient(MoteIF moteIfArg) {
			this.moteIf = moteIfArg;
			this.moteIf.registerListener(new NescCovLineInfoDumpSerialMsg(), this);
			this.byteArrStr = new ByteArrayOutputStream();
		}
		
		public void messageReceived(int to, Message m) {
			if (!(m instanceof NescCovLineInfoDumpSerialMsg))
			{
				printWarning(System.err, "expected a line coverage " +
						"message but received another one with AM ID " +
						m.amType());
				return;
			}
			NescCovLineInfoDumpSerialMsg msg =
					(NescCovLineInfoDumpSerialMsg)m;
			System.out.println("Received a line coverage message of size " +
					msg.get_len() + ".");
			if (msg.get_len() == 0) {
				backupFileIfNecessary(FILE_NAME);
				dumpStreamToFile(this.byteArrStr, "nesccov_line.exec");
			} else {
				dumpMsgToStream(msg.get_data(), msg.get_len(), this.byteArrStr);
			}
		}
		
	}
	
	/**
	 * A listener of condition coverage messages.
	 */
	private static class CondCoverageClient implements MessageListener {

		private static final String FILE_NAME = "nesccov_cond.exec";
		
		private MoteIF                  moteIf;
		private ByteArrayOutputStream   byteArrStr;
		
		public CondCoverageClient(MoteIF moteIfArg) {
			this.moteIf = moteIfArg;
			this.moteIf.registerListener(new NescCovCondInfoDumpSerialMsg(), this);
			this.byteArrStr = new ByteArrayOutputStream();
		}
		
		public void messageReceived(int to, Message m) {
			if (!(m instanceof NescCovCondInfoDumpSerialMsg))
			{
				printWarning(System.err, "expected a line coverage " +
						"message but received another one with AM ID " +
						m.amType());
				return;
			}
			NescCovCondInfoDumpSerialMsg msg =
					(NescCovCondInfoDumpSerialMsg)m;
			System.out.println("Received a condition coverage message of size " +
					msg.get_len() + ".");
			if (msg.get_len() == 0) {
				backupFileIfNecessary(FILE_NAME);
				dumpStreamToFile(this.byteArrStr, FILE_NAME);
			} else {
				dumpMsgToStream(msg.get_data(), msg.get_len(), this.byteArrStr);
			}
		}
		
	}
}
