/*
  This file is a part of the nesccov extensions to the nesc compiler.

  Copyright(c) 2011-2014 University of Warsaw. All rights reserved.
  All rights reserved.

  Redistribution and use in source and binary forms, with or without 
  modification, are permitted provided that the following conditions 
  are met:

    * Redistributions of source code must retain the above copyright 
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright 
      notice, this list of conditions and the following disclaimer in 
      the documentation and/or other materials provided with the 
      distribution.
    * Neither the name of University of Warsaw nor the names of its 
      contributors may be used to endorse or promote products derived 
      from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/


/**
 * The main module of the application demonstrating
 * the use of the nesccov functionality.
 *
 * @author Konrad Iwanicki <iwanicki@mimuw.edu.pl>
 */
module NescCovDemoMainP @safe()
{
    uses
    {
        interface Boot;
        interface Timer<TMilli> as FrequentTimer;
        interface Timer<TMilli> as InfrequentTimer;
        interface Timer<TMilli> as InactiveTimer;
        interface Leds;
#if (defined(NESCCOV_LINE_COVERAGE) || defined(NESCCOV_COND_COVERAGE))
        // Dumps the run-time coverage information,
        // either periodically or on demand.
        interface NescCovInfoDumper;
#endif
    }
}
implementation
{

    uint8_t   counterEqualToZeroOrOne = 0;

    task void coveredTask();
    task void uncoveredTask();
    void coveredFunction();
    void uncoveredFunction();


    event void Boot.booted()
    {
#if (defined(NESCCOV_LINE_COVERAGE) || defined(NESCCOV_COND_COVERAGE))
        // Initialize periodic dumping of the coverage data.
        call NescCovInfoDumper.setInterval(2048);
#endif

        call FrequentTimer.startPeriodic(256);
        call InfrequentTimer.startPeriodic(512);
        if (counterEqualToZeroOrOne > 0)
        {
            // The code below should not be covered.
            call InactiveTimer.startPeriodic(1024);
            post uncoveredTask();
        }
    }


    event void FrequentTimer.fired()
    {
        // This event should be covered.
        coveredFunction();
    }


    event void InfrequentTimer.fired()
    {
        // This event should be covered.
        call Leds.led2Toggle();
        post coveredTask();
    }


    task void coveredTask()
    {
        counterEqualToZeroOrOne = 1 - counterEqualToZeroOrOne;
        counterEqualToZeroOrOne = 1 - counterEqualToZeroOrOne;
    }


    void coveredFunction()
    {
        ++counterEqualToZeroOrOne;
        if (counterEqualToZeroOrOne == 1 || counterEqualToZeroOrOne == 2)
        {
            call Leds.led1Toggle();
            if (counterEqualToZeroOrOne == 2)
            {
                counterEqualToZeroOrOne = 0;
            }
        }
        if (counterEqualToZeroOrOne > 1)
        {
            // This code should not be covered.
            counterEqualToZeroOrOne = 0;
        }
    }


    event void InactiveTimer.fired()
    {
        // This code should not be covered.
        post uncoveredTask();
    }


    task void uncoveredTask()
    {
        // This task should never be posted,
        // hence its code should not be covered.
        call Leds.led0Toggle();
        if (counterEqualToZeroOrOne == 0)
        {
            uncoveredFunction();
        }
    }


    void uncoveredFunction()
    {
        // This function should be completely uncovered.
        counterEqualToZeroOrOne = 1;        
    }
}

