#include "EnumSerialization.hpp"
#include "WeaponModel.hpp"

Weapon::Weapon(UID uid)
	: uid_(uid)
{
}

QString Weapon::weaponTypeName() const
{
	static const QMap <WeaponType, QString> weaponTypeName_ {
		{WeaponType::Sword, QObject::tr("Sword")},
		{WeaponType::Mace, QObject::tr("Mace")},
		{WeaponType::Bow, QObject::tr("Bow")},
		{WeaponType::Axe, QObject::tr("Axe")},
		{WeaponType::Crossbow, QObject::tr("Crossbow")}
	};

	return weaponTypeName_[weaponType_];
}

QDataStream & operator <<(QDataStream &out, const Weapon &weapon)
{
	out << weapon.name_ << toUnderlying(weapon.weaponType_)
	    << weapon.damage_ << weapon.weight_;
	return out;
}

QDataStream & operator >>(QDataStream &in, Weapon &weapon)
{
	in >> weapon.name_ >> toUnderlyingRef(weapon.weaponType_)
	   >> weapon.damage_ >> weapon.weight_;
	return in;
}

WeaponModel::WeaponModel(QObject *parent)
	: QAbstractTableModel(parent)
{
}

Qt::ItemFlags WeaponModel::flags(const QModelIndex &index) const
{
	if (!index.isValid())
		return Qt::ItemIsEnabled;
	if (index.column() == WeaponType)
		return Qt::ItemIsEnabled | Qt::ItemIsSelectable;
	return Qt::ItemIsEnabled | Qt::ItemIsSelectable | Qt::ItemIsEditable;
}

int WeaponModel::columnCount(const QModelIndex &) const
{
	return ColumnCount;
}

int WeaponModel::rowCount(const QModelIndex &) const
{
	return weaponData.count();
}

QVariant WeaponModel::data(const QModelIndex &index, int role) const
{
	if (!index.isValid() || (role != Qt::DisplayRole && role != Qt::EditRole))
		return QVariant();

	const Weapon &weapon = weaponData[index.row()];
	switch (index.column()) {
		case Name: return weapon.name();
		case WeaponType: return weapon.weaponTypeName();
		case Damage: return weapon.damage();
		case Weight: return weapon.weight();
	}

	return QVariant();
}

bool WeaponModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
	if (role != Qt::EditRole)
		return false;

	Weapon &weapon = weaponData[index.row()];
	switch (index.column()) {
		case Name: weapon.setName(value.toString()); break;
		case Damage: weapon.setDamage(value.toInt()); break;
		case Weight: weapon.setWeight(value.toInt()); break;
		default: return false;
	}

	emit dataChanged(index, index);
	return true;
}

QVariant WeaponModel::headerData(int section, Qt::Orientation orientation, int role) const
{
	if (role != Qt::DisplayRole || orientation != Qt::Horizontal)
		return QVariant();

	switch (section) {
		case Name: return tr("Name");
		case WeaponType: return tr("Type");
		case Damage: return tr("Damage");
		case Weight: return tr("Weight");
	}

	return QVariant();
}

bool WeaponModel::insertRows(int row, int count, const QModelIndex &)
{
	beginInsertRows(QModelIndex(), row, row + count - 1);
	for (int i = 0; i < count; ++i) {
		Weapon w(serial_.next());
		w.setName("Default");
		w.setWeaponType(Weapon::WeaponType::Axe);
		w.setWeight(50);
		w.setDamage(25);

		weaponData.insert(row, w);
	}
	endInsertRows();

	return true;
}

bool WeaponModel::removeRows(int row, int count, const QModelIndex &)
{
	beginRemoveRows(QModelIndex(), row, row + count - 1);
	weaponData.remove(row, count);
	endRemoveRows();

	return true;
}

