#ifndef MonsterModel_hpp
#define MonsterModel_hpp

#include <QtWidgets>

#include "WeaponModel.hpp"

class MonsterBase {
public:
	enum class Part : quint8 {
		Head,
		LeftArm,
		RightArm,
		Torso
	};

	QString name() const {return name_;}
	void setName(QString name) {name_ = name;}

	int hitPoints() const {return hitPoints_;}
	void setHitPoints(int hitPoints) {hitPoints_ = hitPoints;}

	static QString monsterPartName(MonsterBase::Part monsterPart)
	{
		static QMap <MonsterBase::Part, QString> monsterPartName_ {
			{Part::Head,     QObject::tr("Head")},
			{Part::LeftArm,  QObject::tr("Left arm")},
			{Part::RightArm, QObject::tr("Right arm")},
			{Part::Torso,    QObject::tr("Torso")},
		};

		return monsterPartName_[monsterPart];
	}

private:
	QString name_;
	int hitPoints_;
	QMap <MonsterBase::Part, const Weapon *> equipment_;
};

class MonsterModel : public QAbstractItemModel {
public:
	enum class Level : quint8 {
		Root,
		Monster,
		MonsterPart,
		Equipment
	};

private:
	class MonsterModelNode {
	public:
		MonsterModelNode(MonsterModelNode *parent, void *data, Level level);
		MonsterModelNode(MonsterModelNode *parent, const void *data, Level level);
		~MonsterModelNode();

		void addChild(int idx, MonsterModelNode *child);
		void appendChild(MonsterModelNode *child);
		MonsterModelNode * child(int idx) const;
		int childrenCount() const;
		void deleteChild(int idx);
		MonsterModelNode * parent() const;
		int row() const;

		Level level() const;
		MonsterBase * monster() const;
		QString monsterPart() const;
		const Weapon * weapon() const;

	private:
		MonsterModelNode *parent_;
		QList <MonsterModelNode *> children_;
		void *data_;
		const void *constData_;
		Level level_;
	};

	MonsterModelNode *treeRoot_;

public:
	struct Monster {
		static const int Name        = 0;
		static const int HitPoints   = 1;
		static const int ColumnCount = 2;
	};

	struct MonsterPart {
		static const int Name        = 0;
		static const int ColumnCount = 1;
	};

	MonsterModel(QObject *parent = nullptr);
	~MonsterModel();

	QModelIndex index(int row, int column, const QModelIndex &parent) const;
	QModelIndex parent(const QModelIndex &index) const;

	int columnCount(const QModelIndex &parent) const;
	int rowCount(const QModelIndex &parent) const;

	Qt::ItemFlags flags(const QModelIndex &index) const;
	QVariant data(const QModelIndex &index, int role = Qt::DisplayRole) const;
	bool setData(const QModelIndex &index, const QVariant &value, int role = Qt::EditRole);

private:
	void clearModel();
	void initModel();

	Level level(const QModelIndex &index) const;
	MonsterModelNode * node(const QModelIndex &index) const;

	QList <Monster *> monsterList_;
};

#endif
