#include "EnumSerialization.hpp"
#include "MonsterModel.hpp"

/********************
 * MonsterModelNode *
 ********************/

MonsterModel::MonsterModelNode::MonsterModelNode(MonsterModelNode *parent, void *data, Level level)
	: parent_(parent), data_(data), constData_(nullptr), level_(level)
{
}

MonsterModel::MonsterModelNode::MonsterModelNode(MonsterModelNode *parent, const void *constData, Level level)
	: parent_(parent), data_(nullptr), constData_(constData), level_(level)
{
}

MonsterModel::MonsterModelNode::~MonsterModelNode()
{
	qDeleteAll(children_);
}

void MonsterModel::MonsterModelNode::addChild(int idx, MonsterModel::MonsterModelNode *child)
{
	children_.insert(idx, child);
}

void MonsterModel::MonsterModelNode::appendChild(MonsterModel::MonsterModelNode *child)
{
	children_.append(child);
}

MonsterModel::MonsterModelNode * MonsterModel::MonsterModelNode::child(int idx) const
{
	return children_.at(idx);
}

int MonsterModel::MonsterModelNode::childrenCount() const
{
	return children_.count();
}

void MonsterModel::MonsterModelNode::deleteChild(int idx)
{
	delete children_.takeAt(idx);
}

MonsterModel::MonsterModelNode * MonsterModel::MonsterModelNode::parent() const
{
	return parent_;
}

int MonsterModel::MonsterModelNode::row() const
{
	//typeof(this) == const MonsterModel::MonsterModelNode *const
	return parent()->children_.indexOf(const_cast<MonsterModelNode *>(this));
}

MonsterModel::Level MonsterModel::MonsterModelNode::level() const
{
	return level_;
}

MonsterBase * MonsterModel::MonsterModelNode::monster() const
{
	return static_cast<MonsterBase *>(data_);
}

QString MonsterModel::MonsterModelNode::monsterPart() const
{
	MonsterBase::Part monsterPart =
		*static_cast<const MonsterBase::Part *>(constData_);
	return MonsterBase::monsterPartName(monsterPart);
}

const Weapon * MonsterModel::MonsterModelNode::weapon() const
{
	return static_cast<const Weapon *>(constData_);
}

/****************
 * MonsterModel *
 ****************/

MonsterModel::MonsterModel(QObject *parent)
	: QAbstractItemModel(parent)
{
	initModel();
}

MonsterModel::~MonsterModel()
{
	clearModel();
}

QModelIndex MonsterModel::index(int row, int column, const QModelIndex &parent) const
{
	if (!hasIndex(row, column, parent))
		return QModelIndex();

	MonsterModelNode *v = node(parent);
	return createIndex(row, column, v->child(row));
}

QModelIndex MonsterModel::parent(const QModelIndex &index) const
{
	if (!index.isValid())
		return QModelIndex();

	MonsterModelNode *v = node(index);
	if (v->parent() == treeRoot_)
		return QModelIndex();
	return createIndex(v->parent()->row(), 0, v->parent());
}

int MonsterModel::columnCount(const QModelIndex &parent) const
{
	switch (level(parent)) {
		case Level::Root:        return Monster::ColumnCount;
		case Level::MonsterPart: return MonsterPart::ColumnCount;
		case Level::Equipment:   return WeaponModel::ColumnCount;
		default:                 return 0;
	}
}

int MonsterModel::rowCount(const QModelIndex &parent) const
{
	return node(parent)->childrenCount();
}

Qt::ItemFlags MonsterModel::flags(const QModelIndex &index) const
{
	if (!index.isValid())
		return Qt::ItemIsEnabled;

	Qt::ItemFlags defaultFlags = Qt::ItemIsEnabled | Qt::ItemIsSelectable;

	if (level(index) == Level::Equipment)
		return defaultFlags | Qt::ItemNeverHasChildren;
	return defaultFlags;
}

QVariant MonsterModel::data(const QModelIndex &index, int role) const
{
	if (role != Qt::DisplayRole && role != Qt::EditRole)
		return QVariant();

	const MonsterModelNode *v = node(index);
	switch (v->level()) {
		case Level::Monster: {
			const MonsterBase *monster = v->monster();
			switch (index.column()) {
				case Monster::Name:      return monster->name();
				case Monster::HitPoints: return monster->hitPoints();
			}
			break;
		}

		case Level::MonsterPart: {
			switch (index.column()) {
				case MonsterPart::Name: return v->monsterPart();
			}
			break;
		}

		case Level::Equipment: {
			const Weapon *weapon = v->weapon();
			switch (index.column()) {
				case WeaponModel::Name:       return weapon->name();
				case WeaponModel::WeaponType: return toUnderlying(weapon->weaponType());
				case WeaponModel::Damage:     return weapon->damage();
				case WeaponModel::Weight:     return weapon->weight();
			}
			break;
		}

		default:
			return QVariant();
	}

	return QVariant();
}

bool MonsterModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
	if (!index.isValid())
		return false;

	MonsterModelNode *v = node(index);
	switch (v->level()) {
		case Level::Monster: {
			MonsterBase *monster = v->monster();
			switch (index.column()) {
				case Monster::Name:      monster->setName(value.toString()); break;
				case Monster::HitPoints: monster->setHitPoints(value.toInt()); break;
				default:                 return false;
			}
		}

		default:
			return false;
	}

	emit dataChanged(index, index);
	return true;
}

void MonsterModel::clearModel()
{
	qDeleteAll(monsterList_);
	monsterList_.clear();
	delete treeRoot_;
}

void MonsterModel::initModel()
{
	treeRoot_ = new MonsterModelNode(nullptr, (const void *)nullptr, Level::Root);
}

MonsterModel::Level MonsterModel::level(const QModelIndex &index) const
{
	if (!index.isValid())
		return Level::Root;
	return node(index)->level();
}

MonsterModel::MonsterModelNode * MonsterModel::node(const QModelIndex &index) const
{
	if (!index.isValid())
		return treeRoot_;
	return static_cast<MonsterModelNode *>(index.internalPointer());
}
