#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/syscall.h>
#include <signal.h>
#include <sys/reg.h>
#include "esyscall.h"
#include <fcntl.h>

#include "supervisor.h"
#include "prototypes.h"

#define MAX_FAILED_READS 100000

typedef int (*quirk_fn)(long);
static quirk_fn quirks_table[MAX_SYSNR];

static int failed_reads_counter;

int execve_called;

int quirk_syscall(long sysnr)
{
    quirk_fn handler = quirks_table[sysnr];
    if (handler == NULL)
        return 1;
    dbgprintf(stderr, "quirking %s\n", syscall_name(sysnr));
    return handler(sysnr);
}

static int quirk_execve(long sysnr)
{
    if (execve_called > 2)
    {
        skip_syscall();
        forbidden("too many execve calls");
    }
    if (execve_called < 2)
        sc_state = SS_IN;
    else
    {
        sc_state = SS_OUT;
        track_exectime();
        zero_exectime();
    }
    execve_called++;
    return 0;
}

static int quirk_kill(long sysnr)
{
    int sig;
    if (sc_state != SS_OUT) return 1;
    if (uread(EBX) != tracked)
    {
        skip_syscall();
        forbidden("tried to kill some process");
    }
    sig = uread(ECX);
    if (sig == SIGSTOP || sig == SIGTRAP || sig == SIGVTALRM)
    {
        skip_syscall();
        forbidden("used forbidden signal");
    }
    return 1;
}

static int quirk_read(long sysnr)
{
    size_t count;
    if (sc_state == SS_IN) {
        ssize_t read = uread(EAX);

        if (read <= 0) {
			if (read == 0)
				dbgprintf(stderr, "read eof\n");
			else
				dbgprintf(stderr, "read error: %s\n", strerror(-read));

			failed_reads_counter++;
			if (failed_reads_counter > MAX_FAILED_READS) {
				report_result(RETVAL_RE, "reading past the end of input");
				kill_and_exit();
			}
        }
    }
	return 1;
}

static int quirk_write(long sysnr)
{
    size_t count;
    if (sc_state == SS_IN) {
        ssize_t written = uread(EAX);

        

        if (written < 0) {
            dbgprintf(stderr, "write error: %s\n", strerror(-written));
            return 1;
        }
        out_limit -= written;
        dbgprintf(stderr, "written %ld chars, %ld left\n", (long)written, out_limit);
        return 1;
    }

    count = (size_t)uread(EDX);

    if (count < 0)
    {
        skip_syscall();
        forbidden("fiddling with write");
    }
    if (count > out_limit)
    {
        skip_syscall();
        report_result(RETVAL_OLE, "output limit exceeded");
        kill_and_exit();
    }
    return 1;
}

static int quirk_open(long sysnr)
{
	/* OI quirk: if user wants to open something with relative address,
	 * then bail out now. If not, then probably this is library looking
	 * in proc. Therefore if we see an absolute address, then we just
	 * skip the call. */

	const char* usraddr;
    char path[PATH_MAX];
    char cpath[PATH_MAX];
    int flags;
    const char** allowed_dirs_iter;

	if (sc_state != SS_OUT)
		return 1;

	usraddr = (const char*)uread(0);
    upeekstring(usraddr, path, sizeof(path));
    flags = (int)uread(1);

    dbgprintf(stderr, "open %05x %s\n", flags, path);
    if (path[0] != '/') {
        skip_syscall();
		forbidden("opening files is forbidden");
        return 0;
    }
    if (realpath(path, cpath) == NULL) {
        /* Emulate the same return code. */
        emulate_syscall(-errno);
        return 0;
    }

    dbgprintf(stderr, "  cpath %s\n", cpath);

    if (flags & O_WRONLY || flags & O_RDWR) {
        if (!strcmp("/dev/null", cpath)) {
            /* /dev/null is safe and used by java! */
            return 1;
        }

        dbgprintf(stderr, "SKIPPING OPEN (requested writing)\n");
        skip_syscall();
        return 0;
    }

    for (allowed_dirs_iter = allowed_open_dirs; *allowed_dirs_iter;
            allowed_dirs_iter++)
        if (!strncmp(cpath, *allowed_dirs_iter, strlen(*allowed_dirs_iter)))
            return 1;

    skip_syscall();
    return 0;
}

static int quirk_exit(long sysnr)
{
    track_exectime();
    return 1;
}

static int sys_svtest(long sysnr)
{
    emulate_syscall(0);
    return 0;
}

static quirk_fn quirks_table[MAX_SYSNR] = {
    [__NR_execve]          = quirk_execve,
    [__NR_kill]            = quirk_kill,
    [__NR_tkill]           = quirk_kill,
    [__NR_tgkill]          = quirk_kill,
    [__NR_read]            = quirk_read,
    [__NR_write]           = quirk_write,
    [__NR_exit]            = quirk_exit,
    [__NR_exit_group]      = quirk_exit,
    [__NR_open]            = quirk_open,
    //[__NR_rt_sigreturn]    = quirk_rt_sigreturn,

    [__NRe_ioshmget]       = sys_ioshmget,
    [__NRe_ioshmat]        = sys_ioshmat,
    [__NRe_iochunk]        = sys_iochunk,
    [__NRe_ioflush]        = sys_ioflush,
    [__NRe_svtest]         = sys_svtest,
};

