#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/select.h>
#include <unistd.h>
#include <errno.h>

#include "iomap.h"

void* iomap_data;
int   iomap_size;

void* iobuf_end;
int   iobuf_size;

extern inline int is_whitespace(unsigned char c);

static void svioparse_err_write(const char* msg)
{
    write(2, msg, strlen(msg));
}

static void svioparse_fail(const char* msg)
{
    svioparse_err_write("libsvio fatal error: ");
    svioparse_err_write(msg);
    svioparse_err_write("\n");
    abort();
}

static void read_entire_stdin(void)
{
    size_t bufsize = 2048;
    size_t off = 0;
    void* buf = malloc(bufsize);
    if (buf == NULL)
        svioparse_fail("OOM allocating input buffer");
    for (;;) {
        ssize_t r = read(0, buf+off, bufsize-off);
        if (r == 0) break;
        if (r < 0)
            svioparse_fail("error reading from stdin");
        off += r;
        if (off == bufsize) {
            bufsize *= 2;
            buf = realloc(buf, bufsize);
            if (buf == NULL)
                svioparse_fail("OOM while growing input buffer");
        }
    }
    iomap_data = buf;
    iomap_size = off;
    iobuf_size = bufsize;
}

void iomap_entire_stdin(void)
{
    struct stat st;
    int res, ps;
    void* mem;

    ps = getpagesize();

    res = fstat(0, &st);
    if (res < 0)
        svioparse_fail("cannot fstat() standard input");

    mem = mmap(NULL, (st.st_size/ps + 1)*ps, PROT_READ, MAP_SHARED, 0, 0);
    if (mem == (void*)-1) {
        read_entire_stdin();
        return;
    }
    iomap_data = mem;
    iomap_size = st.st_size;
}

static void grow_buf(void)
{
    iobuf_size *= 2;
    iomap_data = realloc(iomap_data, iobuf_size);

    if (iomap_data == NULL)
        svioparse_fail("OOM while growing line buffer");
}

static int read_status(int fd)
{
    fd_set rs, ws, es;
    int err;
    struct timeval tv;

    tv.tv_sec = 0;
    tv.tv_usec = 0;

    FD_ZERO(&rs);
    FD_ZERO(&ws);
    FD_ZERO(&es);
    FD_SET(fd, &rs);
    FD_SET(fd, &es);

    err = select(fd+1, &rs, &ws, &es, &tv);
    if (err < 0) {
        if (errno == EINTR)
            return read_status(fd);
        svioparse_fail("select() failed");
    }
    if (FD_ISSET(fd, &es))
        svioparse_fail("error on input file descriptor");

    return FD_ISSET(fd, &rs);
}

int read_some_stdin(void)
{
    int r = 0;
    int has_entire_token = 0;
    char* end_of_token;

    if (iobuf_size == 0) {
        iobuf_size = 2048;
        iomap_data = malloc(iobuf_size);
        if (iomap_data == NULL)
            svioparse_fail("OOM allocating line buffer");
    }

    if (iomap_data + iomap_size < iobuf_end) {
        r = iobuf_end - iomap_data - iomap_size;
        memmove(iomap_data, iomap_data + iomap_size, r);
    }

    for (;;) {
        char* cursor = iomap_data+r;
        char* limit;
        ssize_t rr;
        
        if (!read_status(0) && has_entire_token)
            break;

        rr = read(0, cursor, iobuf_size-r);
        if (rr < 0)
            svioparse_fail("error reading");
        else if (rr == 0) {
            end_of_token = cursor;
            break;
        }

        r += rr;
        limit = cursor + rr;
        while (cursor < limit && is_whitespace(*cursor))
            cursor++;
        while (cursor < limit) {
            if (is_whitespace(*cursor)) {
                has_entire_token = 1;
                end_of_token = cursor+1;
            }
            cursor++;
        }

         if (r == iobuf_size) {
             if (has_entire_token)
                 break;
             grow_buf();
         }
    }

    iomap_size = (void*)end_of_token - iomap_data;
    iobuf_end = iomap_data + r;

    if (r == 0)
        return -1;
    return 0;
}
