#define __USE_GNU

#include <assert.h>
#include <sys/types.h>
#include <sys/select.h>
#include <sys/stat.h>

#include <sys/syscall.h>
#include "syscall.h"

#define NULL ((void*)0)

int errno;
void* stderr;
void* stdout;

static void* malloc_block;
static int malloc_size;

struct rlimit;

_syscall1(int, exit, int, status);
_syscall3(ssize_t, read, int, fd, void*, buf, size_t, nbytes);
_syscall3(ssize_t, write, int, fd, const void*, buf, size_t, count);
_syscall6(void*, mmap2, void*, start, size_t, length, int, prot,
        int, flags, int, fd, off_t, offset);
_syscall2(int, munmap, void*, start, size_t, length);
_syscall4(void*, mremap, void*, old_address, size_t, old_size,
        size_t, new_size, unsigned long, flags);
_syscall2(int, getrlimit, int, resource, struct rlimit*, rlim);
_syscall2(int, setrlimit, int, resource, const struct rlimit*, rlim);
_syscall5(int, _newselect, int, n, fd_set*, readfds, fd_set*, writefds,
        fd_set*, exceptfds, struct timeval*, timeout);
_syscall2(int, fstat, int, filedes, struct stat*, buf);
_syscall3(int, execve, const char*, filename, char *const *, argv,
        char* const*, envp);
_syscall3(int, open, const char*, pathname, int, flags, mode_t, mode);
_syscall1(int, close, int, fd);
_syscall0(pid_t, getpid);


int memcmp(const void* a, const void* b, size_t len)
{
    int i;
    for (i=0; i<len; i++)
        if (*(char*)(a++) != *(char*)(b++))
            return 1;
    return 0;
}

void* memmove(void* dest, const void* src, size_t len)
{
    int i;

    if (dest == src)
        return dest;

    for (i=0; i<len; i++)
        *(char*)(dest++) = *(char*)(src++);

    return dest;
}

void* memcpy(void* dest, const void* src, size_t len)
{
	return memmove(dest, src, len);
}

size_t strlen(const char* str)
{
    size_t r = 0;
    while (*(str++)) r++;
    return r;
}

long long int strtoll(const char* ptr, char** endptr, int base)
{
    long long ret = 0;
    int neg = 0;

    assert(base == 10 || base == 0);

    if (*ptr == '-') {
        neg = 1;
        ptr++;
    }

    while (*ptr) {
        if (*ptr < '0' || *ptr > '9')
            break;
        ret = ret*10 + (*ptr - '0');
        ptr++;
    }

    if (endptr)
        *endptr = (char*)ptr;

    return neg ? -ret : ret;
}

int printf(const char* format, unsigned long long l)
{
    /* assuming "%lld " */

    char buf[30];
    char* f = buf+sizeof(buf)-1;

    *(f--) = '\0';
    *(f--) = ' ';
    if (l == 0)
        *(f--) = '0';
    else {
        while (l) {
            *(f--) = (l%10) + '0';
            l /= 10;
        }
    }

    write(1, f+1, strlen(f+1));

    return 1;
}

int getpagesize(void)
{
    return 4096;
}

int* __errno_location(void)
{
    return &errno;
}

void _IO_putc(char c, void* file)
{
    write(2, &c, 1);
}

void fputc(char c, void* file)
{
    write(2, &c, 1);
}

void fflush(void* file)
{
}

int fwrite(const char* buf, int len, int size, void* file)
{
    if (write(1, buf, len*size) < 0)
        return -1;
    return 0;
}

int fprintf(void* file, const char* format, ...)
{
    write(2, format, strlen(format));
    return 0;
}

void abort(void)
{
    write(2, "Aborted\n", 8);

    for (;;)
        exit(1);
}

void __assert_fail(const char* cond, const char* file,
        unsigned int line, const char* function)
{
    write(2, "Assertion '", 11);
    write(2, cond, sizeof(cond));
    write(2, "' failed.\n", 10);
    abort();
}

static int size_in_pages(size_t size)
{
    return (size + 4095)/4096*4096;
}

void* mmap(void* start, size_t length, int prot,
        int flags, int fd, off_t offset)
{
    return mmap2(start, length, prot, flags, fd, offset);
}

int select(int n, fd_set* readfds, fd_set* writefds,
        fd_set* exceptfds, struct timeval* timeout)
{
    return _newselect(n, readfds, writefds, exceptfds, timeout);
}

void* malloc(size_t size)
{
    assert(malloc_block == NULL);

    int sizep = size_in_pages(size);

    malloc_block = mmap2(NULL, size_in_pages(sizep), 
            3 /*PROT_READ|PROT_WRITE*/, 0x22 /*MAP_PRIVATE|MAP_ANONYMOUS*/,
            0, 0);
    malloc_size = sizep;

    if (malloc_block == (void*)-1)
        return NULL;
    return malloc_block;
}

void* realloc(void* buf, size_t size)
{
    assert(malloc_block == buf);

    int sizep = size_in_pages(size);

    malloc_block = mremap(malloc_block, malloc_size,
            sizep, 1 /*MREMAP_MAYMOVE*/);
    malloc_size = sizep;

    if (malloc_block == (void*)-1)
        return NULL;

    return malloc_block;
}

void free(void* buf)
{
    munmap(malloc_block, malloc_size);
}
