#include <stdlib.h>
#include <sys/syscall.h>
#include <signal.h>
#include <sys/reg.h>
#include "esyscall.h"

#include "supervisor.h"
#include "prototypes.h"

typedef int (*quirk_fn)(long);
static quirk_fn quirks_table[MAX_SYSNR];

int execve_called;

int quirk_syscall(long sysnr)
{
    quirk_fn handler = quirks_table[sysnr];
    if (handler == NULL)
        return 1;
    return handler(sysnr);
}

static int quirk_execve(long sysnr)
{
    if (execve_called > 2)
    {
        skip_syscall();
        forbidden("too many execve calls");
    }
    if (execve_called != 2)
        sc_state = SS_IN;
    else
    {
        sc_state = SS_OUT;
        track_exectime();
        zero_exectime();
    }
    execve_called++;
    return 0;
}

static int quirk_kill(long sysnr)
{
    int sig;
    if (sc_state != SS_OUT) return 1;
    if (uread(EBX) != tracked)
    {
        skip_syscall();
        forbidden("tried to kill some process");
    }
    sig = uread(ECX);
    if (sig == SIGSTOP || sig == SIGTRAP || sig == SIGVTALRM)
    {
        skip_syscall();
        forbidden("used forbidden signal");
    }
    return 1;
}

static int quirk_write(long sysnr)
{
    size_t count;
    if (sc_state == SS_IN) {
        ssize_t written = uread(EAX);

        if (written < 0) {
            dbgprintf(stderr, "write error: %s\n", strerror(-written));
            return 1;
        }
        out_limit -= written;
        dbgprintf(stderr, "written %ld chars, %ld left\n", (long)written, out_limit);
        return 1;
    }
    count = (size_t)uread(EDX);
    if (count < 0)
    {
        skip_syscall();
        forbidden("fiddling with write");
    }
    if (count > out_limit)
    {
        skip_syscall();
        report_result(RETVAL_OLE, "output limit exceeded");
        kill_and_exit();
    }
    return 1;
}

static int quirk_exit(long sysnr)
{
    track_exectime();
    return 1;
}

static int sys_svtest(long sysnr)
{
    emulate_syscall(0);
    return 0;
}

static quirk_fn quirks_table[MAX_SYSNR] = {
    [__NR_execve]     = quirk_execve,
    [__NR_kill]       = quirk_kill,
    [__NR_tkill]      = quirk_kill,
    [__NR_tgkill]     = quirk_kill,
    [__NR_write]      = quirk_write,
    [__NR_exit]       = quirk_exit,
    [__NR_exit_group] = quirk_exit,

    [__NRe_ioshmget]    = sys_ioshmget,
    [__NRe_ioshmat]     = sys_ioshmat,
    [__NRe_iochunk]     = sys_iochunk,
    [__NRe_ioflush]     = sys_ioflush,
    [__NRe_svtest]      = sys_svtest,
};

