#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/ptrace.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>

static pid_t child;

static void fail_msg(const char* str)
{
	fprintf(stderr, "svloader: %s (errno='%s')\n", str, strerror(errno));
}

static void fail(const char* str)
{
	fail_msg(str);
    kill(getppid(), SIGABRT);
	exit(121);
}

static char* get_env_string(const char* name, const char* name2, const char* dflt)
{
	char* ret;
	char* t = getenv(name);
	if (!t)
		if (!name2 || !(t = getenv(name2)))
			return strdup(dflt);
	ret = strdup(t);
	unsetenv(name);
	return ret;
}

static long long get_env_number(const char* name, const char* name2, long dflt)
{
	long long ret;
	char* t = getenv(name);
	if (!t)
		if (!name2 || !(t = getenv(name2)))
			return dflt;
	ret = strtoll(t, NULL, 0);
	unsetenv(name);
	return ret;
}

static void tracked_process(void)
{
	struct rlimit rl;
	int fd;

    long time_limit;
    long hard_time_limit;
    long mem_limit;

	mem_limit = get_env_number("MEM_LIMIT", "MEM", 16000) & ~3;
	time_limit = get_env_number("TIME_LIMIT", "TIME", 10000);
    hard_time_limit = get_env_number("HARD_LIMIT", "HARD", time_limit + 2000);

	if (get_env_number("ET_LIMIT", NULL, 0))
		 hard_time_limit = 0;

	rl.rlim_cur = rl.rlim_max = 0;
	if (setrlimit(RLIMIT_CORE, &rl))
		fail("setrlimit(RLIMIT_CORE) failed");

	rl.rlim_cur = rl.rlim_max = RLIM_INFINITY;
	if (setrlimit(RLIMIT_STACK, &rl))
		fail("setrlimit(RLIMIT_STACK) failed");

	rl.rlim_cur = rl.rlim_max = mem_limit*1024;
	if (setrlimit(RLIMIT_AS, &rl))
		fail("setrlimit(RLIMIT_AS) failed");
	mem_limit = 0;

	if (hard_time_limit)
	{
		rl.rlim_cur = rl.rlim_max = hard_time_limit/1000;
		if (setrlimit(RLIMIT_CPU, &rl))
			fail("setrlimit(RLIMIT_CPU) failed");
	}

	signal(SIGALRM, SIG_IGN);
	
	fd = open("/dev/null", O_WRONLY);
	if (fd < 0)
		fail("cannot open /dev/null");
	if (fd != 2)
	{
		if (dup2(fd, 2) < 0)
			fail("dup2() failed");
		close(fd);
	}

	ptrace(PTRACE_TRACEME, getpid(), 0, 0);
	raise(SIGSTOP);
}

void svioloader_run(void)
{
    char pidt[20];
    int ret;
    const char* supervisor_exe;

    child = fork();
    if (child < 0) {
        perror("libsvloader: fork() failed");
        exit(121);
    } else if (child == 0) {
        tracked_process();
        return;
    }

    supervisor_exe = get_env_string("SUPERVISOR", NULL, "supervisor");

    sprintf(pidt, "%d", (int)child);
    ret = execlp(supervisor_exe, "supervisor", "-w", pidt, "-e", "-r", "-q",
            NULL);

    fprintf(stderr, "libsvloader: cannot run supervisor, falling back \n"
                    "             to standalone mode.\n");

    kill(child, SIGKILL);

}
