#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/resource.h>

#include "svioparse.h"
#include "iomap.h"

static void* token_start;
static void* token_end;
static void* data_end;
static void* string_cursor;

static void* ch_cursor;
static void* ch_start;
static void* ch_limit;

static void* fl_cursor;
static void* fl_start;
static void* fl_limit;

static struct rlimit saved_memlimit;
static int read_by_line;

static int whitespace_tbl[256] = {
    [' ']  = 1,
    ['\t'] = 1,
    ['\r'] = 1,
    ['\n'] = 1,
};

#define MAX_NONSTRING_TOKEN 60
#define RESERVED_END_SPACE  16

inline int is_whitespace(unsigned char c)
{
    return whitespace_tbl[c];
}

static int raise_memlimit(void)
{
    struct rlimit rlim;
    int err = getrlimit(RLIMIT_AS, &rlim);
    
    if (rlim.rlim_max == rlim.rlim_cur) {
        if (rlim.rlim_max == RLIM_INFINITY)
            return 0;

        fprintf(stderr,
                "svioparse: cannot raise memory limits for reading input data\n"
                "           please ensure, that only the soft limit is used\n"
                "           for testing (for example 'ulimit -Sv 32000').\n"
                "           For now, memory limiting may be inaccurate.\n");
        return 0;
    }

    saved_memlimit = rlim;
    rlim.rlim_cur = rlim.rlim_max;

    err = setrlimit(RLIMIT_AS, &rlim);
    if (err < 0) {
        fprintf(stderr,
                "svioparse: error raising memory limit for reading input.\n"
                "           For now, memory limiting may be inaccurate.\n");
        return 0;
    }

    return 1;
}

static void lower_memlimit(void)
{
    struct rlimit rlim = saved_memlimit;
    int err;

    if (rlim.rlim_cur != RLIM_INFINITY)
        rlim.rlim_cur += iomap_size;

    if (rlim.rlim_cur > rlim.rlim_max) {
        fprintf(stderr,
                "svioparse: cannot raise memory limits after reading input data\n"
                "           Memory limiting may be inaccurate.\n");
        rlim.rlim_cur = rlim.rlim_max;
    }

    err = setrlimit(RLIMIT_AS, &rlim);
    if (err < 0)
        fprintf(stderr,
                "svioparse: error raising memory limit after reading input.\n"
                "           Memory limiting may be inaccurate.\n");
}

void svioparse_init(int adjust_memlimit)
{
    if (adjust_memlimit == 2) {
        read_by_line = 1;
        iomap_data = NULL;
        iomap_size = 0;
    } else {
        int restore_memlimit = adjust_memlimit ? raise_memlimit() : 0;
        iomap_entire_stdin();
        if (restore_memlimit)
            lower_memlimit();
    }

    token_start = token_end = iomap_data;
    data_end = iomap_data + iomap_size;
}

static void next_token(void)
{
    token_start = token_end;
    if (token_end == data_end) return;
    while (token_start < data_end && 
            is_whitespace(*(unsigned char*)token_start))
        token_start++;
    token_end = token_start;
    while (token_end < data_end && !is_whitespace(*(unsigned char*)token_end))
        token_end++;
}

static void put_string(void)
{
    if (string_cursor) {
        while (string_cursor < token_end && ch_cursor < ch_limit)
            *(char*)(ch_cursor++) = *(char*)(string_cursor++);
        if (string_cursor == token_end)
            string_cursor = NULL;
    }
}

static void chunk_header(char c)
{
    *(char*)(ch_cursor++) = c;
}

static void parse_string(void)
{
    chunk_header('S');
    *(int*)ch_cursor = token_end - token_start;
    ch_cursor += 4;
    string_cursor = token_start;
    put_string();
}

static void parse_longlong(long long ll)
{
    chunk_header('N');
    *(long long*)ch_cursor = ll;
    ch_cursor += 8;
}

static int parse_token(void)
{
    next_token();
    if (token_start == token_end)
        return -1;
    else if (token_end - token_start > MAX_NONSTRING_TOKEN)
        parse_string();
    else {
        char buf[MAX_NONSTRING_TOKEN+1];
        char* endptr;
        long long ll;
        int len = token_end - token_start;
        memcpy(buf, token_start, len);
        buf[len] = '\0';
        ll = strtoll(buf, &endptr, 10);
        if (*endptr == '\0')
            parse_longlong(ll);
        else
            parse_string();
    }
    
    return 0;
}

int svioparse_next_chunk(void* _ch_cursor, void* _ch_limit)
{
    int terminate = 0;

    ch_cursor = _ch_cursor;
    ch_start = ch_cursor;
    ch_limit = _ch_limit;
    
    /* 1. Finalize string. */
    put_string();

    if (parse_token() == -1)
    {
        if (!read_by_line) 
            terminate = 1;
        else if (read_some_stdin() < 0)
            terminate = 1;
        else {
            token_start = token_end = iomap_data;
            data_end = iomap_data + iomap_size;
            if (parse_token() < 0)
                terminate = 1;
        }
    }

    /* 2. Parse tokens. */
    while (ch_cursor + RESERVED_END_SPACE < ch_limit)
        if (terminate)
            chunk_header('E');
        else if (parse_token() < 0) {
            if (!read_by_line)
                terminate = 1;
            else
                break;
        }

    return ch_cursor - ch_start;
}


static int flush_string(void)
{
    unsigned long len;
    if (fl_cursor+sizeof(unsigned long) > fl_limit)
        return -1;
    len = *((unsigned long*)fl_cursor);
    fl_cursor += sizeof(unsigned long);
    if(fl_cursor+len > fl_limit)
        return -1;
    if (fwrite(fl_cursor, len, 1, stdout) != 1)
        return -2;
    fl_cursor += len;
    return 0;
}

static int flush_char(void)
{
    if (fl_cursor >= fl_limit)
        return -1;
    if (putc(*(unsigned char*)fl_cursor, stdout) == EOF)
        return -2;
    fl_cursor++;
    return 0;
}

static int flush_longlong(void)
{
    if (fl_cursor+sizeof(long long) > fl_limit)
        return -1;
    if (printf("%lld ", *(long long*)fl_cursor) < 0)
        return -2;
    fl_cursor += sizeof(long long);
    return 0;
}

static int flush_token(void)
{
    switch (*(char*)fl_cursor++) {
        case 'S': return flush_string(); break;
        case 'C': return flush_char(); break;
        case 'N': return flush_longlong(); break;
        default:  return -1;
    }
}

int svioparse_flush_obuf(void* _fl_cursor, void* _fl_limit)
{
    fl_cursor = _fl_cursor;
    fl_start = fl_cursor;
    fl_limit = _fl_limit;

    while (fl_cursor < fl_limit) {
        int err = flush_token();
        if (err) return err;
    }

    fflush(stdout);

    return 0;
}

