#include <stdlib.h>
#include <string.h>

#include "supervisor.h"
#include "prototypes.h"
#include "ioshm.h"

static void* token_start;
static void* token_end;
static void* data_end;
static void* string_cursor;

static void* ch_cursor;
static void* ch_start;
static void* ch_limit;

static void* fl_cursor;
static void* fl_start;
static void* fl_limit;

static int whitespace_tbl[256] = {
    [' ']  = 1,
    ['\t'] = 1,
    ['\r'] = 1,
    ['\n'] = 1,
};

#define MAX_NONSTRING_TOKEN 60
#define RESERVED_END_SPACE  16

static inline int is_whitespace(unsigned char c)
{
    return whitespace_tbl[c];
}

void init_parser(void)
{
    iomap_entire_stdin();
    token_start = token_end = iomap_data;
    data_end = iomap_data + iomap_size;
}

static void next_token(void)
{
    token_start = token_end;
    if (token_end == data_end) return;
    while (token_start < data_end && 
            is_whitespace(*(unsigned char*)token_start))
        token_start++;
    token_end = token_start;
    while (token_end < data_end && !is_whitespace(*(unsigned char*)token_end))
        token_end++;
}

static void put_string(void)
{
    if (string_cursor) {
        while (string_cursor < token_end && ch_cursor < ch_limit)
            *(char*)(ch_cursor++) = *(char*)(string_cursor++);
        if (string_cursor == token_end)
            string_cursor = NULL;
    }
}

static void chunk_header(char c)
{
    *(char*)(ch_cursor++) = c;
}

static void parse_string(void)
{
    chunk_header('S');
    *(int*)ch_cursor = token_end - token_start;
    ch_cursor += 4;
    string_cursor = token_start;
    put_string();
}

static void parse_longlong(long long ll)
{
    chunk_header('N');
    *(long long*)ch_cursor = ll;
    ch_cursor += 8;
}

static void parse_token(void)
{
    next_token();
    if (token_start == token_end)
        chunk_header('E');
    else if (token_end - token_start > MAX_NONSTRING_TOKEN)
        parse_string();
    else {
        char buf[MAX_NONSTRING_TOKEN+1];
        char* endptr;
        long long ll;
        int len = token_end - token_start;
        memcpy(buf, token_start, len);
        buf[len] = '\0';
        ll = strtoll(buf, &endptr, 10);
        if (*endptr == '\0')
            parse_longlong(ll);
        else
            parse_string();
    }
    
}

static int next_chunk(void)
{
    ch_cursor = ioshm+ioshm_hdr_shadow->idata_offset;
    ch_start = ch_cursor;
    ch_limit = ioshm+ioshm_hdr_shadow->odata_offset;
    
    /* 1. Finalize string. */
    put_string();

    /* 2. Parse tokens. */
    while (ch_cursor + RESERVED_END_SPACE < ch_limit)
        parse_token();

    ioshm_hdr->idata_bufsize = ch_cursor - ch_start;

    return 0;
}

int sys_iochunk(long sysnr)
{
    emulate_syscall(next_chunk());
    return 0;
}


static void check_cond(int cond)
{
    if (!cond)
        forbidden("invalid ioshm interaction");
}

static void flush_string(void)
{
    unsigned long len;
    check_cond(fl_cursor+sizeof(unsigned long) <= fl_limit);
    len = *((unsigned long*)fl_cursor);
    fl_cursor += sizeof(unsigned long);
    check_cond(fl_cursor+len <= fl_limit);
    if (fwrite(fl_cursor, len, 1, stdout) != 1)
        fail_tracker("error writing output");
    fl_cursor += len;
}

static void flush_char(void)
{
    check_cond(fl_cursor+1 <= fl_limit);
    if (putc(*(unsigned char*)fl_cursor, stdout) == EOF)
        fail_tracker("error writing output");
    fl_cursor++;
}

static void flush_longlong(void)
{
    check_cond(fl_cursor+sizeof(long long) <= fl_limit);
    if (printf("%lld ", *(long long*)fl_cursor) < 0)
        fail_tracker("error writing output");
    fl_cursor += sizeof(long long);
}

static void flush_token(void)
{
    switch (*(char*)fl_cursor++) {
        case 'S': flush_string(); break;
        case 'C': flush_char(); break;
        case 'N': flush_longlong(); break;
        default:
            forbidden("invalid output format identifier");
    }
}

static int flush_obuf(void)
{
    check_cond(ioshm_hdr->odata_bufsize <= ioshm_hdr_shadow->odata_bufsize);

    fl_cursor = ioshm+ioshm_hdr_shadow->odata_offset;
    fl_start = fl_cursor;
    fl_limit = fl_cursor+ioshm_hdr->odata_bufsize;

    while (fl_cursor < fl_limit)
        flush_token();

    return 0;
}

int sys_ioflush(long sysnr)
{
    emulate_syscall(flush_obuf());
    return 0;
}


