#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/syscall.h>
#include <sys/ptrace.h>
#include <signal.h>
#include <sys/reg.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <fcntl.h>
#include <unistd.h>

#include "supervisor.h"
#include "prototypes.h"
#include "ipc.h"
#include "ioshm.h"

static key_t ioshm_key;
static int ioshm_id;
static int ioshm_tracked_id = -1;

void* ioshm;
struct ioshm_hdr* ioshm_hdr;
struct ioshm_hdr* ioshm_hdr_shadow;

static void err_badsetup(void)
{
    report_result(RETVAL_RV, "bad shm setup sequence");
    kill_and_exit();
}

static void cleanup_ioshm(void)
{
    int res = shmctl(ioshm_id, IPC_RMID, NULL);
    if (res < 0)
        fail_tracker("error marking shm segment for destruction");
}

void init_ioshm(void)
{
    int ur = open("/dev/urandom", O_RDONLY);
    ssize_t r;

    if (ur < 0)
        fail_tracker("cannot open /dev/urandom");
    r = read(ur, &ioshm_key, sizeof(ioshm_key));
    if (r != sizeof(ioshm_key))
        fail_tracker("error or short read from /dev/urandom");
    close(ur);

    ioshm_id = shmget(ioshm_key, IOSHM_SIZE, IPC_CREAT | IPC_EXCL | 0600);
    if (ioshm_id < 0)
        fail_tracker("cannot create shm segment");

    atexit(cleanup_ioshm);
    
    ioshm = shmat(ioshm_id, NULL, 0);
    if (ioshm == (void*)-1)
        fail_tracker("cannot attach shm segment");

    ioshm_hdr = (struct ioshm_hdr*)ioshm;
    memcpy(ioshm_hdr->magic, "IOSHM", 5);
    ioshm_hdr->idata_offset = sizeof(*ioshm_hdr);
    ioshm_hdr->idata_bufsize = 0;
    ioshm_hdr->odata_offset = ioshm_hdr->idata_offset + IOSHM_ISIZE;
    ioshm_hdr->odata_bufsize = IOSHM_SIZE - ioshm_hdr->odata_offset;

    ioshm_hdr_shadow = malloc(sizeof(*ioshm_hdr));
    memcpy(ioshm_hdr_shadow, ioshm_hdr, sizeof(*ioshm_hdr));
}

int sys_ioshmget(long sysnr)
{
    if (sc_state == SS_OUT) {
        if (ioshm_tracked_id >= 0)
            err_badsetup();
        change_syscall(__NR_ipc);
        uwrite(EBX, SHMGET);
        uwrite(ECX, ioshm_key);
        uwrite(EDX, IOSHM_SIZE);
        uwrite(ESI, 0);

        init_parser();
    } else {
        ioshm_tracked_id = uread(EAX);
        if (ioshm_tracked_id < 0)
            fail_tracker("e_ioshmget failed");
    }
    return 2;
}

int sys_ioshmat(long sysnr)
{
    if (sc_state == SS_OUT) {
        long dptr = uread(EBX);
        change_syscall(__NR_ipc);
        uwrite(EBX, SHMAT);
        uwrite(ECX, ioshm_tracked_id);
        uwrite(EDI, 0);
        uwrite(EDX, 0);
        uwrite(ESI, dptr);
    } else {
        if (uread(EAX) == -1)
            fail_tracker("e_ioshmat failed");
    }
    return 2;
}
