#!/bin/sh
#
#
# $1 - source
# $2 - output
# [ $3 - language = C | CPP | PAS ] 
#
#
SRC=$1
OUT=$2
LANG=$3

trap "rm -f \"$ASM\"" EXIT

# source file should be in current directory
function compile_pas() {
    FOO=foo
    ppc386 -O2 -XS "$SRC" "-o$FOO" -k'/lib/libsvio.a'
    r=$?
    if [ -f "$FOO" ] ; then 
	mv "$FOO" "$OUT"
    fi
    return $r
}

function compile_ocaml() {
    FOO=foo.ml
    mv "$SRC" "$FOO"
    ocamlopt "$FOO" -o "$OUT"
    r=$?
    return $r
}

function link_svio() {
    if grep -q '#NO_APP' "$2"; then
        echo "Inline assembly not allowed."
        return 1
    fi
    if grep -q '^[ \t]*rep' "$2"; then
        echo "Unexpected REP prefix in generated assembly."
        echo "Please contact system administrator."
        return 1
    fi
    $1 -static -o "$OUT" -x assembler "$2" -x none -lm /lib/libsvio.a /lib/libcmy.a -Wl,-zmuldefs
    RV=$?
    rm -f "$2"
    return $RV
}

function compile_c_with_svio() {
    /usr/bin/gcc -x c -S -O2 -o "$OUT.asm" "$SRC"
    link_svio /usr/bin/gcc "$OUT.asm"
}

function compile_cpp_with_svio() {
    /usr/bin/g++ -x c++ -S -O2 -o "$OUT.asm" "$SRC"
    link_svio /usr/bin/g++ "$OUT.asm"
}

function compile_c() {
    /usr/bin/gcc -x c -O2 -static -o "$OUT" "$SRC" -lm
}

function compile_cpp() {
    /usr/bin/g++ -x c++ -O2 -static -o "$OUT" "$SRC" -lm
}

function repack_zip() {
    zipinfo -t $SRC >info || return 1
    len=`tail -n 1 info | sed -ne 's/.* \([0-9]\+\) bytes uncompressed.*/\1/p'`
    if [ "$len" -gt 11000000 ]; then
	echo File too long 1>&2
	exit 1
    fi

    dir="repack_tmp" 
    dir2="repack_tmp2" 
    rm -rf "$dir" "$dir2"
    mkdir "$dir" "$dir2" || return 1
    unzip -d "$dir" -qqjLo $SRC || return 1

    find "$dir" -iname '*.out' -size -10000k -exec mv {} "$dir2" \;

    outputs=`find "$dir2" -iname '*.out' -size -10000k`
    IFS=$'\n'
    for f in $outputs; do
	d=`echo $f | tr A-Z a-z`
	if [ "$f" != "$d" ]; then
	    mv "$f" "$d"
	fi
    done

    echo foo >"$dir2/_bar" # zip wants a non empty archive
    rm -f tmp.zip
    zip -qqjr tmp.zip "$dir2" || return 1
    rm -rf "$dir" "$dir2"
    mv tmp.zip $OUT
}

function repack_tgz() {
    dir="repack_tmp" 
    dir2="repack_tmp2" 
    rm -rf "$dir" "$dir2"
    mkdir "$dir" "$dir2" || return 1
    tar -xzf "$SRC" -C "$dir" || return 1

    find "$dir" -iname '*.out' -size -10000k -exec mv {} "$dir2" \;

    outputs=`find "$dir2" -iname '*.out' -size -10000k`
    IFS=$'\n'
    for f in $outputs; do
	d=`echo $f | tr A-Z a-z`
	if [ "$f" != "$d" ]; then
	    mv "$f" "$d"
	fi
    done

    echo foo >"$dir2/_bar" # zip wants a non empty archive
    rm -f tmp.zip
    zip -qqjr tmp.zip "$dir2" || return 1
    rm -rf "$dir" "$dir2"
    mv tmp.zip $OUT
}

if [ -f "$SRC" ] ; then
    if [ -z "$OUT" ] ; then
	exit 101
    fi

    if [ -z "$LANG" ] ; then
	LANG=`echo $SRC | sed -e "s/.*\.//g"`
    fi

    case "$LANG" in
    PAS|pas)
	compile_pas
	res=$?
	;;
    C|c)
    read LINE1 < "$SRC"
    if grep -q '^[ \t]*#include <svio.h>' "$SRC"; then
    compile_c_with_svio
    else
	compile_c
    fi
	res=$?
	;;
    CPP|CC|cpp|cc)
    if grep -q '^[ \t]*#include <svio.h>' "$SRC"; then
    compile_cpp_with_svio
    else
	compile_cpp
    fi
	res=$?
	;;
    ML|ml|OCAML|ocaml)
	compile_ocaml
	res=$?
	;;
    ZIP)
	repack_zip
	res=$?
	;;
    TGZ)
	repack_tgz
	res=$?
	;;
    *)
	exit 102
    esac
    exit $res
else
    # no input file
    exit 100
fi

