/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 *                                                                           *
 *  This file is part of Library of Effective GO routines - EGO library      *
 *                                                                           *
 *  Copyright 2006, 2007, 2008 Lukasz Lew                                    *
 *                                                                           *
 *  EGO library isfree software; you can redistribute it and/or modify       *
 *  it under the terms of the GNU General Public License as published by     *
 *  the Free Software Foundation; either version 2 of the License, or        *
 *  (at your option) any later version.                                      *
 *                                                                           *
 *  EGO library is distributed in the hope that it will be useful,           *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *  GNU General Public License for more details.                             *
 *                                                                           *
 *  You should have received a copy of the GNU General Public License        *
 *  along with EGO library; if not, write to the Free Software               *
 *  Foundation, Inc., 51 Franklin St, Fifth Floor,                           *
 *  Boston, MA  02110-1301  USA                                              *
 *                                                                           *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <iostream>
#include <cstring>
#include <sys/resource.h>

using namespace std;

typedef unsigned int uint;
typedef unsigned long long uint64;

// board parameters
const uint board_size = 9;

#define rep(i,n) for (int i = 0; i < (int)(n); i++)

// ----------------------------------------------------------------------

class PerformanceTimer {
  double  sample_cnt;
  double  sample_sum;
  uint64  start_time;
  double  overhead;
public:

  PerformanceTimer () {
    reset ();
    uint64 t1, t2;
    t1 = get_cc_time ();
    t2 = get_cc_time ();
    overhead = double (t2 - t1);
  }

  void reset () {
    sample_cnt = 0;
    sample_sum = 0;
  }

  uint64 get_cc_time () volatile {
    uint64 ret;
    uint64 high;
    __asm__ __volatile__("rdtsc" : "=a" (ret), "=d"(high) : :);
    return ret | (high << 32);
  }

  void start () {
    start_time = get_cc_time ();
  }

  void stop () {
    uint64 stop_time;
    stop_time = get_cc_time ();
    sample_cnt += 1.0;
    sample_sum += double (stop_time - start_time) - overhead;
  }
  
  double ticks () { return sample_sum / sample_cnt; }
};


// ----------------------------------------------------------------------

class PmRandom {             // Park - Miller "minimal standard" 
  static const int cnt = (uint(1)<<31) - 1;  
  uint seed;
public:

  PmRandom (uint seed_ = 12345) { seed = seed_; }

  void set_seed (uint _seed) { seed = _seed; }
  uint get_seed () { return seed; }

  uint rand_int () {       // a number between  0 ... cnt - 1
    uint hi, lo;
    lo = 16807 * (seed & 0xffff);
    hi = 16807 * (seed >> 16);
    lo += (hi & 0x7fff) << 16;
    lo += hi >> 15;
    seed = (lo & 0x7FFFFFFF) + (lo >> 31);
    return seed;
    //return mt (); // equivalen
  }

  // n must be between 1 .. (1<<16) + 1
  inline uint rand_int (uint n) { // 0 .. n-1
    return ((rand_int () & 0xffff) * n) >> 16;
  }
};

// very simple and useful FastMap

template <typename idx_t, typename elt_t> class FastMap {
  elt_t tab [idx_t::cnt];
public:
  elt_t& operator[] (idx_t pl)             { return tab [pl.get_idx ()]; }
  const elt_t& operator[] (idx_t pl) const { return tab [pl.get_idx ()]; }
};

// some usefull functions

double get_seconds () {
  rusage usage [1];
  getrusage (RUSAGE_SELF, usage);
  return double(usage->ru_utime.tv_sec) + double(usage->ru_utime.tv_usec) / 1000000.0;
}

#define no_inline   __attribute__((noinline))
#define flatten     __attribute__((flatten))
#define all_inline  __attribute__((always_inline))

// ----------------------------------------------------------------------

class Player { 
  uint idx;
public:

  const static uint black_idx = 0;
  const static uint white_idx = 1;
  const static uint cnt = 2;

  explicit Player () { idx = -1; }
  explicit Player (uint _idx) { idx = _idx; }

  bool operator== (Player other) const { return idx == other.idx; }

  Player other () const { return Player(idx ^ 1); }
  bool in_range () const { return idx < cnt; } 
  void next () { idx++; }
  uint get_idx () const { return idx; }
  static Player black () { return Player (black_idx); }
  static Player white () { return Player (white_idx); }
};

// faster than non-loop
#define player_for_each(pl) for (Player pl = Player::black (); pl.in_range (); pl.next ())

//------------------------------------------------------------------------------

class Color {
  uint idx;
public:

  const static uint black_idx = 0;
  const static uint white_idx = 1;
  const static uint empty_idx = 2;
  const static uint off_board_idx  = 3;
  const static uint cnt = 4;

  explicit Color () { idx = -1; } // TODO test - remove it
  explicit Color (uint idx_) { idx = idx_; } // TODO test - remove it
  explicit Color (Player pl) { idx = pl.get_idx (); }

  bool is_player     () const { return idx <= white_idx; } // & (~1)) == 0; }
  bool is_not_player () const { return idx  > white_idx; } // & (~1)) == 0; }
  Player to_player () const { return Player (idx); }

  bool in_range () const { return idx < Color::cnt; }
  void next () { idx++; }

  uint get_idx () const { return idx; }
  bool operator== (Color other) const { return idx == other.idx; }
  bool operator!= (Color other) const { return idx != other.idx; }

  static Color black ()      { return Color (black_idx); }
  static Color white ()      { return Color (white_idx); }
  static Color empty ()      { return Color (empty_idx); }
  static Color off_board  () { return Color (off_board_idx); }
};

// TODO test it for performance
#define color_for_each(col) for (Color col = Color::black (); col.in_range (); col.next ())

bool coord_is_on_board (int coord) { return uint (coord) < board_size; }

//--------------------------------------------------------------------------------

class Vertex {
  //static_assert (cnt <= (1 << bits_used));
  //static_assert (cnt > (1 << (bits_used-1)));

  uint idx;

public:

  const static uint dNS = (board_size + 2);
  const static uint dWE = 1;

  const static uint bits_used = 9;     // on 19x19 cnt == 441 < 512 == 1 << 9;
  const static uint pass_idx = 0;
  const static uint any_idx  = 1; // TODO any
  const static uint resign_idx = 2;

  const static uint cnt = (board_size + 2) * (board_size + 2);

  explicit Vertex () { } // TODO is it needed
  explicit Vertex (uint _idx) { idx = _idx; }

 // TODO make this constructor a static function
  Vertex (int r, int c) {
    idx = (r+1) * dNS + (c+1) * dWE;
  }

  uint get_idx () const { return idx; }

  bool operator== (Vertex other) const { return idx == other.idx; }
  bool operator!= (Vertex other) const { return idx != other.idx; }

  bool in_range ()          const { return idx < cnt; }
  void next ()                    { idx++; }

  int row () const { return idx / dNS - 1; }

  int col () const { return idx % dNS - 1; }

  bool is_on_board () const {
    return coord_is_on_board (row ()) & coord_is_on_board (col ());
  }

  Vertex N () const { return Vertex (idx - dNS); }
  Vertex W () const { return Vertex (idx - dWE); }
  Vertex E () const { return Vertex (idx + dWE); }
  Vertex S () const { return Vertex (idx + dNS); }

  Vertex NW () const { return N ().W (); } // TODO can it be faster?
  Vertex NE () const { return N ().E (); } // only Go
  Vertex SW () const { return S ().W (); } // only Go
  Vertex SE () const { return S ().E (); }

  static Vertex pass   () { return Vertex (Vertex::pass_idx); }
  static Vertex any    () { return Vertex (Vertex::any_idx); }
  static Vertex resign () { return Vertex (Vertex::resign_idx); }

};

#define vertex_for_each_all(vv) for (Vertex vv = Vertex(0); vv.in_range (); vv.next ()) // TODO 0 works??? // TODO player the same way!

#define vertex_for_each_nbr(center_v, nbr_v, block) {   \
    Vertex nbr_v;                                       \
    nbr_v = center_v.N (); block;                       \
    nbr_v = center_v.W (); block;                       \
    nbr_v = center_v.E (); block;                       \
    nbr_v = center_v.S (); block;                       \
  }

#define vertex_for_each_diag_nbr(center_v, nbr_v, block) {      \
    Vertex nbr_v;                                               \
    nbr_v = center_v.NW (); block;                              \
    nbr_v = center_v.NE (); block;                              \
    nbr_v = center_v.SW (); block;                              \
    nbr_v = center_v.SE (); block;                              \
  }

// ----------------------------------------------------------------------

class NbrCounter {
public:

  static const uint max = 4;                 // maximal number of neighbours
  static const uint f_size = 4;              // size in bits of each of 3 counters in nbr_cnt::t
  static const uint f_mask = (1 << f_size) - 1;
  static const uint f_shift_black = 0 * f_size;
  static const uint f_shift_white = 1 * f_size;
  static const uint f_shift_empty = 2 * f_size;
  static const uint black_inc_val = (1 << f_shift_black) - (1 << f_shift_empty);
  static const uint white_inc_val = (1 << f_shift_white) - (1 << f_shift_empty);
  static const uint off_board_inc_val = (1 << f_shift_black) + (1 << f_shift_white) - (1 << f_shift_empty);
  
public:

  uint bitfield;

  NbrCounter () { }
  
  NbrCounter (uint black_cnt, uint white_cnt, uint empty_cnt) {
    bitfield = 
      (black_cnt << f_shift_black) +
      (white_cnt << f_shift_white) +
      (empty_cnt << f_shift_empty);
  }
  
  static const uint player_inc_tab [Player::cnt];
  void player_inc (Player player) { bitfield += player_inc_tab [player.get_idx ()]; }
  void player_dec (Player player) { bitfield -= player_inc_tab [player.get_idx ()]; }
  void off_board_inc () { bitfield += off_board_inc_val; }

  uint empty_cnt  () const { return bitfield >> f_shift_empty; }

  static const uint f_shift [3];
  uint player_cnt (Player pl) const { return (bitfield >> f_shift [pl.get_idx ()]) & f_mask; }

  static const uint player_cnt_is_max_mask [Player::cnt];
  uint player_cnt_is_max (Player pl) const { 
    return 
      (player_cnt_is_max_mask [pl.get_idx ()] & bitfield) == 
       player_cnt_is_max_mask [pl.get_idx ()]; 
  }

};

const uint NbrCounter::f_shift [3] = { 
  NbrCounter::f_shift_black, 
  NbrCounter::f_shift_white, 
  NbrCounter::f_shift_empty, 
};

const uint NbrCounter::player_cnt_is_max_mask [Player::cnt] = {  // TODO player_Map
  (max << f_shift_black), 
  (max << f_shift_white) 
};

const uint NbrCounter::player_inc_tab [Player::cnt] = { NbrCounter::black_inc_val, NbrCounter::white_inc_val };

enum play_ret_t { play_ok, play_suicide, play_ss_suicide, play_ko };

// ----------------------------------------------------------------------

class Board {

public:

  FastMap<Vertex, Color>       color_at;
  FastMap<Vertex, NbrCounter>  nbr_cnt; // incremental, for fast eye checking
  FastMap<Vertex, uint>        empty_pos;
  FastMap<Vertex, Vertex>      chain_next_v;

  uint                         chain_lib_cnt [Vertex::cnt]; // indexed by chain_id
  FastMap<Vertex, uint>        chain_id;
  
  Vertex                       empty_v [board_size * board_size];
  uint                         empty_v_cnt;
  uint                         last_empty_v_cnt;

  FastMap<Player, uint>        player_v_cnt;
  FastMap<Player, Vertex>      player_last_v;

  int                          komi;

  Vertex                       ko_v;             // vertex forbidden by ko

  Player                       last_player;      // player who made the last play (other::player is forbidden to retake)
  uint                         move_no;

  play_ret_t                   last_move_status;

public:                         // macros

  #define empty_v_for_each(board, vv, i) {                              \
      Vertex vv = Vertex::any ();                                       \
      rep (ev_i, (board)->empty_v_cnt) {                                \
        vv = (board)->empty_v [ev_i];                                   \
        i;                                                              \
      }                                                                 \
    }

public:                         // consistency checks

  Board () { clear (); }
  
  void clear () {
    uint off_board_cnt;

    komi = -7;
    empty_v_cnt  = 0;
    player_for_each (pl) {
      player_v_cnt  [pl] = 0;
      player_last_v [pl] = Vertex::any ();
    }
    move_no      = 0;
    last_player  = Player::white (); // act player is other
    last_move_status = play_ok;
    ko_v         = Vertex::any ();
    vertex_for_each_all (v) {
      color_at      [v] = Color::off_board ();
      nbr_cnt       [v] = NbrCounter (0, 0, NbrCounter::max);
      chain_next_v  [v] = v;
      chain_id      [v] = v.get_idx ();    // TODO is it needed, is it usedt?
      chain_lib_cnt [v.get_idx ()] = NbrCounter::max; // TODO is it logical? (off_boards)

      if (v.is_on_board ()) {
        color_at   [v]              = Color::empty ();
        empty_pos  [v]              = empty_v_cnt;
        empty_v    [empty_v_cnt++]  = v;

        off_board_cnt = 0;
        vertex_for_each_nbr (v, nbr_v, {
            if (!nbr_v.is_on_board ()) 
              off_board_cnt++;
        });
        rep (ii, off_board_cnt) 
          nbr_cnt [v].off_board_inc ();
      }
    }
  }

  void load (const Board* save_board) { 
    memcpy(this, save_board, sizeof(Board)); 
  }

  int get_komi () const { 
    return komi;
  }

public: // legality functions

  // checks for move legality
  // it has to point to empty vertexand empty
  // can't recognize play_suicide
  flatten all_inline 
  bool is_pseudo_legal (Player player, Vertex v) {
    return 
      v == Vertex::pass () || 
      !nbr_cnt[v].player_cnt_is_max (player.other ()) || 
      (!play_eye_is_ko (player, v) && 
       !play_eye_is_suicide (v));
  }

  bool is_eyelike (Player player, Vertex v) { 
    if (! nbr_cnt[v].player_cnt_is_max (player)) return false;

    FastMap<Color, int> diag_color_cnt; // TODO
    color_for_each (col) 
      diag_color_cnt [col] = 0; // memset is slower

    vertex_for_each_diag_nbr (v, diag_v, {
      diag_color_cnt [color_at [diag_v]]++;
    });

    return diag_color_cnt [Color (player.other ())] + (diag_color_cnt [Color::off_board ()] > 0) < 2;
  }

public: // play move functions

  // accept pass
  // will ignore simple-ko ban
  // will play single stone suicide
  void play_legal (Player player, Vertex v) flatten all_inline {
    if (v == Vertex::pass ()) {
      basic_play (player, Vertex::pass ());
      return;
    }
    
    if (nbr_cnt[v].player_cnt_is_max (player.other ())) {
      play_eye_legal (player, v);
    } else {
      play_not_eye (player, v);
    }
  
  }

public: // auxiliary functions

  bool play_eye_is_ko (Player player, Vertex v) {
    return (v == ko_v) & (player == last_player.other ());
  }

  bool play_eye_is_suicide (Vertex v) {
    uint all_nbr_live = true;
    vertex_for_each_nbr (v, nbr_v, all_nbr_live &= (--chain_lib_cnt [chain_id [nbr_v]] != 0));
    vertex_for_each_nbr (v, nbr_v, chain_lib_cnt [chain_id [nbr_v]]++);
    return all_nbr_live;
  }

  all_inline
  void play_not_eye (Player player, Vertex v) {
    basic_play (player, v);

    place_stone (player, v);

    vertex_for_each_nbr (v, nbr_v, {

      nbr_cnt [nbr_v].player_inc (player);
        
      if (color_at [nbr_v].is_player ()) {
        chain_lib_cnt [chain_id [nbr_v]] --; // This should be before 'if' to have good lib_cnt for empty vertices
      
        if (color_at [nbr_v] != Color (player)) { // same color of groups
          if (chain_lib_cnt [chain_id [nbr_v]] == 0) 
            remove_chain (nbr_v);
        } else {
          if (chain_id [nbr_v] != chain_id [v]) {
            if (chain_lib_cnt [chain_id [v]] > chain_lib_cnt [chain_id [nbr_v]]) {
               merge_chains (v, nbr_v);
            } else {
               merge_chains (nbr_v, v);
            }         
          }
        }
      }
    });
    
    if (chain_lib_cnt [chain_id [v]] == 0) {
      remove_chain(v);
      last_move_status = play_suicide;
    } else {
      last_move_status = play_ok;
    }
  }

  no_inline
  void play_eye_legal (Player player, Vertex v) {
    vertex_for_each_nbr (v, nbr_v, chain_lib_cnt [chain_id [nbr_v]]--);

    basic_play (player, v);
    place_stone (player, v);
    
    vertex_for_each_nbr (v, nbr_v, { 
      nbr_cnt [nbr_v].player_inc (player);
    });

    vertex_for_each_nbr (v, nbr_v, {
      if ((chain_lib_cnt [chain_id [nbr_v]] == 0)) 
        remove_chain (nbr_v);
    });

    if (last_empty_v_cnt == empty_v_cnt) { // if captured exactly one stone, end this was eye
      ko_v = empty_v [empty_v_cnt - 1]; // then ko formed
    } else {
      ko_v = Vertex::any ();
    }
  }

  void basic_play (Player player, Vertex v) { // Warning: has to be called before place_stone, because of hash storing
    ko_v                    = Vertex::any ();
    last_empty_v_cnt        = empty_v_cnt;
    last_player             = player;
    player_last_v [player]  = v;
    move_no                += 1;
  }

  void merge_chains (Vertex v_base, Vertex v_new) {
    Vertex act_v;

    chain_lib_cnt [chain_id [v_base]] += chain_lib_cnt [chain_id [v_new]];

    act_v = v_new;
    do {
      chain_id [act_v] = chain_id [v_base];
      act_v = chain_next_v [act_v];
    } while (act_v != v_new);
    
    swap (chain_next_v[v_base], chain_next_v[v_new]);
  }

  no_inline 
  void remove_chain (Vertex v){
    Vertex act_v;
    Vertex tmp_v;
    Color old_color;

    old_color = color_at[v];
    act_v = v;

    do {
      remove_stone (act_v);
      act_v = chain_next_v[act_v];
    } while (act_v != v);

    do {
      vertex_for_each_nbr (act_v, nbr_v, {
        nbr_cnt [nbr_v].player_dec (old_color.to_player());
        chain_lib_cnt [chain_id [nbr_v]]++;
      });

      tmp_v = act_v;
      act_v = chain_next_v[act_v];
      chain_next_v [tmp_v] = tmp_v;
      
    } while (act_v != v);
  }

  void place_stone (Player pl, Vertex v) {
    player_v_cnt[pl]++;
    color_at[v] = Color (pl);

    empty_v_cnt--;
    empty_pos [empty_v [empty_v_cnt]] = empty_pos [v];
    empty_v [empty_pos [v]] = empty_v [empty_v_cnt];

    chain_id [v] = v.get_idx ();
    chain_lib_cnt [v.get_idx ()] = nbr_cnt[v].empty_cnt ();
  }

  void remove_stone (Vertex v) {
    player_v_cnt [color_at[v].to_player ()]--;
    color_at [v] = Color::empty ();

    empty_pos [v] = empty_v_cnt;
    empty_v [empty_v_cnt++] = v;
    chain_id [v] = v.get_idx ();
  }

public:                         // utils

  Player act_player () const { return last_player.other (); } // TODO/FIXME last_player should be preserverd in undo function

  bool both_player_pass () {
    return 
      (player_last_v [Player::black ()] == Vertex::pass ()) & 
      (player_last_v [Player::white ()] == Vertex::pass ());
  }

  int approx_score () const {
    return komi + player_v_cnt[Player::black ()] -  player_v_cnt[Player::white ()];
  }

  int score () const {
    int eye_score = 0;

    empty_v_for_each (this, v, {
        eye_score += nbr_cnt[v].player_cnt_is_max (Player::black ());
        eye_score -= nbr_cnt[v].player_cnt_is_max (Player::white ());
    });

    return approx_score () + eye_score;
  }
};

PmRandom zobrist_pm;

enum playout_status_t { pass_pass, too_long };

// ----------------------------------------------------------------------

class SimplePolicy {
protected:

  static PmRandom pm; // TODO make it non-static

  uint to_check_cnt;
  uint act_evi;

  Board* board;
  Player act_player;

public:

  SimplePolicy () : board (NULL) { }

  void begin_playout (Board* board_) { board = board_; }

  void prepare_vertex () {
    act_player     = board->act_player ();
    to_check_cnt   = board->empty_v_cnt;
    act_evi        = pm.rand_int (board->empty_v_cnt); 
  }

  Vertex next_vertex () {
    Vertex v;
    while (true) {
      if (to_check_cnt == 0) return Vertex::pass ();
      to_check_cnt--;
      v = board->empty_v [act_evi];
      act_evi++;
      if (act_evi == board->empty_v_cnt) act_evi = 0;
      if (!board->is_eyelike (act_player, v)) return v;
    }
  }
};

PmRandom SimplePolicy::pm(123);

// ----------------------------------------------------------------------

static const uint max_playout_length  = board_size * board_size * 2;

class Playout {
public:
  SimplePolicy*  policy;
  Board*   board;

  Playout (SimplePolicy* policy_, Board*  board_) : policy (policy_), board (board_) {}

  all_inline
  void play_move () {
    policy->prepare_vertex ();
    
    while (true) {
      Player   act_player = board->act_player ();
      Vertex   v          = policy->next_vertex ();

      if (board->is_pseudo_legal (act_player, v) == false) {
        continue;
      } else {
        board->play_legal (act_player, v);
        break;
      }
    }
  }

  all_inline
  playout_status_t run () {
    
    policy->begin_playout (board);
    while (true) {
      play_move ();
      
      if (board->both_player_pass ()) {
        return pass_pass;
      }
      
      if (board->move_no >= max_playout_length) {
        return too_long;
      }
    }
  }
  
};

// ----------------------------------------------------------------------
namespace simple_playout_benchmark {

  Board               mc_board [1];

  FastMap<Vertex, int>   vertex_score;
  FastMap<Player, uint>  win_cnt;
  uint                playout_ok_cnt;
  int                 playout_ok_score;
  PerformanceTimer    perf_timer;

  void run (Board const * start_board, uint playout_cnt) {
    playout_status_t status;
    Player winner;
    int score;

    playout_ok_cnt   = 0;
    playout_ok_score = 0;

    player_for_each (pl) 
      win_cnt [pl] = 0;

    vertex_for_each_all (v) 
      vertex_score [v] = 0;

    perf_timer.reset ();

    SimplePolicy policy;
    Playout playout(&policy, mc_board);

    perf_timer.start ();
    float seconds_begin = get_seconds ();

    rep (ii, playout_cnt) {
      mc_board->load (start_board);
      status = playout.run ();
      switch (status) {
      case pass_pass:
        playout_ok_cnt += 1;

        score = mc_board -> score ();
        playout_ok_score += score;

        winner = Player (score <= 0);  // mc_board->winner ()
        win_cnt [winner] ++;
      case too_long:
        break;
      }
    }
    
    float seconds_end = get_seconds ();
    perf_timer.stop ();
    
    cout << "Initial board:" << endl;
    cout << "komi " << start_board->get_komi () << " for white" << endl;
    
    cout << endl;

    cout << "Black wins    = " << win_cnt [Player::black ()] << endl
        << "White wins    = " << win_cnt [Player::white ()] << endl
        << "P(black win)  = " 
        << double (win_cnt [Player::black ()]) / 
           double (win_cnt [Player::black ()] + win_cnt [Player::white ()]) 
        << endl;

#ifdef SLOW
    int komi =  mc_board->get_komi();
#else
    int komi =  mc_board->komi;
#endif
    cout << "komi = " << komi << endl << endl;

    float seconds_total = seconds_end - seconds_begin;
    float cc_per_playout = perf_timer.ticks () / double (playout_cnt);

    cout << "Performance: " << endl
        << "  " << playout_cnt << " playouts" << endl
        << "  " << seconds_total << " seconds" << endl
        << "  " << float (playout_cnt) / seconds_total / 1000.0 << " kpps" << endl
        << "  " << cc_per_playout << " ccpp (clock cycles per playout)" << endl
        << "  " << 1000000.0 / cc_per_playout  << " kpps/GHz (clock independent)" << endl
      ;
  }
}

int main () { 
  Board board;
  simple_playout_benchmark::run (&board, 100000);
  simple_playout_benchmark::run (&board, 100000);
  simple_playout_benchmark::run (&board, 100000);
  return 0;
}
